/**
 * Copyright (C) 2001 Billy Biggs <vektor@dumbterm.net>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <limits.h>
#include <dirent.h>

#include "config.h"

#include <dvdread/dvd_reader.h>
#define DVD_VIDEO_LB_LE 2048

/**
 * Returns true if the pack is a NAV pack.  This check is clearly insufficient,
 * and sometimes we incorrectly think that valid other packs are NAV packs.  I
 * need to make this stronger.
 */
int is_nav_pack( unsigned char *buffer )
{
    return ( buffer[ 41 ] == 0xbf && buffer[ 1027 ] == 0xbf );
}

/**
 * Searches for <file> in directory <path>, ignoring case.
 * Returns 0 and full filename in <filename>.
 *     or -1 on file not found.
 *     or -2 on path not found.
 */
static int findDirFile( const char *path, const char *file, char *filename ) 
{
    DIR *dir;
    struct dirent *ent;

    dir = opendir( path );
    if( !dir ) return -2;

    while( ( ent = readdir( dir ) ) != NULL ) {
        if( !strcasecmp( ent->d_name, file ) ) {
            sprintf( filename, "%s%s%s", path,
                     ( ( path[ strlen( path ) - 1 ] == '/' ) ? "" : "/" ),
                     ent->d_name );
            return 0;
        }
    }

    return -1;
}

static int findDVDFile( const char *root, const char *file, char *filename )
{
    char video_path[ PATH_MAX + 1 ];
    const char *nodirfile;
    int ret;

    /* Strip off the directory for our search */
    if( !strncasecmp( "/VIDEO_TS/", file, 10 ) ) {
        nodirfile = &(file[ 10 ]);
    } else {
        nodirfile = file;
    }

    ret = findDirFile( root, nodirfile, filename );
    if( ret < 0 ) {
        /* Try also with adding the path, just in case. */
        sprintf( video_path, "%s/VIDEO_TS/", root );
        ret = findDirFile( video_path, nodirfile, filename );
        if( ret < 0 ) {
            /* Try with the path, but in lower case. */
            sprintf( video_path, "%s/video_ts/", root );
            ret = findDirFile( video_path, nodirfile, filename );
            if( ret < 0 ) {
                return 0;
            }
        }
    }

    return 1;
}


int main( int argc, char **argv )
{
    int titleid, domain;
    int first_sector, last_sector, len;
    int fd;
    unsigned int cur_pack;
    char file[ MAX_UDF_FILE_NAME_LEN ];
    char filename[ MAX_UDF_FILE_NAME_LEN ];
    unsigned char dvd_data[ DVD_VIDEO_LB_LEN ];
    unsigned char file_data[ DVD_VIDEO_LB_LEN ];
    dvd_reader_t *dvd;
    dvd_file_t *title;

    /**
     * Usage.
     */
    if( argc != 5 ) {
        fprintf( stderr, "Usage: %s <dvd path> <title number> <domain> <block>"
                         "\n\n"
                         "Title Number  [1-X]: VTS number, 0 for VIDEO_TS\n"
                         "Domain        [0|1]: 0 Menu, 1 Title\n"
                         "Block              : Begin blocks into the file\n\n"
                         "Make sure <dvd path> is the mounted path to a DVD\n",
                 argv[ 0 ]);
        return -1;
    }

    titleid = atoi( argv[ 2 ] );
    domain = atoi( argv[ 3 ] );
    first_sector = atoi( argv[ 4 ] );

    /**
     * Open the disc.
     */
    dvd = DVDOpen( argv[ 1 ] );
    if( !dvd ) {
        fprintf( stderr, "Couldn't open DVD: %s\n", argv[ 1 ] );
        return -1;
    }

    /**
     * We've got enough info, time to open the title set data.
     */
    if(domain == 0) {
	title = DVDOpenFile( dvd, titleid, DVD_READ_MENU_VOBS );
    } else if(domain == 1) {
	title = DVDOpenFile( dvd, titleid, DVD_READ_TITLE_VOBS );    
    } else {
	fprintf( stderr, "Not a vaild domain.\n");
	exit(1);
    }
    if( !title ) {
        fprintf( stderr, "Can't open VTS_%02u_%1u.VOB.\n",
                 (unsigned int)titleid, (unsigned int)domain);
        DVDClose( dvd );
        return -1;
    }
    
    if( title == 0 ) {
        sprintf( file, "/VIDEO_TS/VIDEO_TS.VOB" );
    } else {
        sprintf( file, "/VIDEO_TS/VTS_%02d_%d.VOB", titleid, domain );
    }
    findDVDFile( argv[ 1 ], file, filename );
    fd = open( filename, O_RDONLY );
    if( fd < 0 ) {
        fprintf( stderr, "libdvdread: Can't open %s for reading.\n",
		 filename );
	return 0;
    }

    last_sector = DVDFileSize( title );
    
    /**
     * We loop until we're out of this cell.
     */
    for( cur_pack = first_sector; cur_pack < last_sector; ) {
	
	/**
	 * Read NAV packet.
	 */
	len = DVDReadBlocks( title, (int) cur_pack, 1, dvd_data );
	if( len != 1 ) {
	    fprintf( stderr, "Read failed for block %d\n", cur_pack );
	    DVDCloseFile( title );
	    DVDClose( dvd );
	    close( fd );
	    return -1;
	}
	
	len = read( fd, file_data, DVD_VIDEO_LB_LEN );
	if( len != DVD_VIDEO_LB_LE ) {
	    fprintf( stderr, "Read failed for block %d\n", cur_pack );
	    DVDCloseFile( title );
	    DVDClose( dvd );
	    close( fd );
	    return -1;
	}
	  
	if( memcmp( dvd_data, file_data, DVD_VIDEO_LB_LE ) ) {
	  if( !(file_data[0x14] & 0x30) || (dvd_data[0x14] & 0x30) ) {
	    fprintf( stderr, "diff at block %d (dvd: %02x file: %02x)\n", 
		     cur_pack, (dvd_data[0x14]&0x30), (file_data[0x14]&0x30));
	  }
	}
	if( (file_data[0x14] & 0x30) || (dvd_data[0x14] & 0x30) ) {
	    fprintf( stderr, "scrambled block %d\n", cur_pack);
	}
	cur_pack++;
    }
    len = DVDReadBlocks( title, (int) cur_pack, 1, dvd_data );
    if( len != 1 ) {
      DVDCloseFile( title );
      DVDClose( dvd );
    } else {
      fprintf( stderr, "DVD does not end where is should\n" );
    }
    len = read( fd, file_data, DVD_VIDEO_LB_LEN );
    if( len != DVD_VIDEO_LB_LE ) {
      close( fd );
    } else {
      fprintf( stderr, "File does not end where is should\n" );
    }
    return 0;
}
