#line 2 "bar.c"
/*-
 * C-SaCzech
 * Copyright (c) 1996-2002 Jaromir Dolecek <dolecek@ics.muni.cz>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Jaromir Dolecek
 *	for the CSacek project.
 * 4. The name of Jaromir Dolecek may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JAROMIR DOLECEK ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL JAROMIR DOLECEK BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: bar.c,v 1.77 2002/02/03 11:13:41 dolecek Exp $ */

/* stuff for BAR */

#include "csacek.h"

/* constants used only here */
#define CSA_BAR_LONGNAME            1
#define CSA_BAR_ALWAYSADDBAR        2
#define CSA_BAR_ENGLISH             4
#define CSA_BAR_ADDWHICHCODE        8
#define CSA_BAR_ADDHEAD             16
#define CSA_BAR_ADDTAIL             32

/* this holds information about current settings of BAR */
struct csa_bar {
	csa_String	barsep;		/* separator between links in bar */
	csa_String	head;		/* header */
	csa_String	ehead;		/* english header */
	csa_String	tail;		/* text written after links */
	csa_String	etail;		/* english synonym for tail */
	csa_String	whichcodename;	/* text of link to whichcode script */
	csa_String	codenames[CSTOOLS_NUMFONTS];  /* text of link to code */
	long		flags;		/* bar settings */
};

/* local functions */
static void x_switchflag __P((csa_bar_t *bar, const char *value, int flag));
static void x_set_defaults __P((struct pool *, csa_bar_t *));

#define CSA_COPY(pool, what, str)		\
		csa_fillstring((str), ap_pstrdup((pool), (what)), -1, -1)
			
/*
 * switches the ``flag'' in bar->flags accordingly to ``value''
 */
static void
x_switchflag(bar, value, flag)
  csa_bar_t *bar;
  const char *value;
  int flag;
{
	if (strcasecmp(value, "YES")==0)
		CSA_SET(bar->flags, flag);
	else if (strcasecmp(value, "NO")==0)
		CSA_UNSET(bar->flags, flag);
}

/*
 * set defaults in Bar struct passed
 */
static void
x_set_defaults(workpool, Bar)
  struct pool *workpool;
  csa_bar_t *Bar;
{
   int flags=0, i;

   /* zero old struct */
   memset((void *)Bar, '\0', sizeof(csa_bar_t));

   /* name of link to whichcode */
   CSA_COPY(workpool, CSA_BARWHICHCODENAME, &(Bar->whichcodename));
   
   /* default separator between items in a bar */
   CSA_COPY(workpool, " ", &(Bar->barsep) );

#ifdef CSA_ALWAYSADDBAR
   /* add always a bar on end of document */
   flags |= CSA_BAR_ALWAYSADDBAR; 
#endif

#ifdef CSA_ADDWHICHCODE
   /* add whichcode option to bar */
   flags |= CSA_BAR_ADDWHICHCODE; 
#endif

#ifdef CSA_HEADBARTEXT
   CSA_COPY(workpool, CSA_HEADBARTEXT, &(Bar->head));
#else
   CSA_COPY(workpool, "", &(Bar->head));
#endif

#ifdef CSA_EHEADBARTEXT
   CSA_COPY(workpool, CSA_EHEADBARTEXT, &(Bar->ehead));
#else
   CSA_COPY(workpool, "", &(Bar->ehead));
#endif

#ifdef CSA_TAILBARTEXT
   CSA_COPY(workpool, CSA_TAILBARTEXT, &(Bar->tail));
#else
   CSA_COPY(workpool, "", &(Bar->tail));
#endif

#ifdef CSA_ETAILBARTEXT
   CSA_COPY(workpool, CSA_ETAILBARTEXT, &(Bar->etail));
#else
   CSA_COPY(workpool, "", &(Bar->etail));
#endif

   /* default is to write head and tail, use czech */
   CSA_SET(Bar->flags, flags | CSA_BAR_ADDHEAD | CSA_BAR_ADDTAIL);

   /* set default long names for encodings */
   for(i=0; i<CSTOOLS_NUMFONTS; i++) {
	cstools_t code;

	code = cstools_index2code(i);
	CSA_COPY(workpool, cstools_name(code, CSTOOLS_CODENAME),
		&(Bar->codenames[i]));
   }
}

/*
 * creates new struct Bar; all values will be set to default values
 */
csa_bar_t *
csa_bar_create(workpool)
  struct pool *workpool;
{
   csa_bar_t *Bar;

   /* create new bar */
   Bar = (csa_bar_t *) ap_palloc(workpool, sizeof(csa_bar_t));
   x_set_defaults(workpool, Bar);

   return Bar;
}

/* 
 * sets internal Bar structures accordingly to parameters
 */
/* ARGSUSED1 */
int
csa_BarDef(p, prm)
  csa_params_t *p;
  void *prm;
{
	const csa_arg_t *item;
	const char *key, *value;
	csa_bar_t *Bar=p->Bar;

	for(;;) {
	    item = csa_arg_take(p);
	    if (!item) break;

	    key = csa_arg_getkey(item);
	    value = csa_arg_getvalue(item);
	    if (!value) continue;

	    if (!key)
	    {
		if (!strcasecmp(value, "EN"))
				CSA_SET(Bar->flags, CSA_BAR_ENGLISH);
		else if (!strcasecmp(value, "CZ")
			|| !strcasecmp(value, "CS"))
				CSA_UNSET(Bar->flags, CSA_BAR_ENGLISH);
		else if (!strcasecmp(value, "DEFAULT")) 
				x_set_defaults(p->pool_req, Bar);
#ifdef CSA_COMPAT_13X
		else if (!strcasecmp(value, "LONG")) 
				CSA_SET(Bar->flags, CSA_BAR_LONGNAME);
		else if (!strcasecmp(value, "SHORT")) 
				CSA_UNSET(Bar->flags, CSA_BAR_LONGNAME);
		else if (!strcasecmp(value, "WHICH"))
				CSA_SET(Bar->flags, CSA_BAR_ADDWHICHCODE);
		else if (!strcasecmp(value, "NOWHICH"))
				CSA_UNSET(Bar->flags, CSA_BAR_ADDWHICHCODE);
		else if (!strcasecmp(value, "ALWAYS"))
				CSA_SET(Bar->flags, CSA_BAR_ALWAYSADDBAR);
		else if (!strcasecmp(value, "NOBAR")) 
				CSA_UNSET(Bar->flags, CSA_BAR_ALWAYSADDBAR);
		else if (!strcasecmp(value, "NOHEAD")
			 || !strcasecmp(value, "NOTITLE"))   
				CSA_UNSET(Bar->flags, CSA_BAR_ADDHEAD);
		else if (!strcasecmp(value, "HEAD")
			 || !strcasecmp(value, "TITLE"))
				CSA_SET(Bar->flags, CSA_BAR_ADDHEAD);
		else if (!strcasecmp(value, "NOTAIL")
			 || !strcasecmp(value, "NOBACK"))
				CSA_SET(Bar->flags, CSA_BAR_ADDTAIL);
		else if (!strcasecmp(value, "TAIL")
			 || !strcasecmp(value, "NOBACK"))
				CSA_SET(Bar->flags, CSA_BAR_ADDTAIL);
#endif /* CSA_COMPAT_13X */

#ifdef CSA_DEBUG
		else 
		    csa_debug(p->dbg, "BarDef: unknown parameter ``%s''",
			value);
#endif
	    }
	    else /* key not null */
	    {
		csa_String *set_str=NULL;

		if (!strcasecmp(key, "ADDALWAYS"))
			x_switchflag(Bar, value, CSA_BAR_ALWAYSADDBAR);
		else if (!strcasecmp(key, "ADDHEADER"))
			x_switchflag(Bar, value, CSA_BAR_ADDHEAD);
		else if (!strcasecmp(key, "ADDTAIL"))
			x_switchflag(Bar, value, CSA_BAR_ADDTAIL);
		else if (!strncasecmp(key, "ADDWHICH", 8))
			x_switchflag(Bar, value, CSA_BAR_ADDWHICHCODE);
		else if (!strcasecmp(key, "LONGNAMES"))
			x_switchflag(Bar, value, CSA_BAR_LONGNAME);
		else if (!strcasecmp(key, "EHEADTEXT"))
			set_str = &(Bar->ehead);
		else if (!strcasecmp(key, "ETAILTEXT"))
			set_str = &(Bar->tail);
		else if (!strcasecmp(key, "HEADTEXT"))
			set_str = &(Bar->head);
		else if (!strcasecmp(key, "SEP"))
			set_str = &(Bar->barsep);
		else if (!strcasecmp(key, "TAILTEXT"))
			set_str = &(Bar->tail);
		else if (!strcasecmp(key,"WHICHCODE"))
			set_str = &(Bar->whichcodename);
		else {
			/* new long name for encoding */
			cstools_t i = cstools_whichcode(key, 0);
			int sindex = cstools_code2index(i);
			if (sindex >= 0)
				set_str = &(Bar->codenames[sindex]);
#ifdef CSA_DEBUG
			else
			  csa_debug(p->dbg,
				"BarDef: unknown parameter ``%s=%s''", 
				key, value);
#endif
		}
		
		if (set_str) {
			csa_fillstring(set_str, 
				ap_pstrdup(p->pool_req, value), -1, -1);
			set_str = NULL;
		}
	    } /* if */
	} /* while */
     
	return 0;
}

/*
 * create bar with menu of supported encodings; the shape is dependant
 * on values set via csa_BarDef()
 */
/* ARGSUSED1 */
int 
csa_Bar(p, prm)
  csa_params_t	*p;
  void *prm;
{
   int i;
   size_t len, prefix_len, extra_len;
   const csa_String *query, *path_info;
   char *prefix, *extra;
   csa_bar_t *Bar=p->Bar;
   cstools_t code;

#ifdef CSA_DEBUG
	csa_debug(p->dbg, "csa_Bar: called");
#endif

   /* process parameters (if any) */
   (void) csa_BarDef(p, NULL);
   
   /* insert BAR's head into output */
   if (CSA_ISSET(Bar->flags, CSA_BAR_ADDHEAD))
   {
     if (!CSA_ISSET(Bar->flags, CSA_BAR_ENGLISH))
	csa_add_output(p, Bar->head.value, Bar->head.len, 0);
     else
	csa_add_output(p, Bar->ehead.value, Bar->ehead.len, 0);
   }
   
   /* initialize prefix - part of link just before name of output encoding */
   prefix = (char *) csa_alloca(9 + p->csacek_dir->len + 3 + 1,
		p->pool_tmp);
   prefix_len = sprintf(prefix, "<A HREF=\"%s/to", p->csacek_dir->value);
   
   /* initialize extra info - part of link just _after_ name of output */
   /* encoding + ending "\">" */
   query	= csa_getvar(p, "QUERY_STRING");
   path_info	= csa_getvar(p, "PATH_INFO");
   len = path_info->len + (query ? query->len + 1: 0) + 2 + 1;
   extra = (char *) csa_alloca(len, p->pool_tmp);
   extra_len = sprintf(extra, "%s%s%s\">", path_info->value,
		(query ? "?" : ""), (query ? query->value : ""));
   
   for(i=0;i<CSTOOLS_NUMFONTS; i++) 
   {
	code = cstools_index2code(i);

	csa_add_output(p, prefix, prefix_len, 0);
	csa_add_output(p, cstools_name(code, CSTOOLS_TRUENAME), 0, CSA_OUT_STR);
	csa_add_output(p, extra, extra_len, 0);

	if (CSA_ISSET(Bar->flags, CSA_BAR_LONGNAME))
		csa_add_output(p, Bar->codenames[i].value, 
			Bar->codenames[i].len, 0);
	else
		csa_add_output(p, cstools_name(code, CSTOOLS_ABBRNAME),
			0, CSA_OUT_STR);

	csa_add_output(p, "</A>\n", 5, 0);
	if (i < CSTOOLS_NUMFONTS - 1)
		csa_add_output(p, Bar->barsep.value, Bar->barsep.len, 0);
   }

   /* link to whichcode */
   if (CSA_ISSET(Bar->flags, CSA_BAR_ADDWHICHCODE))
   {
     csa_add_output(p, Bar->barsep.value, Bar->barsep.len, 0);
     csa_add_output(p, prefix, prefix_len-2, 0); /* prefix w/o "to" */
     csa_add_output(p, "whichcode", 9, 0);
     csa_add_output(p, extra, extra_len, 0);
     csa_add_output(p, Bar->whichcodename.value, 0, CSA_OUT_STR);
     csa_add_output(p, "</A>\n", 5, 0);
   }

   /* write the tail of bar */
   if (CSA_ISSET(Bar->flags, CSA_BAR_ADDTAIL))
   {
     if (!CSA_ISSET(Bar->flags, CSA_BAR_ENGLISH))
	csa_add_output(p, Bar->tail.value, Bar->tail.len, 0);
     else
	csa_add_output(p, Bar->etail.value, Bar->etail.len, 0);
   }

#ifdef CSA_DEBUG
        csa_debug(p->dbg, "csa_Bar: end");
#endif

   return 0;
}

/*
 * returns 1 if bar is to be printed always, 0 otherwise
 */
int
csa_bar_printalways(bar)
  const csa_bar_t *bar;
{
	return (CSA_ISSET(bar->flags, CSA_BAR_ALWAYSADDBAR));
}
