#include <jmp-config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifdef HAVE_UNISTD_H
 #include <unistd.h>
#endif
#include <errno.h>

#include <gtk/gtk.h>

#include <jmp-debug.h>

#include <ui_gtk.h>
#include <ui_gtk_prefs.h>

#define NAME_SUFFIX_X			"x"
#define NAME_SUFFIX_Y			"y"
#define NAME_SUFFIX_H			"h"
#define NAME_SUFFIX_W			"w"
#define NAME_SUFFIX_STATE		"state"
#define NAME_SUFFIX_STATE_OPEN		"open"
#define NAME_SUFFIX_STATE_CLOSED	"closed"
#define NAME_SUFFIX_VALUE		"value"

struct ctxt_prefs {
    const char *	label;
    void		(*func)(void *, const char *, const char *);
    char		tmpbuf[256];
    union {
        struct windata {
            GtkWindow *		win;
            int			w;
            int			h;
            int			x;
            int			y;
            int			state;
        } windata;
        struct intdata {
            int			value;
            int			isset;
        } intdata;
    } d;
};

static int build_path (char *pathbuf, int pathbuflen, int tmpflag) {
    const char *kp;
    const char *extra;
    const char *file;
    kp = NULL;
    extra = "";
    file = ".jmp-prefs";
#ifdef __unix__
    kp = getenv ("HOME");
#endif
#ifdef WIN32
    if ((kp = getenv ("USERPROFILE")) != NULL) {
        extra = "/Application Data/JMP";
        file = "jmp-prefs";
        if (tmpflag) {
            if (snprintf (pathbuf, pathbuflen, "%s%s", kp, extra) == pathbuflen)
                return -1;
            if (mkdir(pathbuf) < 0) {
                if (errno != EEXIST)
                    return -1;
            }
        }
    }
#endif
    if (kp == NULL)
        return -1;
    if (snprintf (pathbuf, pathbuflen, "%s%s/%s%s", kp, extra, file, (tmpflag) ? ".tmp" : "") == pathbuflen)
        return -1;
    return 0;
}


static int parse_line (char **n, char **v, char *line) {
    char *name_s, *name_e, *value_s, *value_e;
    char *cp;

    for(cp = line; *cp && isspace(*cp); cp++)
        ;	/* skip space */
    if(!*cp)
        return 1;
    if(*cp == '#')
        return 1;

    name_s = cp;
    for( ; *cp && !isspace(*cp) && *cp != '='; cp++)
        ;	/* skip not space */
    if(!*cp)
        return -1;
    name_e = cp;

    if(*cp != '=') {	/* space to equals */
        for( ; *cp && isspace(*cp); cp++)
            ;
        if(!*cp)
            return -1;
    }

    /* equals */
    if(*cp != '=')	/* error */
        return -1;
    cp++;

    /* space from equals */
    for( ; *cp && isspace(*cp); cp++)
        ;
    if(!*cp)
        return -1;

    value_s = cp;
    for( ; *cp && *cp != '\n'; cp++)
        ;	/* skip not space */
    value_e = cp;

    *name_e = '\0';
    *value_e = '\0';
    
    *n = name_s;
    *v = value_s;

    return 0;
}

static int load_file (struct ctxt_prefs *ctxt_prefs) {
    char path[PATH_MAX];
    char buf[1024];
    FILE *fp;

    if (build_path (path, sizeof (path), 0) < 0)
        return 1;	/* not really an error */

    if ((fp = fopen(path, "r")) == NULL) {
        if (errno == ENOENT || errno == ENOTDIR)
            return 0;
        fprintf(stderr, "fopen(\"%s\"): %s\n", path, strerror(errno));
        return 0;	/* not a fatal error */
    }

    while (fgets(buf, sizeof(buf), fp) != NULL) {
        char *n, *v;
        int i;

        i = parse_line (&n, &v, buf);
        if (i < 0) {
            fclose (fp);
            return -1;
        }
        if (i == 1)
            continue;

        (*ctxt_prefs->func)(ctxt_prefs, n, v);
    }
    
    fclose(fp);
    return 0;
}

static int save_file (const char *label, int w, int h, int x, int y, int state) {
    char tmppath[PATH_MAX];
    char path[PATH_MAX];
    FILE *fp_in, *fp_out;
    int need_write;

    if (build_path (tmppath, sizeof (tmppath), 1) < 0)
        return 1;
    if (build_path (path, sizeof (path), 0) < 0)
        return 1;

    if((fp_out = fopen(tmppath, "w")) == NULL)
        return -1;

    need_write = 1;
    if((fp_in = fopen(path, "r")) != NULL) {
        char labelbuf[256];
        char copybuf[1024];
        char buf[1024];

        strcpy (labelbuf, label);
        strcat (labelbuf, ".");

        while(fgets(buf, sizeof(buf), fp_in) != NULL) {
            char *n, *v;
            int i;

            strcpy(copybuf, buf);

            i = parse_line (&n, &v, copybuf);
            if (i < 0) {
                fclose (fp_in);
                fclose (fp_out);
                unlink (tmppath);
                return -1;
            }
            if (i == 1)
                continue;

            if (strncmp (labelbuf, n, strlen(labelbuf)) == 0) {
                /* delete these lines */
            } else {
                if (fputs (buf, fp_out) == EOF) {
                    break;	/* error */
                }
            }
        }
        if (ferror(fp_in)) {
            fclose (fp_in);
            fclose (fp_out);
            unlink (tmppath);
            return -1;
        }
        if (fclose(fp_in) != 0) {
            fclose (fp_out);
            unlink (tmppath);
            return -1;
        }
    }

    if (need_write) {
        fprintf(fp_out, "%s.%s=%d\n", label, NAME_SUFFIX_W, w);
        fprintf(fp_out, "%s.%s=%d\n", label, NAME_SUFFIX_H, h);
        fprintf(fp_out, "%s.%s=%d\n", label, NAME_SUFFIX_X, x);
        fprintf(fp_out, "%s.%s=%d\n", label, NAME_SUFFIX_Y, y);
        fprintf(fp_out, "%s.%s=%s\n", label, NAME_SUFFIX_STATE, state ? NAME_SUFFIX_STATE_OPEN : NAME_SUFFIX_STATE_CLOSED);
        need_write = 0;
    }

    if (ferror(fp_out)) {
        fclose (fp_out);
        unlink (tmppath);
        return -1;
    }

    if (fclose(fp_out) != 0) {
        unlink (tmppath);
        return -1;
    }

#ifdef WIN32
    unlink (path);	/* Grrrrh! */
#endif
    if (rename (tmppath, path) != 0) {
        unlink (tmppath);
        return -1;
    }

    return 0;
}

static void
parse_windata (void *arg, const char *n, const char *v) {
    struct ctxt_prefs *ctxt_prefs = (struct ctxt_prefs *)arg;
    if (strncmp (ctxt_prefs->tmpbuf, n, strlen (ctxt_prefs->tmpbuf)) == 0) {
        const char *kp;

        kp = &n[strlen (ctxt_prefs->tmpbuf)];
        if (strncmp (kp, NAME_SUFFIX_X, strlen (NAME_SUFFIX_X)) == 0) {
            ctxt_prefs->d.windata.x = atoi (v);
        } else if (strncmp (kp, NAME_SUFFIX_Y, strlen (NAME_SUFFIX_Y)) == 0) {
            ctxt_prefs->d.windata.y = atoi (v);
        } else if (strncmp (kp, NAME_SUFFIX_W, strlen (NAME_SUFFIX_W)) == 0) {
            ctxt_prefs->d.windata.w = atoi (v);
        } else if (strncmp (kp, NAME_SUFFIX_H, strlen (NAME_SUFFIX_H)) == 0) {
            ctxt_prefs->d.windata.h = atoi (v);
        } else if (strncmp (kp, NAME_SUFFIX_STATE, strlen (NAME_SUFFIX_STATE)) == 0) {
            if (strcasecmp (v, NAME_SUFFIX_STATE_CLOSED) == 0) {
                ctxt_prefs->d.windata.state = 0;
            } else {
                ctxt_prefs->d.windata.state = 1;
            }
        }
    }
}

static void
parse_intdata (void *arg, const char *n, const char *v) {
    struct ctxt_prefs *ctxt_prefs = (struct ctxt_prefs *)arg;
    if (strncmp (ctxt_prefs->tmpbuf, n, strlen (ctxt_prefs->tmpbuf)) == 0) {
        const char *kp;

        kp = &n[strlen (ctxt_prefs->tmpbuf)];
        if (strncmp (kp, NAME_SUFFIX_VALUE, strlen (NAME_SUFFIX_VALUE)) == 0) {
            ctxt_prefs->d.intdata.value = atoi (v);
            ctxt_prefs->d.intdata.isset = 1;
        }
    }
}

int
ui_gtk_prefs_load_window (const char *label, int initstate, GtkWindow *win)
{
    struct ctxt_prefs ctxt_prefs;
    memset (&ctxt_prefs, 0, sizeof (ctxt_prefs));
    ctxt_prefs.label	= label;
    ctxt_prefs.func	= parse_windata;
    strcpy (ctxt_prefs.tmpbuf, label);
    strcat (ctxt_prefs.tmpbuf, ".");
    ctxt_prefs.d.windata.win	= win;
    ctxt_prefs.d.windata.w	= -1;
    ctxt_prefs.d.windata.h	= -1;
    ctxt_prefs.d.windata.x	= -1;
    ctxt_prefs.d.windata.y	= -1;
    ctxt_prefs.d.windata.state	= -1;

    if (load_file (&ctxt_prefs) < 0)
        return -1;

    {
        gint sw, sh;
        gint ww, wh;
        sw = gdk_screen_width();
        sh = gdk_screen_height();
        gtk_window_get_size (ctxt_prefs.d.windata.win, &ww, &wh);

        if (ctxt_prefs.d.windata.w >= 0 && ctxt_prefs.d.windata.h >= 0) {
            if (ctxt_prefs.d.windata.w < 50)
                ctxt_prefs.d.windata.w = 50;
            if (ctxt_prefs.d.windata.w > sw)
                ctxt_prefs.d.windata.w = sw;

            if (ctxt_prefs.d.windata.h < 50)
                ctxt_prefs.d.windata.h = 50;
            if (ctxt_prefs.d.windata.h > sh)
                ctxt_prefs.d.windata.h = sh;

            gtk_window_set_default_size (ctxt_prefs.d.windata.win, ctxt_prefs.d.windata.w, ctxt_prefs.d.windata.h);
            ww = ctxt_prefs.d.windata.w;
            wh = ctxt_prefs.d.windata.h;
//trace("LOAD %s ww=%d; wh=%d\n", label, ww, wh);
//GdkWindowHints hints = 0;
//GdkGeometry geom;
//geom.
//hints |= GDK_HINT_USER_POS;
//hints | = GDK_HINT_USER_SIZE;

        }
        if (ctxt_prefs.d.windata.x >= 0 || ctxt_prefs.d.windata.y >= 0) {
            if (ctxt_prefs.d.windata.x < 0)
                ctxt_prefs.d.windata.x = 0;
//            if (ctxt_prefs.d.windata.x > sw - ww)
//                ctxt_prefs.d.windata.x = sw - ww;

            if (ctxt_prefs.d.windata.y < 0)
                ctxt_prefs.d.windata.y = 0;
//            if (ctxt_prefs.d.windata.y > sh - wh)
//                ctxt_prefs.d.windata.y = sh - wh;

            gtk_window_move (ctxt_prefs.d.windata.win, ctxt_prefs.d.windata.x, ctxt_prefs.d.windata.y);
        }
    }

    if (initstate && ctxt_prefs.d.windata.state >= 0) {
        if (ctxt_prefs.d.windata.state)
            gtk_widget_show_all (GTK_WIDGET (ctxt_prefs.d.windata.win));
        else
            gtk_widget_hide_all (GTK_WIDGET (ctxt_prefs.d.windata.win));
    } else {
        gtk_widget_show_all (GTK_WIDGET (ctxt_prefs.d.windata.win));
    }

    return 0;
}

int ui_gtk_prefs_int (const char *label, int *intptr) {
    struct ctxt_prefs ctxt_prefs;
    memset (&ctxt_prefs, 0, sizeof (ctxt_prefs));
    ctxt_prefs.label	= label;
    ctxt_prefs.func	= parse_intdata;
    strcpy (ctxt_prefs.tmpbuf, label);
    strcat (ctxt_prefs.tmpbuf, ".");
    ctxt_prefs.d.intdata.value	= -1;
    ctxt_prefs.d.intdata.isset	= 0;

    if (load_file (&ctxt_prefs) < 0)
        return -1;

    if (ctxt_prefs.d.intdata.isset) {
        *intptr = ctxt_prefs.d.intdata.value;
        return 0;
    }

    return -1;
}

void
ui_gtk_prefs_save_window (const char *label, GtkWindow *win)
{
    gint x, y, h, w;
    int state;

    gtk_window_get_position (win, &x, &y);
    gtk_window_get_size (win, &w, &h);
    state = GTK_WIDGET_VISIBLE (win);

#ifdef JMPDEBUG
    fprintf(stderr, "%s w=%d; h=%d; x=%d; y=%d; state=%d\n", label, w, h, x, y, state);
    fprintf(stderr, "%s.%s=%d\n", label, NAME_SUFFIX_W, w);
    fprintf(stderr, "%s.%s=%d\n", label, NAME_SUFFIX_H, h);
    fprintf(stderr, "%s.%s=%d\n", label, NAME_SUFFIX_X, x);
    fprintf(stderr, "%s.%s=%d\n", label, NAME_SUFFIX_Y, y);
    fprintf(stderr, "%s.%s=%s\n", label, NAME_SUFFIX_STATE, state ? NAME_SUFFIX_STATE_OPEN : NAME_SUFFIX_STATE_CLOSED);
#endif

    if (save_file (label, w, h, x, y, state) < 0) {
        fprintf(stderr, "save_file(): %s\n", strerror(errno));
    }
}


/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
