/*
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   MODULE_NAME:  logdump.c
 *
 *   COMPONENT_NAME:  jfs utilities
 *
 *   FUNCTIONS:
 *       main
 *       parse_parms
 *
 *   This  *** SERVICE-ONLY ***  tool dumps the contents of
 *   the journal log on the specified JFS partition into 
 *   "./jfslog.dmp"
 *
 */
#include "config.h"

#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <unistd.h>
#include <pwd.h>

#include "jfs_types.h"
#include "devices.h"
#include "jfs_version.h"

#define LOGDMP_OK  0
#define LOGDMP_FAILED -1

#define  FULLLOG  -1
#define  CURRLOG   1

int32_t jfs_logdump( char *, int32_t, int32_t );
int32_t parse_parms( int32_t, char ** );

 /* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
  *
  * Device information.
  *
  *     values are assigned when (if) the device is opened.
  */
HFILE     Dev_IOPort;
uint32_t  Dev_blksize;
int32_t   Dev_SectorSize;

char  *Vol_Label = NULL;

int32_t dump_all = 0;

 /* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
  *
  * For message processing in called routines
  *
  */

char message_parm_0[64];
char message_parm_1[64];
char message_parm_2[64];
char message_parm_3[64];
char message_parm_4[64];
char message_parm_5[64];
char message_parm_6[64];
char message_parm_7[64];
char message_parm_8[64];
char message_parm_9[64];

char *msgprms[10] = { message_parm_0, message_parm_1, message_parm_2,
                      message_parm_3, message_parm_4, message_parm_5,
                      message_parm_6, message_parm_7, message_parm_8,
                      message_parm_9 };

int16_t msgprmidx[10] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

char message_buffer_0[256];
char message_buffer_1[1024];
char message_buffer_2[1024];
char *terse_msg_ptr =   &message_buffer_0[0];
char *verbose_msg_ptr = &message_buffer_1[0];
char *msgs_txt_ptr =    &message_buffer_2[0];
unsigned long msgs_txt_maxlen = 1024;

extern char *MsgText[];

extern void fsck_send_msg( int, int, int );  /* defined in fsckmsg.c */
extern int alloc_wrksp( uint32_t, int, int, void ** ); /* defined in fsckwsp.c */

/****************************************************************************
 *
 * NAME: main
 *
 * FUNCTION: call jfs_logdmp()
 *
 * INTERFACE:  
 *            logdump [-a] <block device> 
 *  
 *  where -a => dump entire contents of log instead of just 
 *    committed transactions since last synch point.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: 0
 *      failure: something else
 */
int main(int argc, char **argv)
{
  int      rc = LOGDMP_OK;
  int32_t  sectorsize;

  printf( "logdump version %s, %s\n", 
          VERSION, JFSUTILS_DATE);

  rc = parse_parms( argc, argv );  /* parse the parms and record
                                    * them in the aggregate wsp record
                                    */

  if( rc == LOGDMP_OK ) {  /* parms are good */

    printf("Device Name: %s\n", Vol_Label);

    rc = ujfs_open_device(Vol_Label, &Dev_IOPort, &sectorsize, READONLY);
    
    if( rc == LOGDMP_OK ) {
      rc = jfs_logdump( Vol_Label, Dev_IOPort, dump_all  );
    } else {
      fprintf(stderr,"Error: Cannot open device %s, rc = %d.\n",
              Vol_Label, rc);
      return(-1);
    }

        /*
         * Close the device
         */
    rc = ujfs_close(Dev_IOPort);

  }  /* end parms are good */

  return( rc );
}  /* end of main() */

/* ======================================================================*/


/*****************************************************************************
 * NAME: parse_parms
 *
 * FUNCTION:  Parse the invocation parameters.  If any unrecognized
 *            parameters are detected, or if any required parameter is
 *            omitted, issue a message and exit.
 *
 * PARAMETERS:  as specified to main()     
 *
 * NOTES:
 *
 * RETURNS:
 *      success: LOGDMP_OK
 *      failure: something else
 */
int32_t parse_parms ( int32_t argc, char **argv )
{
  int32_t  pp_rc = LOGDMP_OK;
  int      i = 0;

  char      *argp;

  for (i = 1; i < argc; i++) {  /* for all parms on command line */
    argp = argv[i];

    if (*argp == '-') {  /* leading - */
        argp++;
        if (*argp == 'a' || *argp == 'A') {  /* debug */
              /*
               * Dump entire log Option
               */
            dump_all = -1;
        } else { 
              /* 
               * unrecognized keyword parm 
               */
            printf( "LOGDUMP:  unrecognized keyword detected:   %s\n", argp );
            return( LOGDMP_FAILED );
        }

    } else if (argp[0] == '/') {  /* 2nd char is / */
        Vol_Label = argp;

    } else {  /* unrecognized parm */
        printf( "LOGDUMP:  unsupported parameter detected:   %s\n", argp );
        return( LOGDMP_FAILED );
    }  /* end unrecognized parm */
  }  /* end for all parms on command line */

  if (Vol_Label == NULL) {  /* no device specified */
      printf( "Error: no device given.\n");       
      printf( "Usage: logdump [-a] <block device>\n" );
      return( LOGDMP_FAILED );
  }  /* end no device specified */

  return( pp_rc );
}  /* end of parse_parms()  */
