/////////////////////////////////////////////////////////////////////////////
// This file is part of the "Java-DAP" project, a Java implementation
// of the OPeNDAP Data Access Protocol.
//
// Copyright (c) 2010, OPeNDAP, Inc.
// Copyright (c) 2002,2003 OPeNDAP, Inc.
// 
// Author: James Gallagher <jgallagher@opendap.org>
// 
// All rights reserved.
// 
// Redistribution and use in source and binary forms,
// with or without modification, are permitted provided
// that the following conditions are met:
// 
// - Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
// 
// - Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// 
// - Neither the name of the OPeNDAP nor the names of its contributors may
//   be used to endorse or promote products derived from this software
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
// IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
// TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
/////////////////////////////////////////////////////////////////////////////

package opendap.dap.Server;

import java.util.*;

/**
 * Represents a clause which invokes a function that returns a boolean value.
 *
 * @author joew
 * @see ClauseFactory
 */
public class BoolFunctionClause
        extends AbstractClause
        implements TopLevelClause {

    /**
     * Creates a new BoolFunctionClause.
     *
     * @param function The function invoked by the clause
     * @param children A list of SubClauses, to be given as arguments
     *                 to the function. If all the arguments are constant, the function
     *                 clause will be flagged as constant, and evaluated immediatelyx.
     * @throws DAP2ServerSideException Thrown if either 1) the function does not
     *                        accept the arguments given, or 2) the
     *                        clause is constant, and the attempt to evaluate it fails.
     */
    protected BoolFunctionClause(BoolFunction function,
                                 List children)
            throws DAP2ServerSideException {

        function.checkArgs(children);
        this.function = function;
        this.children = children;
        this.constant = true;
        Iterator it = children.iterator();
        while (it.hasNext()) {
            SubClause current = (SubClause) it.next();
            current.setParent(this);
            if (!current.isConstant()) {
                constant = false;
            }
        }
        if (constant) {
            evaluate();
        }
    }

    public boolean getValue() {
        return value;
    }

    public boolean evaluate()
            throws DAP2ServerSideException {

        if (!constant || !defined) {
            defined = true;
            value = function.evaluate(children);
        }
        return value;
    }

    /**
     * Returns the server-side function invoked by this clause
     */
    public BoolFunction getFunction() {
        return function;
    }

    /**
     * Prints the original string representation of this clause.
     * For use in debugging.
     */
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append(function.getName());
        buf.append("(");
        Iterator it = children.iterator();
        if (it.hasNext()) {
            buf.append(it.next().toString());
        }
        while (it.hasNext()) {
            buf.append(",");
            buf.append(it.next().toString());
        }
        buf.append(")");
        return buf.toString();
    }

    protected BoolFunction function;

    protected boolean value;
}


