/* ------------------------------------------------------------------------
 * $Id: TransformImpl.cc,v 1.11 2001/08/21 21:37:40 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-08-14 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <math.h>

// -- 3Dwm Includes
#include "Celsius/Math.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Celsius/Matrix3D.hh"
#include "Celsius/Vector3D.hh"
#include "Polhem/TransformImpl.hh"

using namespace Nobel;

// -- Code Segment

// @@@OpenGL matrices are column-major, not row-major (like in C)!!!

TransformImpl::TransformImpl()
{
    _matrix.identity();
}

TransformImpl::TransformImpl(const TransformImpl &t)
{
    *this = t;
}

TransformImpl::~TransformImpl()
{
    // empty
}
    
void TransformImpl::load(const Matrix m)
{
    Guard<Mutex> guard(_mutex);
    _matrix.load(m);
}

void TransformImpl::store(Matrix m)
{
    Guard<Mutex> guard(_mutex);
    _matrix.store(m);
}

void TransformImpl::premultiply(Transform_ptr t)
{
    Guard<Mutex> guard(_mutex);

    // @@@NEED TO OPTIMIZE THIS!    
    Matrix m;
    t->store(m);
    Matrix3D trafo(m);
    _matrix = _matrix * trafo;
}

void TransformImpl::postmultiply(Transform_ptr t)
{
    Guard<Mutex> guard(_mutex);

    // @@@NEED TO OPTIMIZE THIS!
    Matrix m;
    t->store(m);
    Matrix3D trafo(m);
    _matrix = trafo * _matrix;
}

void TransformImpl::identity()
{
    Guard<Mutex> guard(_mutex);
    _matrix.identity();
}

void TransformImpl::scale(const Vertex3D &v)
{
    Guard<Mutex> guard(_mutex);
    
    // Scale the matrix components
    _matrix(0, 0) *= v.x; _matrix(0, 1) *= v.x; _matrix(0, 2) *= v.x; 
    _matrix(1, 0) *= v.y; _matrix(1, 1) *= v.y; _matrix(1, 2) *= v.y; 
    _matrix(2, 0) *= v.z; _matrix(2, 1) *= v.z; _matrix(2, 2) *= v.z; 
}

void TransformImpl::rotate(CORBA::Float angle, const Vertex3D &axis)
{
    // Compute radians
    float radians = angle * Math::radians_per_degree;
    float s = sin(radians), c = cos(radians); 
    
    // Normalize axis and compute common factors
    Vector3D u(axis.x, axis.y, axis.z);
    u.normalize();
    float xx = u.x() * u.x(), yy = u.y() * u.y(), zz = u.z() * u.z();
    float xy = u.x() * u.y(), zx = u.x() * u.z(), yz = u.y() * u.z();
    float xs = u.x() * s,     ys = u.y() * s,     zs = u.z() * s;
    float one_c = 1. - c;

    // Compute rotation matrix
    Matrix3D m;
    
    m(0, 0) = (one_c * xx) + c;  m(0, 1) = (one_c * xy) - zs;
    m(0, 2) = (one_c * zx) + ys; 
    
    m(1, 0) = (one_c * xy) + zs; m(1, 1) = (one_c * yy) + c;
    m(1, 2) = (one_c * yz) - xs; 

    m(2, 0) = (one_c * zx) - ys; m(2, 1) = (one_c * yz) + xs;
    m(2, 2) = (one_c * zz) + c;  
    
    m(3, 3) = 1.;

    // Postmultiply the rotation matrix to the current matrix
    //_matrix = _matrix * m;
    _matrix = m * _matrix;
}

void TransformImpl::translate(const Vertex3D& v)
{
    // @@@ Is this really correct? We need to explore this.
    _matrix(0, 3) += v.x; _matrix(1, 3) += v.y; _matrix(2, 3) += v.z;
}

void TransformImpl::invert()
{
    _matrix.invert();
}

TransformImpl &TransformImpl::operator = (const TransformImpl &t)
{
    //Guard<Mutex> guard(t._mutex);
    _matrix = t._matrix;
    return *this;
}
