/* ------------------------------------------------------------------------
 * $Id: Face.hh,v 1.2 2001/07/30 15:18:58 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-17 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Face_hh_
#define _Face_hh_

// -- Class Declarations

/**
 * TriangleFaceface class.
 **/
class TriangleIndex {
public:
    
    /**
     * Constructor. Triangles are normally specified using
     * counter-clockwise winding.
     *
     * @param na first index.
     * @param nb second index.
     * @param nc third index.
     **/
    TriangleIndex(int na = 0, int nb = 0, int nc = 0) {
	a() = na; b() = nb; c() = nc;
    }
    
    /**
     * Mutable indexing operator.
     *
     * @param index number of index to retrieve.
     * @return desired index (mutable).
     **/
    int &operator [] (int index) {
	return _indices[index];
    }

    /**
     * Const indexing operator.
     *
     * @param index number of index to retrieve.
     * @return desired index (const).
     **/
    int operator [] (int index) const {
	return _indices[index];
    }
    
    /**
     * Retrieve the first triangle face index.
     *
     * @return first triangle index (mutable).
     **/
    int &a() { return _indices[0]; }

    /**
     * Retrieve the second triangle face index.
     *
     * @return second triangle index (mutable).
     **/
    int &b() { return _indices[1]; }

    /**
     * Retrieve the third triangle face index.
     *
     * @return third triangle index (mutable).
     **/
    int &c() { return _indices[2]; }

    /**
     * Retrieve the first triangle face index.
     *
     * @return first triangle index (const).
     **/
    int a() const { return _indices[0]; }

    /**
     * Retrieve the second triangle face index.
     *
     * @return second triangle index (const).
     **/
    int b() const { return _indices[1]; }

    /**
     * Retrieve the third triangle face index.
     *
     * @return third triangle index (const).
     **/
    int c() const { return _indices[2]; }

    /**
     * Addition operator. Adding an integer constant to a triangle
     * index will simply increase each index with the integer value,
     * acting as a way of offsetting the indices.
     *
     * @param offset integer value to offset the indices with.
     * @return reference to the modified triangle index.
     **/ 
    TriangleIndex &operator += (int offset) {
	a() += offset; b() += offset; c() += offset;
	return *this;
    }

    /**
     * Flip the winding of the triangle. A triangle wound
     * counter-clockwise will end up wound clockwise, and vice versa.
     **/
    void flip() {
	int temp = a(); a() = c(); c() = temp;
    }
    
private:
    int _indices[3];
};

/**
 * TriangleFacestruct. 
 **/
struct TriangleFace {

    /// Vertex indices
    TriangleIndex v;

    /// Normal indices
    TriangleIndex n;

    /// Texture coordinate indices
    TriangleIndex tc;

    /// Material index
    int material;
};

#endif /* Face.hh */
