/* ------------------------------------------------------------------------
 * $Id: ThreadQueue.hh,v 1.6 2001/07/18 07:47:49 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-06-19 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _ThreadQueue_hh_
#define _ThreadQueue_hh_

// -- System Includes
#include <queue>

// -- Celsius Includes
#include "Mutex.hh"
#include "Semaphore.hh"
#include "Exception.hh"
#include "Guard.hh"

// -- Template Definitions

/**
 * Simple thread queue implemented using POSIX mutexes and semaphores.
 * This is the basic building-block of any producer-consumer scheme
 * for making two threads talk to each other.
 **/
template <class T>
class ThreadQueue : private std::queue<T> {
    
private:
    
    typedef std::queue<T> rep_type;
    
public: 

    /**
     * Constructor.
     *
     * @param capacity the maximum number of items in the queue.
     **/
    ThreadQueue(size_t capacity) : _allowWrite(capacity), _allowRead(0) { }

    /**
     * Push an element to the end of the queue.
     * 
     * @param t element to push.
     **/
    void push(const T &t) {
	_allowWrite.lock();
	rep_type::push(t);
	_allowRead.unlock();
    }

    /**
     * Erase the element at the front of the queue.
     **/
    T pop() {
	_allowRead.lock();
	Guard<Mutex> guard(_mutex);
	T t = rep_type::front();
	rep_type::pop();
	_allowWrite.unlock();
	return t;
    }

    /**
     * Try to retrieve (and remove) the element at the front of the
     * queue. If there is nothing to retrieve (i.e. the queue is
     * empty), signal this to the caller.
     *
     * @param t element at the top of the queue.
     * @return true if the operation was successful, false if not.
     **/
    bool try_top_pop(T &t) {
	if (_allowRead.tryLock() == true) {
	    Guard<Mutex> guard(_mutex);
	    t = rep_type::top();
	    rep_type::pop();
	    _allowWrite.unlock();
	    return true;
	}
	else
	    return false;
    }
    
    /**
     * Retrieve the size of the queue.
     *
     * @return the queue size.
     **/
    size_t size() const {
	Guard<Mutex> guard(_mutex);
	return rep_type::size(); 
    }
    
private:
    Semaphore _allowWrite;
    Semaphore _allowRead;
    Mutex _mutex;
};

#endif /* ThreadQueue.hh */
