/* ------------------------------------------------------------------------
 * $Id: client.cc,v 1.2 2001/07/30 15:18:52 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-24 by Steve Houston.
 *
 * Copyright (c) 2001 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <iostream>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

// -- 3Dwm Includes
#include <Nobel++/Nobel.hh>
#include <Nobel/Shape.hh>
#include <Nobel/NodeKit.hh>
#include <Nobel/Appearance.hh>
#include <Nobel/GeometryKit.hh>
#include <Nobel/TriangleGeometry.hh>
#include <Nobel/Transform.hh>

using namespace Nobel;

// -- Local Function Prototypes

static TriangleGeometry_var createCubeGeo( GeometryKit_ptr geo_kit );

// -- Code Segment

int main(int argc, char *argv[]) {
    
    try {

	// Initialize the connection
	Nobelxx::Client client("Picker Client", argc, argv);
	       
	// Resolve the necessary kits
	NodeKit_var node_kit = 
	    client.resolve<NodeKit>(Nobelxx::name<NodeKit>());
	GeometryKit_var geo_kit =
	    client.resolve<GeometryKit>(Nobelxx::name<GeometryKit>());
	
	// And here's the scene graph root (local to the client)
	Node_var root = client.root();

	// Create some CORBA objects
	Shape_var shape1 = node_kit->createShape();
	Appearance_var appear1 = geo_kit->createAppearance();
	TriangleGeometry_var cubeGeo = createCubeGeo(geo_kit);

	appear1->lighting(true);

	Nobel::Color red = {0.75f, 0.0f, 0.0f, 0.0f};
	Nobel::Color green = {0.0f, 0.75f, 0.0f, 0.0f};
	Nobel::Color blue = {0.0f, 0.0f, 0.75f, 0.0f};
	Nobel::MaterialAttributes material;

	//
	// Red cube
	//
	material.diffuse = red;
	appear1->material(material);
	shape1->setAppearance(appear1);

	Transform_var trafo = shape1->transformation();
	Vertex3D value1 = { -1.0f, 1.0f, -8.0f };
	trafo->translate(value1);

	// Add geometry to shape and shape to root
	shape1->addGeometry(cubeGeo);
	root->insert(shape1);

	//
	// Green cube
	//
	Shape_var shape2 = node_kit->createShape();
	Appearance_var appear2 = geo_kit->createAppearance();
	appear2->lighting(true);

	material.diffuse = green;
	appear2->material(material);
	shape2->setAppearance(appear2);
	
	trafo = shape2->transformation();
	Vertex3D value2 = { 0.0f, 0.0f, -5.0f };
	trafo->translate(value2);

	// TODO: Shouldn't we be able to share a single cube geometry
	// between all shapes?
	TriangleGeometry_var cubeGeo2 = createCubeGeo(geo_kit);
	shape2->addGeometry(cubeGeo2);
	root->insert(shape2);

	//
	// Blue cube
	//
	Shape_var shape3 = node_kit->createShape();
	Appearance_var appear3 = geo_kit->createAppearance();
	appear3->lighting(true);

	material.diffuse = blue;
	appear3->material(material);
	shape3->setAppearance(appear3);
	
	trafo = shape3->transformation();
	Vertex3D value3 = { 2.0f, 0.0f, -5.0f };
	trafo->translate(value3);

	// TODO: Shouldn't we be able to share a single cube geometry
	// between all shapes?
	TriangleGeometry_var cubeGeo3 = createCubeGeo(geo_kit);
	shape3->addGeometry(cubeGeo3);
	root->insert(shape3);
    }
    catch (const CORBA::Exception &e) {

	// The ORB threw an exception
	std::cerr << "Uncaught CORBA exception: " << e << std::endl;
	return EXIT_FAILURE;
    }
    catch (...) {
	
	// Another exception was raised
	std::cerr << "Exception thrown. Exiting." << std::endl;
	return EXIT_FAILURE;
    }
    
    return EXIT_SUCCESS;
}

TriangleGeometry_var createCubeGeo( GeometryKit_ptr geo_kit ) 
{

    TriangleGeometry_var geo = geo_kit->createTriangleGeometry();

    // Initialise the vertex list
    Vertex3D vertex;
    geo->setVertexNumber(8);

    vertex.x = -0.5f; vertex.y = -0.5f; vertex.z = 0.5f;
    geo->setVertex(0, vertex);

    vertex.x = 0.5f; vertex.y = -0.5f; vertex.z = 0.5f;
    geo->setVertex(1, vertex);

    vertex.x = 0.5f; vertex.y = 0.5f; vertex.z = 0.5f;
    geo->setVertex(2, vertex);

    vertex.x = -0.5f; vertex.y = 0.5f; vertex.z = 0.5f;
    geo->setVertex(3, vertex);

    vertex.x = -0.5f; vertex.y = -0.5f; vertex.z = -0.5f;
    geo->setVertex(4, vertex);

    vertex.x = 0.5f; vertex.y = -0.5f; vertex.z = -0.5f;
    geo->setVertex(5, vertex);

    vertex.x = 0.5f; vertex.y = 0.5f; vertex.z = -0.5f;
    geo->setVertex(6, vertex);

    vertex.x = -0.5f; vertex.y = 0.5f; vertex.z = -0.5f;
    geo->setVertex(7, vertex);

    // Initialize the normal list
    Vertex3D normal;
    geo->setNormalNumber(6);

    // Front
    normal.x = 0.0f; normal.y = 0.0f; normal.z = 1.0f;
    geo->setNormal(0, normal);

    // Right
    normal.x = 1.0f; normal.y = 0.0f; normal.z = 0.0f;
    geo->setNormal(1, normal);

    // Back
    normal.x = 0.0f; normal.y = 0.0f; normal.z = -1.0f;
    geo->setNormal(2, normal);

    // Left
    normal.x = -1.0f; normal.y = 0.0f; normal.z = 0.0f;
    geo->setNormal(3, normal);

    // Top
    normal.x = 0.0f; normal.y = 1.0f; normal.z = 0.0f;
    geo->setNormal(4, normal);

    // Bottom
    normal.x = 0.0f; normal.y = -1.0f; normal.z = 0.0f;
    geo->setNormal(5, normal);

    // Set the indices
    geo->setFaceNumber(12);
    Triangle face, normals;

    // Front
    face.a = 0; face.b = 1; face.c = 3;
    normals.a = 0; normals.b = 0; normals.c = 0;
    geo->setVertexIndex(0, face);
    geo->setNormalIndex(0, normals);

    face.a = 1; face.b = 2;
    face.c = 3;
    normals.a = 0; normals.b = 0; normals.c = 0;
    geo->setVertexIndex(1, face);
    geo->setNormalIndex(1, normals);

    // Right
    face.a = 1; face.b = 5; face.c = 2;
    normals.a = 1; normals.b = 1; normals.c = 1;
    geo->setVertexIndex(2, face);
    geo->setNormalIndex(2, normals);

    face.a = 5; face.b = 6; face.c = 2;
    normals.a = 1; normals.b = 1; normals.c = 1;
    geo->setVertexIndex(3, face);
    geo->setNormalIndex(3, normals);

    // Back
    face.a = 5; face.b = 4; face.c = 6;
    normals.a = 2; normals.b = 2; normals.c = 2;
    geo->setVertexIndex(4, face);
    geo->setNormalIndex(4, normals);

    face.a = 4; face.b = 7; face.c = 6;
    normals.a = 2; normals.b = 2; normals.c = 2;
    geo->setVertexIndex(5, face);
    geo->setNormalIndex(5, normals);

    // Left
    face.a = 4; face.b = 0; face.c = 7;
    normals.a = 3; normals.b = 3; normals.c = 3;
    geo->setVertexIndex(6, face);    
    geo->setNormalIndex(6, normals);

    face.a = 0; face.b = 3; face.c = 7;
    normals.a = 3; normals.b = 3; normals.c = 3;
    geo->setVertexIndex(7, face);
    geo->setNormalIndex(7, normals);

    // Top
    face.a = 3; face.b = 2; face.c = 7;
    normals.a = 4; normals.b = 4; normals.c = 4;
    geo->setVertexIndex(8, face);
    geo->setNormalIndex(8, normals);

    face.a = 2; face.b = 6; face.c = 7;
    normals.a = 4; normals.b = 4; normals.c = 4;
    geo->setVertexIndex(9, face);
    geo->setNormalIndex(9, normals);

    // Bottom

    face.a = 0; face.b = 4; face.c = 1;
    normals.a = 5; normals.b = 5; normals.c = 5;
    geo->setVertexIndex(10, face);
    geo->setNormalIndex(10, normals);

    face.a = 4; face.b = 5; face.c = 1;
    normals.a = 5; normals.b = 5; normals.c = 5;
    geo->setVertexIndex(11, face);
    geo->setNormalIndex(11, normals);

    return geo;
}
