/*
	**
	** sort-report.c 
	**
	** Sorts hosts and peers lists
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: sort-report.c,v 1.2 1999/02/04 10:22:42 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>

#include <glib.h>

#include "sort-report.h"
#include "report.h"
#include "util.h"

static char rcsid[] = "$Id: sort-report.c,v 1.2 1999/02/04 10:22:42 dmiller Exp $";

/* Function used to compare peers */
typedef int (*peer_compare_func_t)(peer_t *peer1, peer_t *peer2);

/* Function used to compare hosts */
typedef int (*host_compare_func_t)(host_t *host1, host_t *host2);

/* Prototypes */
host_compare_func_t	choose_host_comparison_function(int sort_mode);
peer_compare_func_t	choose_peer_comparison_function(int sort_mode);
void	insert_host(host_t *host, host_t *root, host_compare_func_t	compare, int host_reverse);
void	insert_peer(peer_t *peer, peer_t *root, peer_compare_func_t compare, int peer_reverse);
host_t *randomise_hosts(host_t *hl);
peer_t *randomise_peers(peer_t *pl);

/* Comparision function prototypes */
int	host_compare_none(host_t *host1, host_t *host2);
int	host_compare_bytes(host_t *host1, host_t *host2);
int	host_compare_bytes_sent(host_t *host1, host_t *host2);
int	host_compare_bytes_received(host_t *host1, host_t *host2);
int	host_compare_packets(host_t *host1, host_t *host2);
int	host_compare_packets_sent(host_t *host1, host_t *host2);
int	host_compare_packets_received(host_t *host1, host_t *host2);
int	host_compare_connections(host_t *host1, host_t *host2);
int	host_compare_connections_sent(host_t *host1, host_t *host2);
int	host_compare_connections_received(host_t *host1, host_t *host2);
int	host_compare_peers(host_t *host1, host_t *host2);
int	host_compare_hostname(host_t *host1, host_t *host2);
int	host_compare_ip_addr(host_t *host1, host_t *host2);
int	host_compare_first_seen(host_t *host1, host_t *host2);
int	host_compare_last_seen(host_t *host1, host_t *host2);

int	peer_compare_none(peer_t *peer1, peer_t *peer2);
int	peer_compare_bytes(peer_t *peer1, peer_t *peer2);
int	peer_compare_packets(peer_t *peer1, peer_t *peer2);
int	peer_compare_connections(peer_t *peer1, peer_t *peer2);

/* Chooses host comparison function based on sort mode key */
host_compare_func_t choose_host_comparison_function(int sort_mode)
{
	switch(sort_mode)
	{
		case SORT_NONE:
			return(host_compare_none);
		case SORT_BYTES:
			return(host_compare_bytes);
		case SORT_BYTES_SENT:
			return(host_compare_bytes_sent);
		case SORT_BYTES_RECEIVED:
			return(host_compare_bytes_received);
		case SORT_PACKETS:
			return(host_compare_packets);
		case SORT_PACKETS_SENT:
			return(host_compare_packets_sent);
		case SORT_PACKETS_RECEIVED:
			return(host_compare_packets_received);
		case SORT_CONNECTS:
			return(host_compare_connections);
		case SORT_CONNECTS_SENT:
			return(host_compare_connections_sent);
		case SORT_CONNECTS_RECEIVED:
			return(host_compare_connections_received);
		case SORT_PEERS:
			return(host_compare_peers);
		case SORT_HOSTNAME:
			return(host_compare_hostname);
		case SORT_IP_ADDR:
			return(host_compare_ip_addr);
		case SORT_FIRST_SEEN:
			return(host_compare_first_seen);
		case SORT_LAST_SEEN:
			return(host_compare_last_seen);
		default:
			abort();
	}
}

/* Chooses peer comparison function based on sort mode key */
peer_compare_func_t choose_peer_comparison_function(int sort_mode)
{
	switch(sort_mode)
	{
		case SORT_NONE:
			return(peer_compare_none);
		case SORT_BYTES:
			return(peer_compare_bytes);
		case SORT_PACKETS:
			return(peer_compare_packets);
		case SORT_CONNECTS:
			return(peer_compare_connections);
		default:
			abort();
	}
}

/* Parses textual host sort key into numeric identifier */
int parse_host_sort_mode(const char *sort_mode)
{
	/* Make sure we got some input */
	if ((sort_mode == NULL) || (sort_mode[0] == '\0'))
		return(-1);
	
	/* Check for single character options */
	if (sort_mode[1] == '\0')
	{
		switch (sort_mode[0])
		{
			case 'N':
				return(SORT_NONE);
			case 'b':
				return(SORT_BYTES);
			case 's':
				return(SORT_BYTES_SENT);
			case 'r':
				return(SORT_BYTES_RECEIVED);
			case 'p':
				return(SORT_PACKETS);
			case 'S':
				return(SORT_PACKETS_SENT);
			case 'R':
				return(SORT_PACKETS_RECEIVED);
			case 'C':
				return(SORT_CONNECTS);
			case 'c':
				return(SORT_CONNECTS_SENT);
			case 'l':
				return(SORT_CONNECTS_RECEIVED);
			case 'P':
				return(SORT_PEERS);
			case 'h':
				return(SORT_HOSTNAME);
			case 'I':
				return(SORT_IP_ADDR);
			case 'F':
				return(SORT_FIRST_SEEN);
			case 'L':
				return(SORT_LAST_SEEN);
			case 'X':
				return(SORT_RANDOM);
			
			default:
				return(-1);
		}
	}
	
	/* Otherwise, look for textual sort key */
	if (g_strcasecmp(sort_mode, "none") == 0)
		return(SORT_NONE);
	else if (g_strcasecmp(sort_mode, "bytes") == 0)
		return(SORT_BYTES);
	else if (g_strcasecmp(sort_mode, "bsent") == 0)
		return(SORT_BYTES_SENT);
	else if (g_strcasecmp(sort_mode, "breceived") == 0)
		return(SORT_BYTES_RECEIVED);
	else if (g_strcasecmp(sort_mode, "packets") == 0)
		return(SORT_PACKETS);
	else if (g_strcasecmp(sort_mode, "psent") == 0)
		return(SORT_PACKETS_SENT);
	else if (g_strcasecmp(sort_mode, "preceived") == 0)
		return(SORT_PACKETS_RECEIVED);
	else if (g_strcasecmp(sort_mode, "connects") == 0)
		return(SORT_CONNECTS);
	else if (g_strcasecmp(sort_mode, "csent") == 0)
		return(SORT_CONNECTS_SENT);
	else if (g_strcasecmp(sort_mode, "creceived") == 0)
		return(SORT_CONNECTS_RECEIVED);
	else if (g_strcasecmp(sort_mode, "peers") == 0)
		return(SORT_PEERS);
	else if (g_strcasecmp(sort_mode, "hostname") == 0)
		return(SORT_HOSTNAME);
	else if (g_strcasecmp(sort_mode, "ipaddr") == 0)
		return(SORT_IP_ADDR);
	else if (g_strcasecmp(sort_mode, "firstseen") == 0)
		return(SORT_FIRST_SEEN);
	else if (g_strcasecmp(sort_mode, "lastseen") == 0)
		return(SORT_LAST_SEEN);
	else if (g_strcasecmp(sort_mode, "random") == 0)
		return(SORT_RANDOM);
	else
		return(-1);
}

/* Parses textual peer sort key into numeric identifier */
int parse_peer_sort_mode(const char *sort_mode)
{
	/* Make sure we got some input */
	if ((sort_mode == NULL) || (sort_mode[0] == '\0'))
		return(-1);
	
	/* Check for single character options */
	if (sort_mode[1] == '\0')
	{
		switch (sort_mode[0])
		{
			case 'N':
				return(SORT_NONE);
			case 'b':
				return(SORT_BYTES);
			case 'p':
				return(SORT_PACKETS);
			case 'C':
				return(SORT_CONNECTS);
			case 'X':
				return(SORT_RANDOM);
			
			default:
				return(-1);
		}
	}
	
	/* Otherwise, look for textual sort key */
	if (g_strcasecmp(sort_mode, "none") == 0)
		return(SORT_NONE);
	else if (g_strcasecmp(sort_mode, "bytes") == 0)
		return(SORT_BYTES);
	else if (g_strcasecmp(sort_mode, "packets") == 0)
		return(SORT_PACKETS);
	else if (g_strcasecmp(sort_mode, "connects") == 0)
		return(SORT_CONNECTS);
	else if (g_strcasecmp(sort_mode, "random") == 0)
		return(SORT_RANDOM);
	else
		return(-1);
}

void	insert_peer(peer_t *peer, peer_t *root, peer_compare_func_t compare, int peer_reverse)
{
	peer_t	*p;
	peer_t	*last_p;
	
	/* Initialise list pointers */
	p = root->next;
	last_p = root;
	
	/* Find correct position in list */	
	if (peer_reverse)
	{
		while ((p != NULL) && (compare(peer, p) > 0))
		{
			last_p = p;
			p = p->next;
		}
	} else
	{
		while ((p != NULL) && (compare(peer, p) <= 0))
		{
			last_p = p;
			p = p->next;
		}
	}
	
	/* last_p is the peer to insert after */
	peer->next = p;
	last_p->next = peer;
}

void	insert_host(host_t *host, host_t *root, host_compare_func_t	compare, int host_reverse)
{
	host_t	*h;
	host_t	*last_h;
	
	/* Initialise list pointers */
	h = root->next;
	last_h = root;
	
	/* Find correct position in list */	
	if (host_reverse)
	{
		while ((h != NULL) && (compare(host, h) > 0))
		{
			last_h = h;
			h = h->next;
		}
	} else
	{
		while ((h != NULL) && (compare(host, h) <= 0))
		{
			last_h = h;
			h = h->next;
		}
	}
	
	/* last_h is the peer to insert after */
	host->next = h;
	last_h->next = host;
}

host_t	*sort_hosts(host_t *host_list, int host_sort_mode, int host_reverse)
{
	host_t					*h;
	host_t					*next_h;
	host_compare_func_t	compare;
	host_t					root;

	/* Special case for random sort order */
	if (host_sort_mode == SORT_RANDOM)
		return(randomise_hosts(host_list));

	compare = choose_host_comparison_function(host_sort_mode);
	root.next = NULL;
	
	/* For each host in the list */
	h = host_list;
	while (h != NULL)
	{
		next_h = h->next;
		
		/* Place it in the new list */
		insert_host(h, &root, compare, host_reverse);

		/* Move on to the next */
		h = next_h;
	}
					
	return(root.next);
}

peer_t	*sort_peers(peer_t *peer_list, int peer_sort_mode, int peer_reverse)
{
	peer_t					*p;
	peer_t					*next_p;
	peer_compare_func_t	compare;
	peer_t					root;

	/* Special case for random sort order */
	if (peer_sort_mode == SORT_RANDOM)
		return(randomise_peers(peer_list));

	compare = choose_peer_comparison_function(peer_sort_mode);
	root.next = NULL;
	
	/* For each host in the list */
	p = peer_list;
	while (p != NULL)
	{
		next_p = p->next;

		/* Place it in the new list */
		insert_peer(p, &root, compare, peer_reverse);

		/* Move on to the next */
		p = next_p;
	}
					
	return(root.next);
}

void sort_all_peers(host_t *h, int peer_sort_mode, int peer_reverse)
{
	while(h != NULL)
	{
		h->peers = sort_peers(h->peers, peer_sort_mode, peer_reverse);
		h = h->next;
	}
}

host_t *randomise_hosts(host_t *hl)
{
	host_t		*h;
	host_t		**host_array;
	u_int32_t	n_hosts;
	u_int32_t	c;

	/* Count number of hosts in list */
	h = hl;
	n_hosts = 0;
	while(h != NULL)
	{
		n_hosts++;
		h = h->next;
	}

	/* Allocate array to randomise hosts into */
	host_array = util_zmalloc(sizeof(*host_array) * n_hosts);
	
	/* For each host in list, place it in a random position in the array */
	h = hl;
	while(h != NULL)	
	{
		/* Find a random empty position in array */
		c = random() % n_hosts;
		while (host_array[c] != NULL)
		{
			c++;
			c %= n_hosts;
		}
		
		/* Copy host there */
		host_array[c] = h;
		
		/* Move on to next host */
		h = h->next;
	}

	/* Make linked list order follow array's */
	for(c = 1; c < n_hosts; c++)
		host_array[c - 1]->next = host_array[c];

	/* Fix up start and end point of list */
	h = host_array[0];
	host_array[n_hosts - 1]->next = NULL;
	
	/* Dispose of host array */
	free(host_array);
	
	return(h);
}

peer_t *randomise_peers(peer_t *pl)
{
	peer_t		*p;
	peer_t		**peer_array;
	u_int32_t	n_peers;
	u_int32_t	c;

	/* Count number of peers in list */
	p = pl;
	n_peers = 0;
	while(p != NULL)
	{
		n_peers++;
		p = p->next;
	}

	/* Allocate array to randomise peers into */
	peer_array = util_zmalloc(sizeof(*peer_array) * n_peers);
	
	/* For each peer in list, place it in a random position in the array */
	p = pl;
	while(p != NULL)	
	{
		/* Find a random empty position in array */
		c = random() % n_peers;
		while (peer_array[c] != NULL)
		{
			c++;
			c %= n_peers;
		}
		
		/* Copy peer there */
		peer_array[c] = p;
		
		/* Move on to next peer */
		p = p->next;
	}

	/* Make linked list order follow array's */
	for(c = 1; c < n_peers; c++)
		peer_array[c - 1]->next = peer_array[c];

	/* Fix up start and end point of list */
	p = peer_array[0];
	peer_array[n_peers - 1]->next = NULL;
	
	/* Dispose of peer array */
	free(peer_array);
	
	return(p);
}

/* Comparison functions */

int host_compare_none(host_t *host1, host_t *host2)
{
	/* Always insert at head of list */
	return(1);
}

int host_compare_bytes(host_t *host1, host_t *host2)
{
	u_int64_t	host1_bytes;
	u_int64_t	host2_bytes;
	
	host1_bytes = host1->bytes_sent + host1->bytes_received;
	host2_bytes = host2->bytes_sent + host2->bytes_received;

	if (host1_bytes == host2_bytes)
		return(0);
	else if (host1_bytes > host2_bytes)
		return(1);
	else
		return(-1);
}

int host_compare_bytes_sent(host_t *host1, host_t *host2)
{
	if (host1->bytes_sent == host2->bytes_sent)
		return(0);
	else if (host1->bytes_sent > host2->bytes_sent)
		return(1);
	else
		return(-1);
}

int host_compare_bytes_received(host_t *host1, host_t *host2)
{
	if (host1->bytes_received == host2->bytes_received)
		return(0);
	else if (host1->bytes_received > host2->bytes_received)
		return(1);
	else
		return(-1);
}

int host_compare_packets(host_t *host1, host_t *host2)
{
	u_int64_t	host1_packets;
	u_int64_t	host2_packets;
	
	host1_packets = host1->packets_sent + host1->packets_received;
	host2_packets = host2->packets_sent + host2->packets_received;

	if (host1_packets == host2_packets)
		return(0);
	else if (host1_packets > host2_packets)
		return(1);
	else
		return(-1);
}

int host_compare_packets_sent(host_t *host1, host_t *host2)
{
	if (host1->packets_sent == host2->packets_sent)
		return(0);
	else if (host1->packets_sent > host2->packets_sent)
		return(1);
	else
		return(-1);
}

int host_compare_packets_received(host_t *host1, host_t *host2)
{
	if (host1->packets_received == host2->packets_received)
		return(0);
	else if (host1->packets_received > host2->packets_received)
		return(1);
	else
		return(-1);
}

int host_compare_connections(host_t *host1, host_t *host2)
{
	u_int64_t	host1_connections;
	u_int64_t	host2_connections;
	
	host1_connections = host1->connections_sent + host1->connections_received;
	host2_connections = host2->connections_sent + host2->connections_received;

	if (host1_connections == host2_connections)
		return(0);
	else if (host1_connections > host2_connections)
		return(1);
	else
		return(-1);
}

int host_compare_connections_sent(host_t *host1, host_t *host2)
{
	if (host1->connections_sent == host2->connections_sent)
		return(0);
	else if (host1->connections_sent > host2->connections_sent)
		return(1);
	else
		return(-1);
}

int host_compare_connections_received(host_t *host1, host_t *host2)
{
	if (host1->connections_received == host2->connections_received)
		return(0);
	else if (host1->connections_received > host2->connections_received)
		return(1);
	else
		return(-1);
}

int host_compare_peers(host_t *host1, host_t *host2)
{
	if (host1->n_peers == host2->n_peers)
		return(0);
	else if (host1->n_peers > host2->n_peers)
		return(1);
	else
		return(-1);
}

int host_compare_hostname(host_t *host1, host_t *host2)
{
	/* Use IP addresses if no hostnames found */
	if ((host1->hostname == NULL) && (host2->hostname == NULL))
		return(host_compare_ip_addr(host1, host2));
	
	/* Hosts without names come last */
	if (host1->hostname == NULL)
		return(-1);
	if (host2->hostname == NULL)
		return(1);

	/* If both hosts have names, compare them */		
	return(-g_strcasecmp(host1->hostname, host1->hostname));
}

int host_compare_ip_addr(host_t *host1, host_t *host2)
{
	if (host1->ip_addr == host2->ip_addr)
		return(0);
	else if (host1->ip_addr > host2->ip_addr)
		return(1);
	else
		return(-1);
}

int host_compare_first_seen(host_t *host1, host_t *host2)
{
	if (host1->first_seen == host2->first_seen)
		return(0);
	else if (host1->first_seen > host2->first_seen)
		return(1);
	else
		return(-1);
}

int host_compare_last_seen(host_t *host1, host_t *host2)
{
	if (host1->last_seen == host2->last_seen)
		return(0);
	else if (host1->last_seen > host2->last_seen)
		return(1);
	else
		return(-1);
}

int peer_compare_connections(peer_t *peer1, peer_t *peer2)
{
	if (peer1->connections_sent == peer2->connections_sent)
		return(0);
	else if (peer1->connections_sent > peer2->connections_sent)
		return(1);
	else
		return(-1);
}

int peer_compare_packets(peer_t *peer1, peer_t *peer2)
{
	if (peer1->packets_sent == peer2->packets_sent)
		return(0);
	else if (peer1->packets_sent > peer2->packets_sent)
		return(1);
	else
		return(-1);
}

int peer_compare_bytes(peer_t *peer1, peer_t *peer2)
{
	if (peer1->bytes_sent == peer2->bytes_sent)
		return(0);
	else if (peer1->bytes_sent > peer2->bytes_sent)
		return(1);
	else
		return(-1);
}

int peer_compare_none(peer_t *peer1, peer_t *peer2)
{
	/* Always insert at head of list */
	return(1);
}
