/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <math.h>
#include <stdio.h>
#include "rgbdraw.h"
#include "iconlist.h"

#define sqrt2 1.41421356237

void drawvline(GdkPixbuf *image, int x, int y1, int y2, int r, int g, int b)
{
	int i, w, h, rs;
	guchar *base, *pixels;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);
	pixels = gdk_pixbuf_get_pixels(image);

	if(x < 0 || x >= w) return;
	if(y1 >= h || y2 < 0) return;
	if(y1 < 0) y1 = 0;
	if(y2 >= h) y2 = h-1;

	for(i = y1; i <= y2; i++)
	{
		base = pixels + 3*x+i*rs;
                base[0] = r;
                base[1] = g;
                base[2] = b;
	}
}

void drawhline(GdkPixbuf *image, int x1, int y, int x2, int r, int g, int b)
{
	int i, w, h, rs;
	guchar *base, *pixels;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);
	pixels = gdk_pixbuf_get_pixels(image);

	if(y < 0 || y >= h) return;
	if(x1 >= w || x2 < 0) return;
	if(x1 < 0) x1 = 0;
	if(x2 >= w) x2 = w-1;

	for(i = x1; i <= x2; i++)
	{
		base = pixels + 3*i+y*rs;
                base[0] = r;
                base[1] = g;
                base[2] = b;
	}
}

void drawicon(GdkPixbuf *image, int x, int y, struct picture *icon)
{
	int w, h, rs;
	guchar *pixels;
        int x0, y0;
        int u, v;
        int r, s;
        unsigned char rr, gg, bb;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);
	pixels = gdk_pixbuf_get_pixels(image);

        for(v = 0; v < icon->h; v++)
        {
                y0 = y + v - icon->hoty;
                if(y0 < 0 || y0 >= h) continue;

                for(u = 0; u < icon->w; u++)
                {
                        x0 = x + u - icon->hotx;
                        if(x0 < 0 || x0 >= w) continue;
                        r = 3*x0+y0*rs;
                        s = 3*(v*icon->w+u);
                        rr = icon->data[s++];
                        gg = icon->data[s++];
                        bb = icon->data[s];
                        if(rr == 0 && gg == 0 && bb == 0) continue;
                        pixels[r++] = rr;
                        pixels[r++] = gg;
                        pixels[r] = bb;
                }
        }
}

void fillcircle(GdkPixbuf *image, int x, int y, float rad, 
	int r, int g, int b)
{
        int i, j;
        int dx, dy;
	int irad;
	float D, r2 = rad*rad;
	int w, h, rs;
	guchar *base, *pixels;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);
	pixels = gdk_pixbuf_get_pixels(image);

	irad = ceil(rad);

        for(j=y-irad; j <= y+irad; j++)
        {
                if(j < 0 || j >= h) continue;
        	for(i=x-irad; i <= x+irad; i++)
                {
                        if(i < 0 || i >= w) continue;
                        dx = x - i;
                        dy = y - j;
                        D = dx*dx+dy*dy;
			if(D < r2) D = 1.0;
			else D = 1.0+rad-sqrt(D);
                        if(D > 0.0)
                        {
				base = pixels + 3*i+rs*j;
				base[0] = (int)(D*(float)r);
				base[1] = (int)(D*(float)g);
				base[2] = (int)(D*(float)b);
                        }
                }
        }
}

void drawline_(GdkPixbuf *image, int x1, int y1, int x2, int y2, float thick,
	int r, int g, int b)
{
        float x0, y0, sx, sy, m, D;
	float rad;
        int dx, dy;
	int x, y, irad, min, max;
	int w, h, rs;
	guchar *base, *pixels;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);
	pixels = gdk_pixbuf_get_pixels(image);

        dx = x2 - x1;
        dy = y2 - y1;

        if(dx == 0 && dy == 0)
        {
                fillcircle(image, x1, y1, thick, r, g, b);
                return;
        }

        if(abs(dx) > abs(dy))
        {
                if(x2 < x1)
                {
                        x2 = x1;
                        x1 = dx + x1;
                        y2 = y1;
                        y1 = dy + y1;
                        dx = -dx;
                        dy = -dy;
                }
                m = (float)dy/(float)dx;
                rad = (0.5*thick+1.0)*sqrt(1.0+m*m);
		irad = rint(rad);
                sx = 1.0/sqrt(1.0+m*m);
                sy = m*sx;
                for(x = x1; x <= x2; x++)
                {
                        if(x < 0 || x >= w) continue;
                        y0 = (float)y1 + (float)(x - x1)*m;
			min = (int)(y0-rad-0.5);
			max = (int)(y0+rad+0.5);
                        for(y = min; y <= max; y++)
                        {
                                if(y < 0 || y >= h) continue;
                                D = fabs(sy*((float)x-x1) - sx*((float)y-y1));
                                D = 0.5 + 0.5*thick - D;
                                if(D > 1.0) D = 1.0;
                                if(D > 0.0)
                                {
					base = pixels + 3*x+rs*y;
					base[0] = (int)(D*(float)r);
					base[1] = (int)(D*(float)g);
					base[2] = (int)(D*(float)b);
                                }
                        }
                }
        }
        else
	{
                if(y2 < y1)
                {
                        x2 = x1;
                        x1 = dx + x1;
                        y2 = y1;
                        y1 = dy + y1;
                        dx = -dx;
                        dy = -dy;
                }
                m = (float)dx/(float)dy;
                rad = (0.5*thick+1.0)*sqrt(1.0+m*m);
		irad = rint(rad);
                sy = 1.0/sqrt(1.0+m*m);
                sx = m*sy;
                for(y = y1; y <= y2; y++)
                {
                        if(y < 0 || y >= h) continue;
                        x0 = (float)x1 + ((float)y - y1)*m;
			min = (int)(x0-rad-0.5);
			max = (int)(x0+rad+0.5);
                        for(x = min; x <= max; x++)
                        {
                                if(x < 0 || x >= w) continue;
                                D = fabs(sx*((float)y-y1) - sy*((float)x-x1));
                                D = 0.5 + 0.5*thick - D;
                                if(D > 1.0) D = 1.0;
                                if(D > 0.0)
                                {
					base = pixels + 3*x+rs*y;
					base[0] = (int)(D*(float)r);
					base[1] = (int)(D*(float)g);
					base[2] = (int)(D*(float)b);
                                }
                        }
                }
        }
}

void drawline(GdkPixbuf *image, int x1, int y1, int x2, int y2, float thick,
	int r, int g, int b)
{
	double x0, y0, dx, dy;
	double tx1, tx2, ty1, ty2;
	double tmax = 1.0, tmin = 0.0;
	int w, h, rs;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);

	x0 = x1;
	y0 = y1;
	dx = x2-x1;
	dy = y2-y1;

	if(x2-x1 != 0)
	{
		tx1 = -x0/dx;
		tx2 = ((double)w-1.0-x0)/dx;
		if(tx1 > tx2)
		{
			if(tx1 < tmax) tmax = tx1;
			if(tx2 > tmin) tmin = tx2;
		}
		else
		{
			if(tx2 < tmax) tmax = tx2;
			if(tx1 > tmin) tmin = tx1;
		}
	}
	else if(x1 < 0 || x1 >= w) return;

	if(y2-y1 != 0)
	{
		ty1 = -y0/dy;
		ty2 = ((double)h-1.0-y0)/dy;
		if(ty1 > ty2)
		{
			if(ty1 < tmax) tmax = ty1;
			if(ty2 > tmin) tmin = ty2;
		}
		else
		{
			if(ty2 < tmax) tmax = ty2;
			if(ty1 > tmin) tmin = ty1;
		}
	}
	else if(y1 < 0 || y1 >= h) return;

	if(tmin > tmax) return;

	x1 = x0+tmin*dx;
	x2 = x0+tmax*dx;
	y1 = y0+tmin*dy;
	y2 = y0+tmax*dy;
	drawline_(image, x1, y1, x2, y2, thick, r, g, b);
}

void drawcircle(GdkPixbuf *image, int x0, int y0, float rad, 
	int r, int g, int b)
{
	float v1, v2, x2, r2;
	int x, y, m, px, py;
	int w, h, rs;
	guchar *base, *pixels;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);
	pixels = gdk_pixbuf_get_pixels(image);

	m = rint(rad/sqrt2);

	x = rint(rad);
	r2 = rad*rad;

	for(y = 0; y <= m; y++)
	{
		x2 = x-1;
		v1 = fabs(x*x + y*y - r2);
		v2 = fabs(x2*x2 + y*y - r2);
		if(v2 < v1) x = x2;
	
		px = x0+x;
		if(px >= 0 && px < w)
		{
			py = y0+y;
			if(py >= 0 && py < h)
			{
				base = pixels + 3*px+rs*py;
				base[0] = r;
				base[1] = g;
				base[2] = b;
			}
			py = y0-y;
			if(py >= 0 && py < h)
			{
				base = pixels + 3*px+rs*py;
				base[0] = r;
				base[1] = g;
				base[2] = b;
			}
		}
		px = x0-x;
		if(px >= 0 && px < w)
		{
			py = y0+y;
			if(py >= 0 && py < h)
			{
				base = pixels + 3*px+rs*py;
				base[0] = r;
				base[1] = g;
				base[2] = b;
			}
			py = y0-y;
			if(py >= 0 && py < h)
			{
				base = pixels + 3*px+rs*py;
				base[0] = r;
				base[1] = g;
				base[2] = b;
			}
		}
		px = x0+y;
		if(px >= 0 && px < w)
		{
			py = y0+x;
			if(py >= 0 && py < h)
			{
				base = pixels + 3*px+rs*py;
				base[0] = r;
				base[1] = g;
				base[2] = b;
			}
			py = y0-x;
			if(py >= 0 && py < h)
			{
				base = pixels + 3*px+rs*py;
				base[0] = r;
				base[1] = g;
				base[2] = b;
			}
		}
		px = x0-y;
		if(px >= 0 && px < w)
		{
			py = y0+x;
			if(py >= 0 && py < h)
			{
				base = pixels + 3*px+rs*py;
				base[0] = r;
				base[1] = g;
				base[2] = b;
			}
			py = y0-x;
			if(py >= 0 && py < h)
			{
				base = pixels + 3*px+rs*py;
				base[0] = r;
				base[1] = g;
				base[2] = b;
			}
		}
	}
}

/* Take advantage of extra symmetry for speed, and avoid special cases in
 * the general code.
 */

void drawvertellipse(GdkPixbuf *image, int x0, int y0, float m, float n,
	int r, int g, int b)
{
	float A, B, R;
	float v1, v2, x2, y2;
	int x, y, lx, ly;
	int w, h, rs;
	guchar *base, *pixels;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);
	pixels = gdk_pixbuf_get_pixels(image);

	A = 1.0/(m*m);
	B = 1.0/(n*n);

	R = A/B;

	lx = rint(m/sqrt(1+(n*n)/(m*m)));
	ly = rint(n/sqrt(1+(m*m)/(n*n)));

	x = rint(m);
	for(y = 0; y <= ly; y++)
	{
		x2 = x-1;
		v1 = fabs(A*x*x + B*y*y - 1.0);
		v2 = fabs(A*x2*x2 + B*y*y - 1.0);
		if(v2 < v1) x = x2;
	
		base = pixels + 3*(x0+x)+rs*(y0+y);
		base[0] = r;
		base[1] = g;
		base[2] = b;
		base = pixels + 3*(x0-x)+rs*(y0-y);
		base[0] = r;
		base[1] = g;
		base[2] = b;
		if(y != 0)
		{
			base = pixels + 3*(x0+x)+rs*(y0-y);
			base[0] = r;
			base[1] = g;
			base[2] = b;
			base = pixels + 3*(x0-x)+rs*(y0+y);
			base[0] = r;
			base[1] = g;
			base[2] = b;
		}
	}

	y = rint(n);
	R = B/A;
	for(x = 0; x <= lx; x++)
	{
		y2 = y-1;
		v1 = fabs(A*x*x + B*y*y - 1.0);
		v2 = fabs(A*x*x + B*y2*y2 - 1.0);
		if(v2 < v1) y = y2;
	
		base = pixels + 3*(x0+x)+rs*(y0+y);
		base[0] = r;
		base[1] = g;
		base[2] = b;
		base = pixels + 3*(x0-x)+rs*(y0-y);
		base[0] = r;
		base[1] = g;
		base[2] = b;
		if(y != 0)
		{
			base = pixels + 3*(x0+x)+rs*(y0-y);
			base[0] = r;
			base[1] = g;
			base[2] = b;
			base = pixels + 3*(x0-x)+rs*(y0+y);
			base[0] = r;
			base[1] = g;
			base[2] = b;
		}
	}
}

/* Needs to be fixed up -- has a few crash cases and is not too pretty */

void drawellipse(GdkPixbuf *image, int x0, int y0, float maj, float min,
	float pa, int r, int g, int b)
{
	float A, B, C;
	float s, c;
	int x, y, x2, y2;
	float v1, v2, sx, sy;
	int dx, dy, px, py;
	int safe;
	int w, h, rs;
	guchar *base, *pixels;

	w = gdk_pixbuf_get_width(image);
	h = gdk_pixbuf_get_height(image);
	rs = gdk_pixbuf_get_rowstride(image);
	pixels = gdk_pixbuf_get_pixels(image);

	maj = maj*maj;
	min = min*min;

	s = sin(pa); c = cos(pa);

	A = c*c/min + s*s/maj;
	B = s*s/min + c*c/maj;
	C = 2.0*s*c*(1.0/min-1.0/maj);

	x = 0;
	y = rint(sqrt(1.0/B));
	dx = dy = 1;
	safe = 0;

	do
	{
		sx = 2.0*B*y + C*x;
		sy = -(2.0*A*x + C*y);

		if(fabs(sx) > fabs(sy))
		{
			if(dx*sx < 0.0)
			{
				sx = -sx;
				sy = -sy;
			}
			if(sy > 0) dy = 1; else dy = -1;

			x+=dx;
			y2 = y+dy;
			v1 = fabs(A*x*x + B*y*y + C*x*y - 1.0);
			v2 = fabs(A*x*x + B*y2*y2 + C*x*y2 - 1.0);
			if(v2 < v1) y = y2;
			safe=1;
		}
		else
		{
			if(dy*sy < 0.0)
			{
				sx = -sx;
				sy = -sy;
			}
			if(sx > 0) dx = 1; else dx = -1;

			y+=dy;
			x2 = x+dx;
			v1 = fabs(A*x*x + B*y*y + C*x*y - 1.0);
			v2 = fabs(A*x2*x2 + B*y*y + C*x2*y - 1.0);
			if(v2 < v1) x = x2;
		}

		px = x0+x;
		py = y0+y;
		if(px >= 0 && px < w && py >= 0 && py < h)
		{
			base = pixels + 3*px+rs*py;
			base[0] = r;
			base[1] = g;
			base[2] = b;
		}
		px = x0-x;
		py = y0-y;
		if(px >= 0 && px < w && py >= 0 && py < h)
		{
			base = pixels + 3*px+rs*py;
			base[0] = r;
			base[1] = g;
			base[2] = b;
		}
	} while(!(safe && x==0));

	x = 0;
	y = rint(sqrt(1.0/B));
	dx = dy = -1;
	safe = 0;

	do
	{
		sx = 2.0*B*y + C*x;
		sy = -(2.0*A*x + C*y);

		if(fabs(sx) > fabs(sy))
		{
			if(dx*sx < 0.0)
			{
				sx = -sx;
				sy = -sy;
			}
			if(sy > 0) dy = 1; else dy = -1;

			x+=dx;
			y2 = y+dy;
			v1 = fabs(A*x*x + B*y*y + C*x*y - 1.0);
			v2 = fabs(A*x*x + B*y2*y2 + C*x*y2 - 1.0);
			if(v2 < v1) y = y2;
			safe=1;
		}
		else
		{
			if(dy*sy < 0.0)
			{
				sx = -sx;
				sy = -sy;
			}
			if(sx > 0) dx = 1; else dx = -1;

			y+=dy;
			x2 = x+dx;
			v1 = fabs(A*x*x + B*y*y + C*x*y - 1.0);
			v2 = fabs(A*x2*x2 + B*y*y + C*x2*y - 1.0);
			if(v2 < v1) x = x2;
		}

		px = x0+x;
		py = y0+y;
		if(px >= 0 && px < w && py >= 0 && py < h)
		{
			base = pixels + 3*px+rs*py;
			base[0] = r;
			base[1] = g;
			base[2] = b;
		}
		px = x0-x;
		py = y0-y;
		if(px >= 0 && px < w && py >= 0 && py < h)
		{
			base = pixels + 3*px+rs*py;
			base[0] = r;
			base[1] = g;
			base[2] = b;
		}
	} while(!(safe && x==0));
}
