// Copyright (C)  2001 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/jar_zip_utils/JarFile.cpp,v 1.2 2001/11/10 05:36:02 gwu2 Exp $
//

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "JarFile.h"
#include "Jar_Entry.h"

/************************************************************************************/
/*Constructor*/
JarFile::JarFile()
{
	m_Is_JarFile_Info_OK = false;
	m_Entry_Info = NULL;
	m_Manifest = NULL;
	m_Have_Parse_Manifest = false;
}

JarFile::JarFile(const char* jarflname)
{
	m_Is_JarFile_Info_OK = false;
	m_Entry_Info = NULL;

	Open(jarflname);			//m_Is_JarFile_Info_OK will be true if successful.

	m_Manifest = NULL;
	m_Have_Parse_Manifest = false;
}

JarFile::JarFile(general_info gi,Properties* entry,Manifest* m,bool have_parse)
{
	m_Is_JarFile_Info_OK = true;
	m_JarFile_Info = gi;
	m_Entry_Info = entry;
	m_Manifest = m;
	m_Have_Parse_Manifest = have_parse;
}

/************************************************************************************/
JarFile::~JarFile()
{
	Close();
	if(m_Manifest)
		delete m_Manifest;
	if(m_Entry_Info)
		delete m_Entry_Info;
}
/************************************************************************************/
bool JarFile::Is_Open_Jar_OK()
{
	return m_Is_JarFile_Info_OK;
}

general_info* JarFile::Get_JarFile_Info()
{
	if(m_Is_JarFile_Info_OK)
		return &m_JarFile_Info;
	return NULL;
}

/************************************************************************************/
/************************************************************************************/
/*Locate the Central directory of a zip or jar file (at the end, just before the global
  comment). In one word, that is to look for the tag "504b0506" in the stream,and return
  the offset of this tag.

  It is better to begin from the end. Because this tag is ususally at the end of a zip 
  or jar file.
*/

/*Length of the buffer used for one reading when looking for the tag.*/
#define BUFLENGTH_READ_COMMENT (0x400)

uLong JarFile::Search_Central_Dir(FILE* fl)
{
	uLong max_need_read = 0xffff;			/* maximum size of global comment */
	uLong central_end_pos = 0;

	if(fseek(fl,0,SEEK_END)!=0)			/*seek to the end of the file,return 0 if successful*/
		return 0;						/*means error*/

	uLong filesize = ftell(fl);					/*get the size of the file*/

	/*now look for 504b0506,the end tag of central directory of a zip or jar file
	we begin from the end to the start*/

	if(filesize < max_need_read)
		max_need_read = filesize;

	unsigned char* buf = (unsigned char*)malloc(BUFLENGTH_READ_COMMENT);
	if(NULL == buf)
		return 0;

	uLong begin_pos = filesize;

	/*the length of the end tag,which means the first 4 bytes need not be red*/
	uLong have_red = 4;
	while(have_red < max_need_read)
	{
		uLong current_read_len = (have_red + BUFLENGTH_READ_COMMENT < max_need_read)?
								BUFLENGTH_READ_COMMENT : max_need_read-have_red;

		begin_pos -= current_read_len;
		have_red += current_read_len;

		if(fseek(fl,begin_pos,SEEK_SET)!=0)
			break;
		if(fread(buf,(uInt)current_read_len,1,fl)!=1)
			break;

		for (int i=(int)current_read_len-4;i>=0;i--)
			if (((*(buf+i))==0x50) && ((*(buf+i+1))==0x4b) && 
				((*(buf+i+2))==0x05) && ((*(buf+i+3))==0x06))
			{
				central_end_pos = begin_pos+i;
				break;
			}
		if (central_end_pos!=0)
			break;
	}
	free(buf);
	return central_end_pos;
}

/************************************************************************************/
/*Data after central directory's tag includes following infomations:
	2 bytes:number of the current dist, used for spaning ZIP, unsupported, always 0
	2 bytes:number the the disk with central dir, used for spaning ZIP, unsupported, always 0
	2 bytes:number of entries in the central directory on this disk.
	2 bytes:total number of entries in the central dir (same than number_entry on nospan)
	4 bytes:size of the central directory
	4 bytes:offest of the start of the central directory
	2 bytes:length of the comment;
	n bytes:the comment;
*/

int JarFile::Get_Info_After_Central_Dir(FILE* fl)
{
	uLong endtag;
	uLong number_disk;		       
	uLong number_disk_CD;
	uLong total_number_entry;

	int err=UNJAR_OK;

	/*The central directory's tag 0x06054b50, 
	 already checked in unjar_Search_Central_Dir*/
	if (unjar_get_long(fl,&endtag)!=UNJAR_OK)
		err=UNJAR_ERROR;

	/*Number of this disk*/
	if (unjar_get_short(fl,&number_disk)!=UNJAR_OK)
		err=UNJAR_ERROR;

	/*Number of the disk with the start of the central directory*/
	if (unjar_get_short(fl,&number_disk_CD)!=UNJAR_OK)
		err=UNJAR_ERROR;

	/*Total number of entries in the central directory on this disk*/
	if (unjar_get_short(fl,&(m_JarFile_Info.number_entry))!=UNJAR_OK)
		err=UNJAR_ERROR;

	/*Total number of entries in the central directory*/
	if (unjar_get_short(fl,&total_number_entry)!=UNJAR_OK)
		err=UNJAR_ERROR;

	/*Size of the central directory*/
	if (unjar_get_long(fl,&(m_JarFile_Info.size_central_dir))!=UNJAR_OK)
		err=UNJAR_ERROR;

	/*Offset of start of central directory with respect to the starting disk number */
	if (unjar_get_long(fl,&(m_JarFile_Info.offset_central_dir))!=UNJAR_OK)
		err=UNJAR_ERROR;

	/*The rest of the zip or jar file are the length of the comment and the comment.
	we do not need them here.
	*/

	/*number_dis_CD and number_disk are always 0.*/
	if ((total_number_entry!=m_JarFile_Info.number_entry) ||
		(number_disk_CD!=0) ||
		(number_disk!=0))
		err|=UNJAR_BADJARFILE;

	return err;
}

/************************************************************************************/
/*
Open a zip or jar file. Get the general infomation of the zip or jar file.

jarflname: full file name of the zip or jar file;

return: UNJAR_OK if there is no problem.
*/

int JarFile::Open(const char* jarflname)
{
	int err = UNJAR_OK;

	/*open the zip or jar file in mode read and binary,NULL if error*/
	FILE* fl = fopen(jarflname,"rb");
	if(NULL == fl)
		return UNJAR_ERROR;

	/*Search the positon of the central directroy' end tag(0x06054b50) in the zip or jar file*/
	uLong central_pos = Search_Central_Dir(fl);
	/*0 if error*/
	if(0 == central_pos)
		return UNJAR_ERROR;

	/*Seek to the end fo the central directory*/
	if(fseek(fl,central_pos,SEEK_SET)!=0)
		err = UNJAR_ERROR;
	/*Get some useful infomation:entry_number,size_central_dir,offset_central_dir*/
	err = Get_Info_After_Central_Dir(fl);

	if ((central_pos < m_JarFile_Info.offset_central_dir + m_JarFile_Info.size_central_dir) && 
		(UNJAR_OK == err))
		err |= UNJAR_BADJARFILE;

	if (err != UNJAR_OK)
	{
		fclose(fl);
		return UNJAR_ERROR;
	}

	m_JarFile_Info.file_handle = fl;
	m_JarFile_Info.filename = strdup(jarflname);
	m_JarFile_Info.byte_before_file =
		central_pos - (m_JarFile_Info.offset_central_dir + m_JarFile_Info.size_central_dir);
	m_JarFile_Info.pos_central_dir = central_pos;
	err = Get_Entry_Info();
	
	m_Is_JarFile_Info_OK = (UNJAR_OK == err);
//	m_Entry_Info.file_name=NULL;

	return err;
}


/************************************************************************************/
/************************************************************************************/
/*Close a zip or jar file opened with Open.

  return UNJAR_OK if there is no problem. 
*/

int JarFile::Close ()
{
	if(!m_Is_JarFile_Info_OK)			//The file is not be opened successfully.
		return UNJAR_OK;

	fclose(m_JarFile_Info.file_handle);
	if(m_JarFile_Info.filename)
		delete[] m_JarFile_Info.filename;

	m_Is_JarFile_Info_OK=false;			//An important variable.

	return UNJAR_OK;
}

/************************************************************************************/
/************************************************************************************/
int JarFile::Get_Entry_Info()
{
	int err = UNJAR_OK;
	
	// m_Entry_Info = new Properties();
	// wgs: I wonder if there're better empirical configuration of bucket number
	m_Entry_Info = new Properties(m_JarFile_Info.number_entry / 10 + 1);

	uLong pos_in_central_dir = m_JarFile_Info.offset_central_dir
								+ m_JarFile_Info.byte_before_file;
	uLong index_file = 0;
	while(UNJAR_OK == err)
	{
		uLong temp_data,length_filename,length_restdata,offset;
		/*get info*/
		if(fseek(m_JarFile_Info.file_handle,pos_in_central_dir,SEEK_SET)!=0)
			return UNJAR_ERROR;

		/*Get the tag 504b0102,which is the begin of the file info in the central directory*/
		if(unjar_get_long(m_JarFile_Info.file_handle,&temp_data) != UNJAR_OK)
			err = UNJAR_ERROR;
		else if(temp_data!=0x02014b50)
			err = UNJAR_ERROR;

		/*Look over the next 24 bytes : version(2 bytes), version needed(2 bytes), flags(2 bytes),
		  compression method(2 bytes), dos date(4 bytes), CRC(4 byte), compressed length(4 bytes),
		  uncompressed method(4 bytes)*/
		if(fseek(m_JarFile_Info.file_handle,24,SEEK_CUR)!=0)
			err = UNJAR_ERROR;

		/*Get the length of the file name of the current file*/
		if(unjar_get_short(m_JarFile_Info.file_handle,&length_filename)!=UNJAR_OK)
			err = UNJAR_ERROR;

		/*Get the size of the file extra*/
		if(unjar_get_short(m_JarFile_Info.file_handle,&length_restdata)!=UNJAR_OK)
			err = UNJAR_ERROR;

		/*Get the size of the file comment*/
		if(unjar_get_short(m_JarFile_Info.file_handle,&temp_data)!=UNJAR_OK)
			err = UNJAR_ERROR;
		if(UNJAR_OK == err)
			length_restdata += temp_data;

		/*Look over the next 8 bytes :	start disk number(2 bytes),
		internal_fa(2 bytes) , external_fa(4 bytes)*/
		if(fseek(m_JarFile_Info.file_handle,8,SEEK_CUR)!=0)
			err = UNJAR_ERROR;

		/*Get the offset of the current file in the zip or jar file*/
		if(unjar_get_long(m_JarFile_Info.file_handle,&offset)!=UNJAR_OK)
			err = UNJAR_ERROR;

		/*Get the file name of the current file*/
		char* filename;
		if((UNJAR_OK == err )&&(length_filename > 0))
		{
			filename = new char[length_filename +1];
			*(filename + length_filename) = '\0';

			if(fread(filename,(uInt)length_filename,1,m_JarFile_Info.file_handle)!=1)
				err = UNJAR_ERROR;

			uLong* pos = new uLong[1];
			*pos = pos_in_central_dir;
			Prop_uLong* pu = new Prop_uLong(pos);
			m_Entry_Info->add(filename,pu);
			delete[] filename;
			delete pu;
		}
		if(index_file+1 >=m_JarFile_Info.number_entry)
			break;
		pos_in_central_dir += SIZE_CENTRAL_DIR_ITEM + length_filename + length_restdata;
		index_file ++;
	}
	return err;
}

/************************************************************************************/
/************************************************************************************/
uLong JarFile::Get_Entry_Pos_CD(const char* entryflname)
{
	if((!m_Is_JarFile_Info_OK)||(NULL == m_Entry_Info))
		return -1;
	Prop_uLong* pu = (Prop_uLong*)m_Entry_Info->get(entryflname);

	if(NULL == pu)
		return -1;

	return *(pu->value);
}



/************************************************************************************/
/************************************************************************************/
/*Get the Manifest of this jar file.
  If successful,return a pointer to a Manifest,
  else return NULL.
*/
Manifest* JarFile::Get_Manifest()
{
	if(m_Have_Parse_Manifest)			//Hava parsed the manifest
		return m_Manifest;

	m_Have_Parse_Manifest = true;

	Jar_Entry manifest(this,"META-INF/MANIFEST.MF");

	char* buf = (char*)manifest.Read();
	if(NULL == buf)						//Meet Errors
		return NULL;

	m_Manifest=new Manifest(buf);
	if(!m_Manifest->parse())			//Parse the manifest
	{
		delete m_Manifest;
		m_Manifest = NULL;				//Meet Errors
	}

	delete[] buf;
	return m_Manifest;
}

/************************************************************************************/
/************************************************************************************/
JarFile* JarFile::clone()
{
	if(!m_Is_JarFile_Info_OK)
		return (new JarFile());

	general_info cloned_gi = m_JarFile_Info;
	FILE* fl = fopen(cloned_gi.filename,"rb");
	if(NULL == fl)
		return (new JarFile());

	cloned_gi.file_handle = fl;
	cloned_gi.filename = strdup(m_JarFile_Info.filename);

	if(NULL == m_Entry_Info)
	{
		if(NULL == m_Manifest)
			return (new JarFile(cloned_gi,NULL,NULL,m_Have_Parse_Manifest));
		return (new JarFile(cloned_gi,NULL,m_Manifest->clone(),m_Have_Parse_Manifest));
	}
	if(NULL == m_Manifest)
		return (new JarFile(cloned_gi,m_Entry_Info->clone(),NULL,m_Have_Parse_Manifest));
	return (new JarFile(cloned_gi,m_Entry_Info->clone(),m_Manifest->clone(),m_Have_Parse_Manifest));
}