// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/train_generation.cpp,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//


#include "platform.h"
#include "orp_cout.h"
#include "train_generation.h"
#include "train.h"
#include "card_table.h"
#include "nursery_step_gen.h"
#include "gc_plan.h"
#include "gc_hooks.h"
#include "gc_walk.h"

Train_Generation::Train_Generation(unsigned char generation_number,
                                   Gc_Fast_Hooks *p_gc_hooks,
                                   Gc_Plan       *p_gc_plan,
                                   Gc_Interface  *p_container,
                                   Generation    *p_superior,
                                   Card_Table    *p_card_table,
                                   Block_Store   *p_bs)
                : Generation(generation_number,
                             p_gc_hooks,
                             p_gc_plan,
                             p_container,
                             p_superior,
                             p_card_table,
                             p_bs)
{

#if (GC_DEBUG>2)
    assert(_p_block_store);
#endif // _DEBUG

    //
    // Trains are created lazily when needed.
    //
    _number_of_trains = 0;

#if (GC_DEBUG>0)
    train_generation_creation_hook(this);
#endif
}

Train_Generation::~Train_Generation() 
{
#if (GC_DEBUG>0)
    train_generation_deletion_hook(this);
#endif

    for (unsigned long idx = 0; idx < _number_of_trains; idx++) {
        delete _p_train[idx];
    }
}

void
Train_Generation::run_all_finalizers() 
{
    for (unsigned long idx = 0; idx < _number_of_trains; idx++) {
        _p_train[idx]->run_all_finalizers();
    }
}

//
// We are called when the reference starts and ends in our (the MOS) generation.
// We check to see if it straddles cars, and update appropriate car remembered
// sets. This is a virtual function. We have an
// alternative fast non-virtual inline function for exclusive use
// by train generations, called update_car_write_barriers. If that
// works, then retire this one.
//
// We do not check to see if we have a reference into the focus train since
// the cars will be checked before we release an entire train.
//

//
// RLH-TRAIN This is correct.
//
void
Train_Generation::add_entry_to_generation_write_barriers(Java_java_lang_Object **pp_obj_ref,
                                                         Java_java_lang_Object *p_obj)
{
#if (GC_DEBUG>2)
    assert(is_reference_in_my_generation(pp_obj_ref));
    assert(is_object_in_my_generation(p_obj));
#endif // _DEBUG

    Train *p_source_train = p_global_bs->p_get_reference_train(pp_obj_ref);
    Train *p_target_train = p_global_bs->p_get_object_train(p_obj);

    if (p_source_train->get_train_id() > p_target_train->get_train_id()) {
        //
        // Record pointers from younger to older trains.
        // (i.e. higher numbered to lower numbered cars. Think birth date.)
        //
        Car *p_car = p_global_bs->p_get_object_car(p_obj);
        p_car->update_container_write_barrier(pp_obj_ref);

    } else if (p_source_train->get_train_id() == p_target_train->get_train_id()) {

        Car *p_source_car = p_global_bs->p_get_reference_car(pp_obj_ref);
        Car *p_target_car = p_global_bs->p_get_object_car(p_obj);
        //
        // Only record pointers from younger to older cars in same train.
        // (i.e. higher numbered to lower numbered cars.)
        //
        if (p_source_car->get_car_id() > p_target_car->get_car_id()) {
            p_target_car->update_container_write_barrier(pp_obj_ref);
        }
    } 
    gc_trace (p_obj, "This is a target noted by the write barrier."); 
}

void
Train_Generation::_add_new_train()
{
    _p_train[_number_of_trains] = new Train(_number_of_trains,
                                            _p_gc_hooks,
                                            _p_gc_plan,
                                            this,
                                            _p_card_table,
                                            _p_block_store);
    
#ifdef GC_TRAIN_TRACE
    orp_cout << " Adding a new train " << _number_of_trains << endl;
#endif

    _number_of_trains++;
}

void
Train_Generation::_execute_policy (Java_java_lang_Object **pp_child_ref, bool doing_mos_collection)
{
    // First get object to final location.

    if (is_object_in_my_generation(*pp_child_ref)) {
        //
        // This is a pointer into an object in a
        // car in mature space.
        //
        if (is_object_in_focus_car(*pp_child_ref)) {
            //
            // This is a pointer to an object in the
            // focus car that is being moved.
            //
            if (is_object_forwarded(*pp_child_ref)) {
                assert (doing_mos_collection);
                Java_java_lang_Object *check = *pp_child_ref; // keep around for debugging...
                update_reference_forwarded(pp_child_ref);
                assert (!(is_object_in_focus_car (*pp_child_ref)));
            } else {
                // This will move the object to the train of the object pointed to.
                // The object will be moved into either the car containing
                // the pointer of into an older car. In either case no
                // remembered set information needs to be maintained for the
                // simple train algorithm. If we ever move to something like
                // the Darko algorithm or the PMOS/DMOS then we will need to 
                // remember this pointer.
                if (doing_mos_collection) {
                    *pp_child_ref = move_object_to_referencing_train(pp_child_ref);
                }
            } // end if (is_object_forwarded(*pp_child_ref))
        }
        // Update remset.
        Car *p_target_car   = 
            p_global_bs->p_get_object_car(*pp_child_ref);                    
        p_target_car->check_update_car_write_barrier(pp_child_ref);
    } else { 
        // object is in young generation
        //
        // Need to update younger generation's remembered set
        // to reflect the move of this object. (Note that the
        // old, stale pointer in the write barrier will get
        // flushed out when the car gets deleted.)
        //
        if (doing_mos_collection) {
            // We are doing a mos collection so we can only see live unforwarded objects in
            // YOS
            _p_young_generation->add_entry_to_generation_write_barriers(pp_child_ref,
                                                                        *pp_child_ref);
            assert(!(is_object_forwarded(*pp_child_ref)));
        } else {
            // We are doing a YOS collection so we might see a forwarded reference or
            // an object that needs to be evicted.
            if (is_object_forwarded(*pp_child_ref)) {
                Java_java_lang_Object *check = *pp_child_ref; // keep around for debugging...
                update_reference_forwarded(pp_child_ref);
                if (!(p_global_bs->is_object_in_large_object_space(*pp_child_ref))) {
                    assert(!(is_object_forwarded(*pp_child_ref)));
                }
                //
                // Object is already copied.
                // Notify the target car that it may need to update
                // its write barrier due to this additional reference.
                //
            } else {
                *pp_child_ref = p_evict_object (pp_child_ref, doing_mos_collection);
            }
            // Object is now in it's final location, deal with remsets.
            if (is_object_in_my_generation (*pp_child_ref)) {
                //object was moved into MOS. Update remsets as needed.
                Car   *p_target_car   = 
                    p_global_bs->p_get_object_car(*pp_child_ref);
                p_target_car->check_update_car_write_barrier(pp_child_ref);
            } else {
                // object is still in YOS, so update remset.
                _p_young_generation->add_entry_to_generation_write_barriers(pp_child_ref,
                                                                            *pp_child_ref);
        
            }
        } // end if (doing_mos_collection) {
    } // end if (is_object_in_my_generation(*pp_child_ref)) {
}

//
// Every generation needs to define a policy for handling objects
// found during cheney scans. Train generations evacuate all followers
// into the train they are referenced from.
// The container (Gc_Space or Block_List) calls the policy routine of
// the generation of the FROM container. In this case, the Train Generation.
//
// Sapphire is only an issue in the train generation and the policy is carried out
// by the routine sapphire_cheney_scan_execute_policy in the sapphire code.  
//
void 
Train_Generation::cheney_scan_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection)
{
#ifdef GC_SAPPHIRE
extern void sapphire_cheney_scan_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection);
    sapphire_cheney_scan_execute_policy(p_obj, doing_mos_collection);
    return;
#endif // GC_SAPPHIRE
 
    gc_trace (p_obj, "train_generation ln 170 is scanning this object.");

    // This object had better not be in a nursery, it could be in LOS however.
    assert (!(p_global_bs->is_object_in_nursery(p_obj)));

    // If this is an array call cheney_scan_array_execute_policy and return.
 
    Partial_Reveal_Class *p_class = (*((Partial_Reveal_VTable **)p_obj))->clss;
    // delete    Partial_Reveal_Class *p_class = get_object_class(p_obj);
 
    bool array_p = is_array(p_class);
    
    if (array_p) {
        cheney_scan_array_execute_policy (p_obj, doing_mos_collection);
        return;
    }

    //
    // Cycle through all the descendents.
    //

    unsigned int *offset_scanner = init_object_scanner (p_obj);

    Java_java_lang_Object **pp_child_ref;

    while ((pp_child_ref = p_get_ref(offset_scanner, p_obj)) != NULL) { 
        gc_trace (*pp_child_ref, 
                  "Found pointer to this object during train_generation.cpp ln. 193 cheney scan."); 
        // Move the scanner to the next reference.
        offset_scanner = p_next_ref (offset_scanner);

        if (*pp_child_ref == NULL) {
            continue;
        }

#if (GC_DEBUG>0)
        root_reference_hook(pp_child_ref);
#endif

        //_cheney_scan_reference_hook(pp_child_ref);
#if (GC_DEBUG>2)
        assert(*pp_child_ref != 0);
#endif
        _execute_policy (pp_child_ref, doing_mos_collection);

    }  // end while ((pp_child_ref = p_get_ref(offset_scanner, p_obj)) != NULL)
    // DEBUG
/// delete p_descendents;
    // end DEBUG
}

// This routine does arrays....
 

void 
Train_Generation::cheney_scan_array_execute_policy(Java_java_lang_Object *p_object, bool doing_mos_collection)
{

    Partial_Reveal_Class *p_class = (*((Partial_Reveal_VTable **)p_object))->clss;
    // delete    Partial_Reveal_Class *p_class = get_object_class(p_object);

    // If array is an array of primitives, then there are no references, so return.
    if (is_array_of_primitives(p_class))
        return;

    // Initialize the array scanner which will scan the array from the
    // top to the bottom. IE from the last element to the first element.

    unsigned int offset = init_array_scanner (p_object);
    
    //
    // Cycle through all the descendents.
    //
    
    Java_java_lang_Object **pp_child_ref;
    while ((pp_child_ref = p_get_array_ref(p_object, offset)) != NULL) {

        offset = next_array_ref (offset);
        
#if (GC_DEBUG>0)
        root_reference_hook(pp_child_ref);
#endif
        // Skip null entries.
        if (*pp_child_ref == NULL) {
            continue;
        }

        _execute_policy (pp_child_ref, doing_mos_collection);

    } // end while
}  //

#ifndef PER_CAR_BARRIERS
//
// A contained car is either being destroyed.
// Need to purge any entries that this car has in the
// young generation's remembered set. 
//
//
void 
Train_Generation::flush_rs_entries_of_car(Car *p_car)
{
    //
    // We pass this request up to our parent GC Interface.
    //
    _p_young_generation->selective_flush_write_barrier(p_car);
}
#endif

//
// The entire train is garbage - free it.
//
void
Train_Generation::_free_oldest_train()
{
#if (GC_DEBUG>2)
    assert(_number_of_trains > 1);
#endif // _DEBUG

#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - freeing oldest train. " << endl;
#endif
    //
    // Delete the oldest train.
    //
//    orp_cout << "NOT Deleting oldest train" << endl;
    delete _p_train[0];
    _number_of_trains--;

    //
    // Shift all the others one slot to the left.
    //
    for (unsigned long train_idx = 0; train_idx < _number_of_trains; train_idx++) {
        _p_train[train_idx] = _p_train[train_idx + 1];
        //
        // Set the new ID of this train.
        //
        _p_train[train_idx]->set_id(train_idx);
    }
    //
    // Add a new train, so that we maintain a constant 
    // number of trains.
    //
    _add_new_train();

    // It is reasonable to have trains without cars.
}


Train *
Train_Generation::p_get_youngest_train()
{
    if (_number_of_trains == 0) {
        // We don't have any trains yet so create some and return the youngest.
        //
        // Lazily create some trains, try 2 for now since that is the minimum required by
        // the train algorithm.  
        //
    
        unsigned trains_to_create = 2;
        for (unsigned idx = 0; idx < trains_to_create; idx++) {
            _add_new_train();
        }       
    }

    return _p_train[_number_of_trains - 1];
}

//
// Return the oldest train in my generation.
//
Train *
Train_Generation::p_nominate_focus_train()
{
#ifdef GC_SAPPHIRE
    if (_number_of_trains == 0) {
        // We don't have any trains yet so create some and return the youngest.
        //
        // Lazily create some trains, try 2 for now since that is the minimum required by
        // the train algorithm.  
        //
        unsigned trains_to_create = 2;
        for (unsigned idx = 0; idx < trains_to_create; idx++) {
            _add_new_train();   
            // To get things started add an empty car to the oldest train so we have something to
            // move the nursery blocks into.
            _p_train[idx]->_add_car();
        }
    }
    assert (_number_of_trains > 0);
#endif // GC_SAPPHIRE

    if (_number_of_trains > 0) {
        return _p_train[0];
    }

    return NULL;
}


bool 
Train_Generation::is_focus_car(Car *p_car) 
{
    Car *p_focus_car = ((Mrl_Gc_V1 *)p_gc)->p_get_focus_car();
    if (p_focus_car == p_car) {
        return true;
    }
    return false;
}
//
// Return the focus car.
//
Car *
Train_Generation::p_nominate_focus_car()
{
#ifdef GC_SAPPHIRE
    // Get the oldest train, creating one if need be.
    Train *oldest_train = p_nominate_focus_train();
    Car *oldest_car;
    assert (oldest_train != NULL);
    oldest_car = oldest_train->p_nominate_focus_car();
    if (oldest_car == NULL) {
        oldest_train->_add_car ();
        oldest_car = oldest_train->p_nominate_focus_car();
        assert (oldest_car);
    }
    return oldest_car;
#else
    Train *oldest_train = p_nominate_focus_train();
    if (oldest_train == NULL) {
        return NULL;
    }
    return oldest_train->p_nominate_focus_car();
#endif // GC_SAPPHIRE
}

bool executing_mature_generation_collection;


void 
Train_Generation::set_executing_mature_generation_collection (bool value)
{
    executing_mature_generation_collection = value;   
}


bool 
collecting_mature_generation()
{
//    p_gc->get_focus_car()
    orp_cout << "executing_mature_generation_collection is " ;
    orp_cout << (executing_mature_generation_collection == true) << endl;
    return (executing_mature_generation_collection == true);
}
//
// This routine implements the plan for placing imported objects 
// into optimal trains. 
//
Java_java_lang_Object *
Train_Generation::_heuristic_place_yos_object(Java_java_lang_Object **pp_obj_ref)
{
#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
    assert (_number_of_trains == 2);
#endif // GC_GEN_V3
#endif // GC_TRAIN_V5
    
    Java_java_lang_Object *result = NULL; 

    // ******* SHOULDN"T THIS BE OUTSIDE TRAIN????
    if (p_global_bs->is_reference_outside_heap(pp_obj_ref)) {
        //
        // If it is a reference from a non-gc'd location, put it
        // in the youngest train.
        //
        if (_p_train[_number_of_trains - 1]->get_number_of_cars() > 5) {
#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
            assert (0); // At most we should have only 1 car in each of two trains.
#endif
#endif // GC_TRAIN_V5
            // **************************************************************
            // Opportunity for tuning here.
            // **************************************************************
            // Limit the cars in trains to five (once again, because
            // 5 is Rick's lucky number :-)
            //
            _add_new_train();
        }
#ifdef GC_TRAIN_V5
       // RLH-TRAIN we promote into the youngest train now for the train.
       // Add the object to the youngest train since the ref is from
       // outside the heap.
       //
       result = _p_train[_number_of_trains - 1]->
                  p_scavenge_object(pp_obj_ref);
#else

#ifdef GC_GEN_V3
        //
        // For a simple generational collector we just add the
        // object to the OLDEST train since the only time we
        // have any objects in a younger train is when we collect
        // the train generation and at the conclusion of the collection
        // only a single, the oldest train exists.
        // -- PERFORMANCE NOTE --
        // When we do collect the train this might result
        // in an extra copy of objects tenured during that collection
        // but that is a minority of objects since we intend to only collect
        // this area once in a great while and eventually a full train will
        // be used with better performance.
        //
        // This problem of multiple copies is resolved by checking to see if 
        // we have a focus car that is also being collected. We do not want to
        // move any objects into a focus car. Instead we move the object into the
        // YOUNGEST tr
        //
        // The oldest train had better be at index 0.
        //
        if (_p_train[0]->is_my_object (*pp_obj_ref)){
            // If the object is in this train move it to the other train.
            result = _p_train[1]->
                      p_scavenge_object(pp_obj_ref);
//        } else if (collecting_mature_generation()) {
//            assert (0); // How can I be collecting MOS and importing an YOS object????
            // If we have a focus car we are doing a collection so we want to move this object into
            // the youngest train. This could create refs to the focus car but they should
            // be in the remembered set and not a problem except maybe for some asserts in mrl_v1
//            result = _p_train[1]->p_scavenge_object(pp_obj_ref);
        } else {
            // Otherwise it is being tenured when we aren't doing a YOS collection
            // so move it to the oldest (only) train.
            assert (_p_train[1]->get_number_of_cars() == 0);
            result =  _p_train[0]->p_scavenge_object(pp_obj_ref);
        }
#endif  // GC_GEN_V3

#endif // GC_TRAIN_V5
 
    } else {
#ifdef GC_TRAIN_V5
        
        // The reference is from inside the heap. It can be from two possible
        // locations of interest. If it is from YOS then we scavenge the object
        // into the youngest train.
        if (is_address_in_my_generation (pp_obj_ref)) {  // The reference is in
                                                         // the train generation.
            // Scavenge the object into the car holding the reference.
            // If the car holding the reference is the focus car figure out 
            // how we got here and short circuit the processing either here
            // or at some higher level. 
            result = move_object_to_referencing_train(pp_obj_ref);
        } 
        else { // The reference is not in the train generation so we move the
            // object into the youngest train.
            result = _p_train[_number_of_trains - 1]->
                        p_scavenge_object(pp_obj_ref); 
        }
#else
#ifdef GC_GEN_V3
        // See comment about GC_GEN_V3 above..
        result = _p_train[0]->
                  p_scavenge_object(pp_obj_ref);

#endif  // GC_GEN_V3
#endif // GC_TRAIN_V5

    }
    assert (result != NULL);
    return result;
}

Car * 
Train_Generation::slot_to_car (Java_java_lang_Object **pp_obj_ref)
{
  
    Gc_Space *p_space = 
        p_global_bs->p_get_address_container((void *)pp_obj_ref);
    assert(p_space->is_car());
    assert(((Car *)p_space)->p_get_train());
    return (Car *)p_space;
}

// Given a pointer to a slot in a train return the train.
Train *
Train_Generation::slot_to_train (Java_java_lang_Object **pp_obj_ref)
{
  
    Gc_Space *p_space = 
        p_global_bs->p_get_address_container((void *)pp_obj_ref);
    assert(p_space->is_car());
    assert(((Car *)p_space)->p_get_train());
    return ((Car *)p_space)->p_get_train();
}
// We have an object in a focus car that we need to move.
// If it is referenced from outside the train generation we move it
// to the youngest train.

// If the slot and object are in the focus car we assert and
// figure out why we are tracking pointers that don't leave the
// car.

// If the slot is in another train we move the object to that train.

// If the slot is in another car in the same train we *** RETURN NULL *** 
// Such objects will be moved using p_move_train_object.

Java_java_lang_Object *
Train_Generation::p_move_train_object(Java_java_lang_Object **pp_obj)
{
#ifndef GC_GEN_V3
    assert (0);
#endif

#if (GC_DEBUG>0)
    // Make sure it is in the train generation.
    assert (is_object_in_my_generation (*pp_obj));
    // Make sure slot is not in focus car
    assert(!((Mrl_Gc_V1 *)p_gc)->p_get_focus_car()->is_address_in_this_car (pp_obj));
    // Make sure object is in the focus car.
    // This might not be the case if we are freeing up an entire train and
    // we have encountered a soft or phantom reference that we need to evict.
    //    assert (p_get_focus_car()->is_address_in_this_car (*pp_obj));
    // Make sure we have somewhere to move the object.
    assert (_number_of_trains > 1); 
#endif

    if ((p_global_bs->is_reference_outside_heap(pp_obj)) ||
        (!is_address_in_my_generation(pp_obj))) {
        //
        // Slot is outside of train generation.
        // The algorithm allows us to put it into any train. The oldest train
        // being collected seems like a bad idea due to progress concerns. Putting 
        // them in train 1 will cause these objects to be collected sooner, 
        // build and measure to see if this is worthwhile.
        //
        // TUNING OPPORTUNITY - where do we put objects pointed to from
        // outside the MOS.
        //
        return _p_train[_number_of_trains - 1]->
                  p_scavenge_object(pp_obj);
    }
    // 
    // In a non-train generational algorithm we can only be doing a cheney 
    // scan and the ref must be from 
    // the oldest train so assert that and scavenge the object. The only
    // other time we can be in this routine is when we are dealing with
    // refs from outside the train. Remember that in GC_GEN_V3 we only have
    // one active car except for when we do an MOS collection at which point
    // we have a 'To' car and during the cheney scan of that car is the only
    // time we should end up here.
    // 

    // For TRAIN_V5 - We have a slot in MOS and an object in the focus car.
    assert(is_address_in_my_generation (pp_obj));
    Train *train_with_slot = slot_to_train (pp_obj);
    if (((Mrl_Gc_V1 *)p_gc)->p_get_focus_train() != train_with_slot) {
        // Slot is not in focus train.
        return train_with_slot->p_scavenge_object(pp_obj);
    }

    // Slot is in focus train but not in focus car.
    // This means there had better be another car in this train.
    assert (train_with_slot->get_number_of_cars() > 1);
//    orp_cout << "Not Moving object to end of train before completing out of train moves."
//        << endl;
//    return train_with_slot->p_scavenge_object(pp_obj);

//    assert (0);
    return NULL;
}

// We have an object in a focus car that we need to move.
// If it is referenced from outside the focus train generation it should have
// been moved by p_move_train_object

// If the slot and object are in the focus car we assert and
// figure out why we are tracking pointers that don't leave the
// car.

// If the slot is in another car in the same train we move
// it to the end of the train.

Java_java_lang_Object *
Train_Generation::p_move_car_object(Java_java_lang_Object **pp_obj)
{
#ifndef GC_GEN_V3
    assert (0);
#endif

#if (GC_DEBUG>0)
    // Make sure it is in the train generation.
    assert (is_object_in_my_generation (*pp_obj));
    // Make sure slot is not in focus car
    assert(!((Mrl_Gc_V1 *)p_gc)->p_get_focus_car()->is_address_in_this_car (pp_obj));
    // Make sure object is in the focus car.
    assert (((Mrl_Gc_V1 *)p_gc)->p_get_focus_car()->is_address_in_this_car (*pp_obj));
    // Make sure we have somewhere to move the object.
    assert (_number_of_trains > 1); 
    // Make sure the slot is in this train
    assert (is_address_in_my_generation(pp_obj));
    assert (((Mrl_Gc_V1 *)p_gc)->p_get_focus_train() == slot_to_train (pp_obj));
    // Slot is in focus train but not in focus car.
    // This means there had better be another car in this train.
    assert (((Mrl_Gc_V1 *)p_gc)->p_get_focus_train()->get_number_of_cars() > 1);
#endif
    // Now that we have checked all the seatbelts, do the deed.
    return ((Mrl_Gc_V1 *)p_gc)->p_get_focus_train()->p_scavenge_object(pp_obj);
}
//
// A lower generation is tenuring an object or some object in a 
// car is referenced from outside this generation. Put the object in
// any train except the youngest. Perhaps create a new
// train. Currently an attempt by a car to move an object
// into a different train also uses this routine.
//

Java_java_lang_Object *
Train_Generation::p_import_object_from_yos(Java_java_lang_Object **pp_obj) 
{
    if (_number_of_trains == 0) {
        unsigned trains_to_create = 5;    // just a good number.

#ifdef GC_GEN_V3
        trains_to_create = 2;       // When we do a collection we need 
                                    // 2 trains for GC_GEN_V3.
                                    // At the end of the collection the oldest 
                                    // is reclaimed. This routine will only be
                                    // called during a collection.
#endif
    
        //
        // Lazily create some trains.  
        //

            for (unsigned idx = 0; idx < trains_to_create; idx++) {
                _add_new_train();
            }
#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
    assert (_number_of_trains == trains_to_create);
#endif
#endif // GC_TRAIN_V5
    }
    //
    // Use a heuristic to put the object in an optimal train.
    //

    return _heuristic_place_yos_object(pp_obj);

}

#if (GC_DEBUG>3)
//
// Routine to display the contents of the generation.
//
void Train_Generation::inspect(unsigned int level)
{
    cout << "Inspecting Train Generation" << endl;
    cout << "    there are " << _number_of_trains << " trains" << endl;
    for (unsigned long idx = 0; idx < _number_of_trains; idx++) {
        cout << "Inspecting train # " << idx << endl;
        _p_train[idx]->inspect(level);
    }
}
#endif // _DEBUG


//
// A contained car is asking us to move an object it contains to a
// specific different train. This is because that object is referenced
// from that particular train.
//
Java_java_lang_Object *
Train_Generation::move_object_to_referencing_train(Java_java_lang_Object **pp_ref)
{
    Train *p_train = p_global_bs->p_get_reference_train(pp_ref);

    return p_train->p_scavenge_object(pp_ref);
}

void 
Train_Generation::free_empty_oldest_train()
{
    _free_oldest_train();
}
//
// A minor collection has just been completed.
// The Mrl_Gc_Vxx has passed us a remembered set
// that represents all the incoming pointers (live
// references from the ORP, and the young-to-old refs
// that were discovered during the minor
// collection.) We now do an incremental unit
// of collection.
//
Remembered_Set *
Train_Generation::p_reclaim_generation(Remembered_Set *all_refs_to_focus_car,
                                       Remembered_Set *p_weak_references,
                                       bool train_is_alive)

{
#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - reclaiming MOS generation train is alive is " 
        << train_is_alive << endl;
#endif

#if (GC_DEBUG>0)
    incremental_collection_start_hook(this);
#endif

#if (GC_DEBUG>0)
    verify_trains_cars_write_barriers();
#endif // _DEBUG
    //
    // Empty the remembered set which will store all the pending
    // scans. (i.e. the cars who have scan pointers that lag
    // free pointers.)
    //
#if (GC_DEBUG>2)
    assert(_p_pending_scans->is_empty());
#endif // _DEBUG

    assert ((_number_of_trains == 0) || (_number_of_trains > 1)); 

    //
    // Only collect if we have at least two trains.
    //
    if (_number_of_trains > 1 ) {
        //
        // If the minor reclamation determined that there were no
        // pointers incoming into this train, see if we can reclaim it now.
        //
        if ((train_is_alive == false) &&
            (_p_train[0]->no_incoming_refs())) {
            //
            // OK, there are no incoming pointers into the oldest train.
            //
            _free_oldest_train();
        } else {

//            orp_cout << " *** Trains before reclaim_train, train 1 should be empty. *****" << endl;
//            inspect(0);
//            orp_cout << " *** End of trains before reclaim train *****" << endl;

#if (GC_DEBUG>0)
            // The refs can be NULL if they are from an younger car or younger train
            // since they could have been overwritten with NULL after being placed
            // into the remembered set of the focus car.
            // Make sure all the p_incoming_refs are valid.
            all_refs_to_focus_car->rewind();
            Java_java_lang_Object **pp_obj_ref_quick_check;
            while (pp_obj_ref_quick_check = all_refs_to_focus_car->next()) {
                if (*pp_obj_ref_quick_check != NULL) {  
                    verify_is_object (*pp_obj_ref_quick_check, all_refs_to_focus_car);
                }
            }
            all_refs_to_focus_car->rewind();
#endif
            //
            // Do one incremental unit of reclamation on the oldest train.
            //
#ifdef GC_TRAIN_TRACE
            orp_cout << " Reclaiming train 0. " << endl;
#endif
            if ((_p_train[0]->reclaim_train(all_refs_to_focus_car, p_weak_references)) && 
                (train_is_alive == false)) {
                //
                // A return value of true from reclaim_train means that
                // the entire train is not reachable from other trains but 
                // we still need train_is_alive to be false since other 
                // cars in the train could have pointers from stacks into them.
                //
#ifdef GC_TRAIN_TRACE
                orp_cout << "@@@@@@@ Freeing a train and train_is_alive is false. " << endl;
#endif
                _free_oldest_train();
#if (GC_DEBUG>2)
                assert(_p_pending_scans->is_empty());
#endif // _DEBUG
            }
#ifdef GC_TRAIN_TRACE
            orp_cout << " Done reclaiming train 0. " << endl;
#endif // GC_TRAIN_TRACE
            // If there are no cars left in a train we can free it up also.
            if (_p_train[0]->get_number_of_cars() == 0) {
                _free_oldest_train();
            }
        }
    } else {
        // Why do we have only 1 train???
        assert (0);
    }

#if (GC_DEBUG>2)
    assert(_p_pending_scans->is_empty());
#endif // _DEBUG

#if (GC_DEBUG>0)
    verify_trains_cars_write_barriers();
#endif // _DEBUG

#if (GC_DEBUG>0)
    incremental_collection_end_hook(this);
#endif

    return NULL;
}

#if (GC_DEBUG>3)
//
// At debug time we do a painfully slow scan of all of mature
// space to determine if there are any mature to young pointers
// that didn't get recorded by the write barriers.
//
void 
Train_Generation::verify_all_mature_to_young_references_are_recorded()
{
    for (unsigned long idx = 0; idx < _number_of_trains; idx++) {
        _p_train[idx]->walk_train(verify_old_to_young_reference_is_in_rs);
    }
}
#endif
//
// This is a fast version (train-specific, non-virtual) version
// of add_entry_to_generation_write_barriers. Retire the other
// if this works well.
//
void Train_Generation::update_car_write_barriers(Java_java_lang_Object **pp_obj_ref,
                                                 Java_java_lang_Object *p_obj) 
{
#if (GC_DEBUG>2)
        assert(is_reference_in_my_generation(pp_obj_ref));
        assert(is_object_in_my_generation(p_obj));
#endif // _DEBUG

        Train *p_source_train = p_global_bs->p_get_reference_train(pp_obj_ref);
        Train *p_target_train = p_global_bs->p_get_object_train(p_obj);

        if (p_source_train->get_train_id() > p_target_train->get_train_id()) {
            //
            // Record pointers from younger to older trains.
            // (i.e. higher numbered to lower numbered cars.)
            //
            Car *p_car = p_global_bs->p_get_object_car(p_obj);
            p_car->update_container_write_barrier(pp_obj_ref);

        } else if (p_source_train->get_train_id() == p_target_train->get_train_id()) {

            Car *p_source_car = p_global_bs->p_get_reference_car(pp_obj_ref);
            Car *p_target_car = p_global_bs->p_get_object_car(p_obj);
            //
            // Only record pointers from younger to older cars in same train.
            // (i.e. higher numbered to lower numbered cars.)
            //
            if (p_source_car->get_car_id() > p_target_car->get_car_id()) {
                p_target_car->update_container_write_barrier(pp_obj_ref);
            }
        }
}

//
// Scan cheney spaces determines if any of the cars have objects 
// that need to be scanned. If so it scans them and returns true
// to indicate that more work might need to be done.
// Otherwise it returns false.
// If all the generations return false then there is no more cheney 
// scanning to be done.
bool
Train_Generation::scan_cheney_generation (bool doing_mos_collection) 
{ // we have too many trains for the p_train array...
//    orp_cout << "Scanning all " << _number_of_trains << " trains." << endl;
    // Scan each of the trains once.
    bool result = false;
    for (unsigned long train_idx = 0; train_idx < _number_of_trains; train_idx++) {
        if (_p_train[train_idx]->cheney_scan_cars(doing_mos_collection)) {
#ifdef GC_TRAIN_TRACE
            orp_cout << "Train trace - train " << train_idx << " did work scanning." << endl;
#endif
            result = true;
        }
#ifdef GC_TRAIN_TRACE
        else {
            orp_cout << "Train trace - train " << train_idx << " did NO work scanning." << endl;
        }
#endif      


    }
    return result;
}


bool
Train_Generation::cheney_scan_pending ()
{
    for (unsigned long train_idx = 0; train_idx < _number_of_trains; train_idx++) {
        if (_p_train[train_idx]->cheney_scan_pending()) {
            return true;
        }
    }
    return false;
}

// #endif // end NEW CHENEY CODE

void Train_Generation::enumerate_reference_queues ()
{
    for (unsigned long idx = 0; idx < _number_of_trains; idx++) {
        _p_train[idx]->enumerate_reference_queues();
    }
#ifdef GC_TRACE_WEAK_REF
    orp_cout << "Train generation just enumerated the weak/soft/phantom reference queues" << endl;
#endif // GC_TRACE_WEAK_REF
}

//
// Since only a single car is collected at once, we need to associate
// reference lists with the cars. Figure out which car the referent is in
// and enqueue the reference on that car.
//
void Train_Generation::enqueue_soft_ref (java_lang_ref_Reference *a_reference)
{
    p_global_bs->p_get_object_car(a_reference->referent)->enqueue_soft_ref (a_reference);
}
void Train_Generation::enqueue_weak_ref (java_lang_ref_Reference *a_reference)
{
    p_global_bs->p_get_object_car(a_reference->referent)->enqueue_weak_ref (a_reference);
}
void Train_Generation::enqueue_phantom_ref (java_lang_ref_Reference *a_reference)
{
    p_global_bs->p_get_object_car(a_reference->referent)->enqueue_phantom_ref (a_reference);
}

#if (GC_DEBUG>3)
//
// Routine to verify the consistency of all the spaces of this generation.
//
bool Train_Generation::verify_generation()
{
    for (unsigned long idx = 0; idx < _number_of_trains; idx++) {
        if (_p_train[idx]->verify_space()) {
            continue;
        } else {
            return false;
        }
    }
    return true;
}
#endif // _DEBUG

#if (GC_DEBUG>0)
//
// Debug routine to verify all the cars write barriers in all the trains.
//
void 
Train_Generation::verify_trains_cars_write_barriers()
{
    for (unsigned long idx = 0; idx < _number_of_trains; idx++) {
        _p_train[idx]->verify_cars_write_barriers();
    }
}
#endif // _DEBUG

// end file train_generation.cpp

