// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/step.cpp,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#include "step.h"
#include "card_table.h"
#include "gc_asserts.h"
#include "gc_hooks.h"
#include "gc_debug.h"
#include "gc_globals.h"
#include "gc_plan.h"

//
// Create a step (done by Step_Plus_Nursery_Generation).
//
Step::Step(unsigned long step_id,
     Gc_Fast_Hooks *p_gc_hooks,
     Gc_Plan       *p_gc_plan,
	 Generation    *p_container,
	 Gc_Space      *p_superior,
     Card_Table    *p_card_table,
	 Block_Store   *p_block_store)

   : Block_List(step_id, 
                p_gc_hooks,
                p_gc_plan,
                p_container, 
                p_superior, 
                p_card_table,
                p_block_store)	{
#if (GC_DEBUG>2)
assert(_p_block_store);
#endif // _DEBUG
	//
	// Discover the sizes of blocks in steps.
	//
	_step_block_size_bytes = 
		_p_gc_plan->step_block_size_bytes();

	_number_of_from_blocks = 0;
}

//
// After a reclaim, the Mrl_Gc_Vxxx calls each generation 
// to give it an opportunity to clean up. Step_Generation calls us.
//
void
Step::cleanup()
{
#ifdef GC_COPY_V2
    assert (_p_scan); // After any cleanup the scan pointer should be set.
    assert (_p_free);
    assert (_p_free == _p_scan); // check that cheney scan worked.
#endif
#if (GC_DEBUG>3)
	_debug_check_block_integrity();
#endif // _DEBUG

#ifdef CLASS_DEMOGRAPHICS
    _walk_from_space(walk_report_dead_object);
#endif

#if 0 // _DEBUG 
	//
	// FOR DEBUGGING, AVOID RECYCLING FROM BLOCKS AND READ PROTECT
	// AGAINST ANY FUTURE ACCESS BY THE ORP.
	//
    //debug_set_obsolete_container();
    debug_read_protect_from_space();

#else // NOT DEBUG
	//
	// Release all the FROM space blocks:
	//

    // Since nurseries can promote their blocks directly into a step 
    // and not all nursery blocks are super blocks we need to be prepared to
    // release the blocks one at a time.

    for (unsigned long blk_idx = 0; 
	     blk_idx < _number_of_from_blocks; 
		 blk_idx++) {
                 _p_block_store->release_super_block(_p_from_block[blk_idx]);
    }
#endif // _DEBUG

	_number_of_from_blocks = 0;

	return;
}

#if (GC_DEBUG>3)
//
// Set to obsolete spaces blocks in FROM space.
//
void
Step::debug_set_obsolete_from_space()
{
    for (unsigned long idx = 0; idx < _number_of_from_blocks; idx++) {
	    p_global_bs->set_address_obsolete_container(_p_from_block[idx]);
    }
}
#endif // _DEBUG

#if (GC_DEBUG>3)
//
// Write protect used up blocks in FROM space.
//
void
Step::debug_read_protect_from_space()
{
    for (unsigned long idx = 0; idx < _number_of_from_blocks; idx++) {
	    p_global_bs->set_address_obsolete_container(_p_from_block[idx]);
        unsigned long old_prot;
        // WARNING WARNING WARNING WARNING
        // This will not work if we allow nurseries to promote blocks that
        // are much smaller than _step_block_size_bytes.
        int success = VirtualProtect(_p_from_block[idx],
			                         _step_block_size_bytes,
									 PAGE_NOACCESS,
									 &old_prot);
		if (success == 0) {
			printf("Virtual protect error %d (0x%x)\n", GetLastError(), GetLastError());
			orp_exit(1);
		}
    }
}
#endif // _DEBUG

//
// A live object was traced down to my container.
// Evict it and update the *pp_obj (the slot) to point to the new object.
// and return the new object.
//
Java_java_lang_Object *
Step::p_evict_object(Java_java_lang_Object **pp_obj, bool doing_mos_collection) 
{
    Java_java_lang_Object *return_result;

    gc_trace (*pp_obj, "Evicting object in step, ln 154");

#if (GC_DEBUG>0)
    evict_hook(pp_obj);
#endif

#if (GC_DEBUG>2)
    assert(not_in_to_block(*pp_obj));
#endif // GC_DEBUG>2

    if (_p_superior) {
#ifdef GC_COPY_V2
        assert (0); //There should be no superiors in a simple copy scheme.
#endif
        return_result = _p_superior->p_scavenge_object(pp_obj);
	} else {    // We are the highest step

#ifdef GC_COPY_V2
        // Version 2 is a simple copy collector without generations
        // This means that only one generation exists that scavenges
        // into itself.
        return_result = this->p_scavenge_object(pp_obj);
#else
		return_result = p_container->p_tenure_object(pp_obj);
#endif // GC_COPY_V2
	}
    assert (return_result);
    return return_result;
}

//
// Beforp the stop-the-world collection, our containing generation
// gives an opportunity to set up cheney spaces.
//
void 
Step::set_up_cheney_spaces()
{
#if (GC_DEBUG>2)
	assert(_number_of_from_blocks == 0);
    assert(_number_of_blocks < MAXIMUM_BLOCKS_IN_STEP);
#endif // _DEBUG
	//
	// Rename "TO" to "FROM" space.
	//
#if (GC_DEBUG>3)
    if (stats_gc) {
        orp_cout << "Number of blocks in From space = " << _number_of_blocks << endl;
    }
#endif // GC_DEBUG>0
    for (unsigned long idx = 0; idx < _number_of_blocks; idx++) {
		_p_from_block[idx]     = _p_block[idx];
		_p_from_block_end[idx] = _p_block_end[idx];

        // clear out to block

        _p_block[idx]=NULL;
        _p_block_end[idx]=NULL;
	}
	_number_of_from_blocks = _number_of_blocks;
    // Check that the previous scans completed.
    assert (_current_block_id == _number_of_blocks -1 );
	_number_of_blocks   = 0;

    assert(sizeof(_current_block_id) == 4);
    _current_block_id = 0xFFFFFFFF;
	//
	// Prepare the TO space by adding one block and setting up _p_scan.
	//
#ifdef GC_FIXED_V1
    // ignore for fixed collector since we don't have steps.
#else
    // This is the start of the block. Make sure the scan pointer was valid
    // from any previous manipulations before we change it.
    assert ((_p_free == _p_scan) || (_p_scan == NULL));
#endif
    // Initialize _p_free and _p_scan.
    _p_free = NULL;
    _p_scan = NULL;
    
	_add_block();

    assert (_number_of_blocks == 1);
    assert (_current_block_id == 0); // Used for cheney scanning

	_resident_object_count   = 0;
	_resident_occupied_bytes = 0;
	
#if (GC_DEBUG>3)
	_debug_check_block_integrity();
#endif // _DEBUG
}

//
//  Debug time: ensure that the object being evicted is not in TO space.
// 
//  This only looks in the blocks of this step.
//
bool
Step::not_in_to_block(Java_java_lang_Object *p_obj)
{
	void *p_block = 
		_p_block_store->p_get_address_super_block_start_address((void *)p_obj);

	for (unsigned long blk_idx = 0; blk_idx < _number_of_blocks; blk_idx++) {
		if (_p_block[blk_idx] == p_block) {
			return false;
		}
	}
	return true;
}

#if (GC_DEBUG>3)
void Step::inspect(unsigned int level)
{
	cout << "    Step " << this << " :" << endl;
	cout << "    There are " << _number_of_from_blocks << " FROM blocks" << endl;
	cout << "    There are " << _number_of_blocks << " TO blocks" << endl;
	cout << "    Container is " << p_container << endl;
	cout << "    Superior  is " << _p_superior  << endl;
	cout << "    I am " << this << endl;
	cout << "    _p_base = " << _p_base << endl;
	cout << "    _p_scan = " << _p_scan << endl;
	cout << "    _p_free = " << _p_free << endl;

	for (unsigned long idx = 0; idx < _number_of_from_blocks; idx++) {
		cout << "    Inspecting From block idx# " << idx << endl;
		_inspect_block(_p_from_block[idx], _p_from_block_end[idx], idx);
	}

	for (idx = 0; idx < _number_of_blocks; idx++) {
		cout << "    Inspecting To block idx# " << idx << endl;
		_inspect_block(_p_block[idx], _p_block_end[idx], idx);
	}
}
#endif // _DEBUG

void 
Step::_walk_from_space(bool (*func)(Object_Gc_Header *,
					                Remembered_Set *))
{
    for (unsigned idx = 0; idx < _number_of_from_blocks; idx++) {
        _walk_block(_p_from_block[idx],
                    _p_from_block_end[idx],
                    func);
    }
}

// end file gc\step.cpp
