// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/nursery.cpp,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#include "card_table.h"
#include "nursery.h"
#include "gc_space.h"
#include "gc_header.h"
#include "generation.h"
#include "gc_asserts.h"
#include "gc_hooks.h"
#include "gc_debug.h"
#include "gc_globals.h"
//#include "los.h"
#include "gc_plan.h" 
#include "step.h"
#include "car.h"
#include "card_table.h"


#include <string.h>

Nursery::Nursery(unsigned long nursery_number,
                 Gc_Fast_Hooks *p_gc_hooks,
                 Gc_Plan       *p_gc_plan,
				 Generation    *p_container,
				 Gc_Space      *p_superior,
                 Card_Table    *p_card_table,
				 Block_Store   *p_bs,
				 int           block_size_bytes)

		 : Gc_Space(nursery_number,
                    p_gc_hooks,
                    p_gc_plan,
		            p_container,
					p_superior,
                    p_card_table,
					p_bs)
{
#if (GC_DEBUG>2)
	assert(_p_block_store);
#endif // _DEBUG

	//
	// NOTE that we set the generation to zero, signifying
	// that the nursery block is in young space.
	//
    void *new_block = 
        _p_block_store->p_get_new_super_block(  this, 
									            block_size_bytes,
										        YOS_GEN_NUMBER,
                                                false);

    _init_nursery (new_block, block_size_bytes);


	p_next_nursery = NULL;

    //
	// Cache the value that represents the threshold over which
	// an object is deemed large.
	//
	//_los_threshold_bytes = _p_gc_plan->los_threshold_bytes();
    // USE GLOBAL los_threshold_bytes INSTEAD
}


Nursery::~Nursery()
{
    // Nursery should last for an entire run to promote thread affinity.
//    assert (0);
    _p_block_store->release_super_block(_p_base);
}


// GC_FALSE_COLLECTION
// Given a clean block initialize the important fields. This can
// happen not only at constructor time but at times when blocks
// are moved into stesp during a 'false' collection. 
//

void
Nursery::_init_nursery (void * new_block, int block_size_bytes)
{
    Nursery *phony = this;
    if (phony) {
        phony = NULL;
    }

    _nursery_block_size_bytes = block_size_bytes;

    _p_base = _p_free = new_block; 

#ifdef OBJECT_SPLITTING
	_p_cold_free = (void *) ((POINTER_SIZE_INT) _p_base + (_nursery_block_size_bytes / 2));
#endif // OBJECT_SPLITTING   

    _p_ceiling =
        (void *)((Byte *)_p_base +
                 _nursery_block_size_bytes); // _p_block_store->get_block_size_bytes());


    // Set/reset to NULL since this block is empty.
    _p_last_object_gc_header = NULL;


#ifndef GC_CLEAR_EACH_OBJECT
//
//  For NT boxes and other OSes if memory is allocated cleared
//  we can set this to true.
//
#ifdef GC_ZERO_THREAD
    
    _is_nursery_clear = NURSERY_NOT_CLEARED;

#else
   	
    _is_nursery_clear = false; 

#endif

//    memset(_p_base, 
//	       0, 
//	       nursery_size_blocks * _block_size_bytes);
#endif
}

//
// Do any required cleanup at the end of the collection.
// We are called by our container generation, when it is called
// to do cleanup by the GC Interface.
//
void 
Nursery::cleanup() 
{

    assert (_p_scan == NULL); // We should never set this pointer in a nursery.

//  orp_cout << "Clearing nursery in cleanup." << endl;

#ifndef GC_CLEAR_EACH_OBJECT

#ifdef GC_ZERO_THREAD
    
    _is_nursery_clear = NURSERY_NOT_CLEARED;

#else
    _is_nursery_clear = false; 

#endif

//   	memset(_p_base, 
//	       0, 
//	       (int)_p_free - (int)_p_base);

//    assert ( (int _p_free - (int)_p_base) 
//             <= (_p_gc_plan->nursery_size_blocks() * _block_size_bytes));
#endif


	_resident_object_count = 0;
	_resident_occupied_bytes = 0;
	_p_free = _p_base;

#ifdef OBJECT_SPLITTING
	_p_cold_free = (void *) ((POINTER_SIZE_INT) _p_base + (_nursery_block_size_bytes / 2));
#endif // OBJECT_SPLITTING
		
#ifdef CLASS_DEMOGRAPHICS
    _walk_nursery(walk_report_dead_object);
#endif

#if 0 // _DEBUG
	_walk_nursery(verify_is_object); // why?
#endif // _DEBUG
    
#if (GC_DEBUG>3)
    // 
	// Clear out the nursery. Do it only if we aren't doing
    // intense debugging. In the latter case, it may be useful
    // to have dead objects lying around.
	//
	memset(_p_base, 
		   0xF9, 
		   _nursery_block_size_bytes);
#endif // (GC_DEBUG>3)
}

//
// If we are going to move blocks from a nursery to a step we need
// to find the base and the size of this nursery. This gives us the base.
//

void *
Nursery::get_nursery_base ()
{
    return _p_base;
}

//
// If we are going to move blocks from a nursery to a step we need
// to find the free pointer so we can tell the step about it.
//

void *
Nursery::get_nursery_free ()
{
    return _p_free;
}

#ifdef trashme
//
// If we are going to move blocks from a nursery to a step we need
// to know what space to move the objects into so we need to return the
// superior.
//

Step *
Nursery::_get_superior ()
{
    return (Step *)_p_superior;
}

#endif

#if (GC_DEBUG>4)
//
// Our generation is telling us to write protect the blocks
// in this nursery and allocate fresh ones. This is for
// debugging ORP problems.
//
void 
Nursery::debug_write_protect()
{	
	unsigned long old_prot;
	//
	// First overwrite the old nursery with a distinct pattern.
	//
	memset(_p_base, 
		   DEBUG_NURSERY_DISCARDED, 
		   _nursery_block_size_bytes);

	int success = 
		VirtualProtect(_p_base, 
		               _nursery_block_size_bytes,
				       PAGE_NOACCESS, 
				       &old_prot);
	if (success == 0) {
		//cout << "Virtual Protect failed with " << GetLastError() << endl;
		printf("Virtual protect error %d (0x%x)\n", GetLastError(), GetLastError());
		orp_exit(1);
	}
	debug_replace_intact();
}
#endif // _DEBUG

#if (GC_DEBUG>4)
//
// The in-use blocks are set aside (perhaps they were write protected
// earlier, or they are being left intact for the ORP to keep using),
// and allocate new blocks instead.
//
void
Nursery::debug_replace_cleared()
{
	//
	// First overwrite the old nursery with a distinct pattern.
	//
	memset(_p_base, 
	       DEBUG_NURSERY_DISCARDED, 
	       _nursery_block_size_bytes);
	//
	// Replace these blocks with new ones.
	//
	debug_replace_intact();
}
#endif // _DEBUG

//
// The old nursery blocks are left intact (without re-cycling). Instead, fresh
// ones are allocated for the next ORP working cycle.
//
#if (GC_DEBUG>4)
void
Nursery::debug_replace_intact()
{
	//
	// First flag the old blocks in the block store as obsolete.
	//
	debug_flag_old_blocks_obsolete();

	//
	// Replace with fresh blocks.
	//
    _p_base = 
        _p_free =
            _p_block_store->p_get_super_block(_nursery_block_size_bytes, 
			                                  this);
    _p_ceiling =
        (void *)((Byte *)_p_base +
                    _nursery_block_size_bytes -
					sizeof(void *));
	//
	// Write a distinct pattern to indicate unallocated nursery space.
	//
	memset(_p_base, 
		   DEBUG_NURSERY_REPLACED_FREE, 
		   _nursery_block_size_bytes);
}
#endif

//
// GC_FALSE_COLLECTION
//
// Take the empty block that starts at new_block and overwrite the
// current block. Reset all relevant data structures.
// 
void 
Nursery::replace_nursery_block (void *new_block, int size)
{
    _init_nursery (new_block, size);
};

#ifdef GC_SAPPHIRE
// This is used by sapphire to evict a block into the focus car.
bool Nursery::sapphire_evict_blocks(Car *focus_car)
{
   	int nursery_block_size_bytes = my_block_size_bytes(); // Maximum size.
	void *block_end = _p_free;

    void *p_new_block_base =
        _p_block_store->                            // p_global_bs if needed.
	        p_get_new_super_block(this, 
		    					    nursery_block_size_bytes,
			    					YOS_GEN_NUMBER,
                                    false); // extend heap if we need to....

    if (p_new_block_base == NULL) {
		assert (0); // stop and debug - for some reason we don't have enough blocks left.
       // This didn't work try a full collection.
       return false;
    }

    void *old_block_base = get_nursery_base();
	// Since these were nursery blocks they do not have values for the
	// _p_card_last_object_table table so they can not be scanned. This is not
	// an issue since the object are about to be collected and the new objects
	// are about to be scanned in "to" space using the cheney scan.
    // Clearing these does no good since the WB may reset one of them.
	p_global_card_table_manager->clear_cards_in_super_block(old_block_base, nursery_block_size_bytes);
    // Move it into car about to be collected.
    focus_car->scavenge_block (
       old_block_base, 
       nursery_block_size_bytes,
	   block_end); 

    replace_nursery_block (p_new_block_base, 
       nursery_block_size_bytes);
	
	verify_empty_nursery();
	
    return true;

}
#endif // GC_SAPPHIRE

//
// GC_FALSE_COLLECTION
//
// We are about to do a deferred collection and we need to evict the nursery block
// from this nursery into whatever is superior.
bool 
Nursery::evict_blocks ()
{


//       orp_cout << " spent nursery ->    " << spent_nursery << endl;
    //
    // Get enough free blocks to ensure that the nursery can be refilled before
    // we drain it of it's block.
    //
       
    int nursery_block_size_bytes = my_block_size_bytes();
	void *block_end = _p_free;

    void *p_new_block_base =
        _p_block_store->                            // p_global_bs if needed.
	        p_get_new_super_block(this, 
		    					    nursery_block_size_bytes,
			    					YOS_GEN_NUMBER,
                                    true);

//       orp_cout << " The new block base is " << p_new_block_base << " of size " << nursery_block_size_bytes << endl;
    if (p_new_block_base == NULL) {
       // This didn't work try a full collection.
//       orp_cout << " *** *** *** We are out of blocks to pass along. sorry. " << endl;
       return false;
    }

//  orp_cout << "Moving the blocks into the steps." << endl;
    void *old_block_base = get_nursery_base();
    // Move it into the first step.
    ((Step *)_p_superior)->scavenge_block (
       old_block_base, 
       nursery_block_size_bytes,
       block_end); 
    replace_nursery_block (p_new_block_base, 
       nursery_block_size_bytes);
    return true;
}



//
// This routine is used when the nursery has been scavenged, i.e. all
// remaining objects are garbage, and the nursery has not yet been
// recycled and returned for re-use. The "Obsolete Container" space
// that is associated with these blocks is used to catch illegal
// ORP access to any of these blocks. Such an illegal access could occur
// either due to a ORP bug that didn't enumerate all references, or due
// to a GC bug that did not transitively reach and scavenge all reachable
// objects.
//
#if (GC_DEBUG>3)
void
Nursery::debug_flag_old_blocks_obsolete()
{
	void *p_block = _p_base;
		
	p_global_bs->set_address_obsolete_container(p_block);

}
#endif // _DEBUG

//
// This is the code we need to implement in order to do the shared nursery optimization.
//
// Nursery::gc_malloc(unsigned object_size_bytes, Partial_Reveal_VTable *p_vtable)
// {
    // unsigned int old_amount_available;
    // old_amount_available = amount_available;
    // if (old_amount_available < object_size_bytes) {return NULL;}
    // new_amount_available = amount_available - object_size_bytes;
    // result = nursery_base + nursery_size - amount_available;
    // lock cmpxchg old_amount_available(the accumlator), 
    //              amount_available (the dest), 
    //              new_amount_available (the new value);
    // if (!(ZeroFlag)) {
    //    return NULL;
    // }
    // return result;
// }

Java_java_lang_Object *
 
Nursery::gc_malloc(unsigned object_size_bytes, Partial_Reveal_VTable *p_vtable)
 
{
#if (GC_DEBUG>3)
//	assert(_current_owner_thread==GetCurrentThreadId());
	assert(los_threshold_bytes>0);
	assert(this);
#endif

#if (GC_DEBUG>1)
    //_p_gc_hooks->_before_malloc_hook(this, object_size_bytes, p_vtable);
#endif

#ifdef RECLAIM_EVERY_ALLOCATION
	if (debug_reclaim_frequency > 0) {
		if (orp_initialized &&
			(get_thread_allocation_count() > debug_reclaim_frequency)) {
			return NULL;
		} else {
			increment_thread_allocation_count();
		}
	}
#endif

#ifdef SPACE_DEMOGRAPHICS
    void record_for_space_demographics(unsigned obj_size); // in los logic now.
    record_for_space_demographics(object_size_bytes);
#endif

#ifdef CLASS_DEMOGRAPHICS
    record_for_class_demographics(p_vtable);
#endif


	// If we are allocated near the top of memory this still works since
	// we aren't using the highest (sign) bit to indicate that we need
	// to pay special attention to this object, instead we are using the 
	// next to highest bit. This ensures that all sizes are positive and
	// that we can use signed arithmetic for these pointer compares.
    //
    // Does an addition of a constant and a modulo to deal with padding.
	unsigned int real_size_bytes = object_size_bytes;

#ifdef OBJECT_SPLITTING
	// The size of the hot and cold sections of the object being allocated.
	unsigned int sub_section_real_size_bytes = real_size_bytes / 2;
#endif // OBJECT_SPLITTING


#if (GC_DEBUG>3)
	real_size_bytes = 
		add_gc_space_overhead(object_size_bytes);
#endif

    //
    // old_amount_available = amount_available;
    // if (old_amount_available < real_size_bytes) {return NULL;}
    // new_amount_available = amount_available - real_size_bytes;
    // result = nursery_base + amount_available;
    // lock cmpxchg old_amount_available(the accumlator), 
    //             amount_available (the dest), 
    //             new_amount_available (the new value);
    // if (!(ZeroFlag)) {
    //    return NULL;
    // }
    // return result;

#ifdef OBJECT_SPLITTING
	// real_size_bytes/2 will be allocated in the hot section, and the same
	// in the cold section for the object for which space is requested.
	// Here, _p_free points the first free byte in the hot section, and 
	// _p_cold_free points to the first free byte in the cold section.
	// _p_ceiling points the last byte in the cold section, as well as the
	// nursery itself.

	// Assert that the pointers corresponding to hot and cold free areas are moving in step.
	assert(((POINTER_SIZE_INT) _p_cold_free - (POINTER_SIZE_INT)_p_free) == 
                (((POINTER_SIZE_INT) _p_ceiling - (POINTER_SIZE_INT) _p_base) / 2));

	if(((POINTER_SIZE_INT)_p_cold_free + sub_section_real_size_bytes) >= 
            (POINTER_SIZE_INT)_p_ceiling)
		        return NULL;
#else
    if(((POINTER_SIZE_INT)_p_free + real_size_bytes) >= (POINTER_SIZE_INT)_p_ceiling) { 
        return NULL;
	}
#endif // OBJECT_SPLITTING

#if (GC_DEBUG>1)
	_resident_object_count++;
	_resident_occupied_bytes += real_size_bytes;
#endif

#ifdef GC_CLEAR_EACH_OBJECT 

#ifdef OBJECT_SPLITTING
    memset (_p_free, 0, sub_section_real_size_bytes);
	memset (_p_cold_free, 0, sub_section_real_size_bytes);
#else
    // clear out each object individually.
    memset (_p_free, 0, real_size_bytes);
#endif // OBJECT_SPLITTING

#else // defined GC_CLEAR_EACH_OBJECT

#ifdef GC_ZERO_THREAD

    if (_is_nursery_clear == NURSERY_CLEARED) { 

        // Nothing to do ...clearing is done. Likeliest case.

    } else { 

        // I got to it first. So zero it myself.
        // But need to use cmpxchg since there could be a race condition.

        volatile int nursery_clear_state = 
            
                        (int) InterlockedCompareExchange((PVOID *)&(_is_nursery_clear), 
                                                            (PVOID)NURSERY_CLEARING_IN_PROGRESS, 
                                                            (PVOID)NURSERY_NOT_CLEARED);
        
        if (nursery_clear_state == NURSERY_NOT_CLEARED) { 
            
            // I grabbed it. So clear it.
            memset(_p_free, 0, ((POINTER_SIZE_INT)_p_ceiling - (POINTER_SIZE_INT)_p_free));
            _is_nursery_clear = NURSERY_CLEARED;
            //printf("thread just cleared its own nursery\n");

        } else if (nursery_clear_state == NURSERY_CLEARING_IN_PROGRESS) { 

            // Zero thread got to it and is clearing it. So sleep till it is done.
            while (_is_nursery_clear == NURSERY_CLEARING_IN_PROGRESS) {
                // printf("Stuck");
                Sleep(0);
            }
            // Zeroing should be done by now.
            assert(_is_nursery_clear == NURSERY_CLEARED);

        } else if (nursery_clear_state == NURSERY_CLEARED) { 

            // Surprising, but nothing to do here...
        } else { 

            // Wrong value!!
            assert(0);
        }
    }
   
#else // defined GC_ZERO_THREAD

    if (!_is_nursery_clear) { // See below if this conditional shows up when tuning.

        // Unless we clear the nursery in bulk we need to clear each object
        // one at a time. This seems to provide about a 2% speedup in some
        // Spec marks while slowing done not more than a 0.5%. The overal
        // value is at least a 0.5% improvement in the Spec marks.
        // We can fold this check into the limit check by setting
        // the _p_ceiling to 0 when we release a nursery and then
        // checking _is_nursery_clear in the slow code. If it is false we know
        // we have an unused nursery. If it is not we know we have an overflow.
        memset(_p_free, 0, ((POINTER_SIZE_INT)_p_ceiling - (POINTER_SIZE_INT)_p_free));

        _is_nursery_clear = true;
#if (GC_DEBUG>0)
        assert (_p_base == _p_free);
#endif // GC_DEBUG>0
    }

#endif // defined GC_ZERO_THREAD

#endif // defined GC_CLEAR_EACH_OBJECT

    Object_Gc_Header *p_header = (Object_Gc_Header *)_p_free;
	
	//
	// Record the most recently allocated object into this space to
	// facilitate subsequent scanning.
	//
	_p_last_object_gc_header = p_header;

#if (GC_DEBUG>2)
    //
    // Mark the trailer for debugging
    //
    *(int *)((char *)p_header + real_size_bytes - 4) = OBJECT_DEBUG_TRAILER;
#endif
 
    // Another addition here.
    Partial_Reveal_JavaObject *temp_p_object = 
        (Partial_Reveal_JavaObject *)get_object_from_gc_header(p_header);

    // We need to take a Partial_Reveal_VTable and move it into a Java_java_lang_Object
    // but Java_java_lang_Object thinks it is getting a VTable. 
    temp_p_object->vt = p_vtable;
    Java_java_lang_Object *p_object = (Java_java_lang_Object *)temp_p_object; 
 
#ifdef OBJECT_SPLITTING
	// Move the hot and cold section free pointers by the same number of bytes ahead.
	
	_p_free = (void *)((char *)_p_free + sub_section_real_size_bytes);		
	_p_cold_free = (void *)((char *)_p_cold_free + sub_section_real_size_bytes);	
#else
    _p_free = (void *)((char *)_p_free + real_size_bytes);
#endif	// OBJECT_SPLITTING

#if (GC_DEBUG>1)
    //p_gc_hooks->_after_malloc_hook(this, object_size_bytes, p_vtable, p_object);
#endif

#if (GC_DEBUG>2)
#ifdef OBJECT_SPLITTING
	assert(_p_cold_free < _p_ceiling);
#else
	assert(_p_free < _p_ceiling);
#endif // OBJECT_SPLITTING
#endif

    return p_object;
}

#if (GC_DEBUG>0)
void Nursery::verify_empty_nursery ()
{
	assert (_p_base == _p_free);
	assert (_p_base < _p_ceiling);
}
#endif // (GC_DEBUG>0)

#if (GC_DEBUG>3)
void Nursery::inspect(unsigned int level)
{
	orp_cout << "    Nursery id# " << identity << endl;
	orp_cout << "    " << resident_count() << " objects resident" << endl;
	orp_cout << "    " << resident_occupied_bytes() << " bytes occupied" << endl;
	orp_cout << "    My container is " << p_container << endl;
	orp_cout << "    My superior is " << _p_superior << endl;
	orp_cout << "    My base is " << _p_base << " and ceiling is " << _p_ceiling << endl;
	orp_cout << "    My free is " << _p_free << " and my scan is " << _p_scan << endl;
	orp_cout << "    My owning thread is " << _current_owner_thread << endl;
} 

bool Nursery::verify_space()
{
	//
	// Ensure that all contents are objects and they are properly aligned.
	//
	_walk_nursery(verify_is_object);
	//
	// Make sure none of the objects have their busy bit set.
	//
	_walk_nursery(assert_busy_bit_not_set);
	//
	// Make sure none of the objects have their forward bit set.
	//
	_walk_nursery(assert_object_not_forwarded);

	return true;
}
#endif // _DEBUG

//
// This routine takes a function argument that accepts a GC header
// and walks the entire nursery applying that function to each
// object. The walk terminates if the function returns false.
//
bool Nursery::_walk_nursery(bool(*func)(Object_Gc_Header *,
							            Remembered_Set *))
{
	bool retval;
	Object_Gc_Header *p_scan_gc_hdr = (Object_Gc_Header *)_p_base;
	while ((void *)p_scan_gc_hdr < _p_free) {
		if (retval = func(p_scan_gc_hdr, NULL)) {
			p_scan_gc_hdr = p_scan_forward_over_object(p_scan_gc_hdr);
			//continue;
		} else {
			return retval;
		}
	}
	return retval;
}

// end file gc\nursery.cpp
