// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/exceptions.cpp,v 1.5 2002/01/07 10:14:06 gwu2 Exp $
//


#include "platform.h"
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <float.h>

#include "orp_types.h"
#include "jit_runtime_support.h"
#include "Class.h"
#include "environment.h"
#include "method_lookup.h"
#include "stack_manipulation.h"
#include "object_layout.h"
#include "orp_utils.h"
#include "orp_synch.h"
#include "orp_threads.h"
#include "jit_intf.h"
#include "jit_intf_cpp.h"
#include "compile.h"
#include "ini.h"
#include "orp_stats.h"

#include "exceptions.h"

#include "jvmdi_clean.h"

#ifdef ORP_POSIX
#include "platform2.h"
#endif



////////////////////////////////////////////////////////////////////////////
// begin corp exception throwing and catching functions


//
// Create an exception object of a given class if the exc_obj_is_dead flag
// is false.  The context and the is_first flag are needed by
// java_lang_Throwable_fillInStackTrace.
//
static volatile Java_java_lang_Object *
create_object_lazily(Class *clss,
                     bool exc_obj_is_dead,
                     Frame_Context *context,
                     Boolean is_first,
                     Frame_Context *handler_context,
                     Method *constructor,
                     uint8 *constructor_args
                     )
{
    Global_Env *env = ORP_Global_State::loader_env;
    p_TLS_orpthread->throw_context = context;
    p_TLS_orpthread->throw_context_is_first = is_first;

    volatile Java_java_lang_Object *obj_volatile = 0;
    GC_Frame gcf;
    orp_push_gc_frame(&gcf, (void *)&obj_volatile, sizeof(volatile Java_java_lang_Object *)); 

    if(exc_obj_is_dead) {
        obj_volatile = 0;
#ifdef ORP_STATS
        orp_stats_total.num_exceptions_object_not_created++;
#endif
    } else {
        //obj_volatile = class_alloc_new_object_and_run_default_constructor(clss);
        obj_volatile = class_alloc_new_object_and_run_constructor(clss,
                                                                  constructor,
                                                                  constructor_args);
    }

    p_TLS_orpthread->throw_context = 0;
    orp_pop_gc_frame(&gcf);

    return (Java_java_lang_Object *)obj_volatile;
} //create_object_lazily



//
// The main logic for catching an exception is implemented here.  There are
// several wrappers which call orp_throw.
//
void  __stdcall orp_throw(Frame_Context *context,
                          volatile Java_java_lang_Object **p_obj,
                          Boolean is_first,
                          Method *constructor,
                          uint8 *constructor_args)
{
#ifdef ORP_STATS
    orp_stats_total.num_exceptions++;
#endif

    // Reset the fp.
    _fpreset();

    Global_Env *env = ORP_Global_State::loader_env;

    // The object can be one of two things:
    //  1. The actual exception object, or
    //  2. A class handle representing the type of the exception object that
    //     was thrown.  The actual object hasn't been created yet, and may have
    //     to be created lazily if the ORP can't prove that the object is dead
    //     at the entry to exception handler.
    // We refer to the former scenario as EEO (Eager Exception Object Creation) and
    // to the latter one -- LEO (Lazy Exception Object Creation).
    volatile Java_java_lang_Object *obj = *p_obj;
    assert(obj);

    assert(obj->vt);
    Class *clss = obj->vt->clss;
    // At this point the clss represents a subclass of java.lang.Throwable for EEO and
    // java.lang.Class for LEO.

    Boolean no_exc_obj = clss == env->JavaLangClass_Class;
    if(no_exc_obj) {
        // LEO.
        // Make clss point to the class of the exception object rather than
        // java.lang.Class.
        clss = (Class *)obj;
    }
    // At this point clss is the type of the thrown exception no matter whether
    // we create the exception object lazily or eagerly.
#if 0
    printf("orp_throw: '%s'\n", clss->name->bytes);
#endif

#ifdef ORP_STATS
    clss->num_throws++;
    int depth = -1;
#endif

    // Save a copy of the context so that we can later construct the stack trace.
    Boolean saved_is_first = is_first;
    Frame_Context saved_context;
    orp_copy_frame_context(&saved_context, context);

    JIT_Specific_Info *jit_info = methods.find((void *)*(context->p_eip));
    ORP_Code_Type orpct = orp_identify_eip((void *)*(context->p_eip));
#ifdef ORP_STATS
    jit_info->num_throws++;
#endif
    for(; orpct == ORP_TYPE_JAVA; is_first = FALSE) {
#ifdef ORP_STATS
        depth++;
#endif
        unsigned num_exc = jit_info->get_num_target_exception_handlers();


#if 0
        printf("There are %d handlers for ip=0x%p in %s.%s%s\n",
               num_exc,
               (void *)(*context->p_eip),
               jit_info->get_method()->get_class()->name->bytes,
               jit_info->get_method()->get_name()->bytes,
               jit_info->get_method()->get_signature()->descriptor->bytes
               );
#endif


        for(unsigned i = 0; i < num_exc; i++) {
            Target_Exception_Handler_Ptr handler =
                jit_info->get_target_exception_handler_info(i);
            if(!handler) {
                continue;
            }
            if(handler->is_in_range((void *)*(context->p_eip), is_first)) {
                if(handler->is_assignable(clss)) {
#ifdef ORP_STATS
                    jit_info->num_catches++;
                    if(depth == 0) {
                        orp_stats_total.num_exceptions_caught_same_frame++;
                    }
                    if(handler->is_exc_obj_dead()) {
                        orp_stats_total.num_exceptions_dead_object++;
                    }
#endif
                    // We found the right handler.
                    jit_info->get_jit()->fix_handler_context(jit_info->get_method(),
                                                             context,
                                                             is_first);
#ifdef POINTER64
                    context->eip = (uint64)handler->get_handler_ip();
#else
                    context->eip = (uint32)handler->get_handler_ip();
#endif
                    context->p_eip = &(context->eip);

                    jvmdi_exception(&saved_context, p_obj, context);
#ifdef _DEBUG

                    if (get_current_thread_exception() != 0) {
                        orp_cout << "Removed assert(get_current_thread_exception() == 0)" << endl;
//                    assert(get_current_thread_exception() == 0);
                    }
#endif
                    
                    if (no_exc_obj)
                        set_current_thread_exception(0);
                    else
                        set_current_thread_exception( (void *)obj);

					// toss this, gc_malloc() will block instead -->> gc_safepoint_check(context, p_obj, no_exc_obj);

                    set_current_thread_exception(0);

                    if(no_exc_obj) {
                        *p_obj = create_object_lazily(clss,
                                                      handler->is_exc_obj_dead(),
                                                      &saved_context,
                                                      saved_is_first,
                                                      context,
                                                      constructor,
                                                      constructor_args);
                    }
                    assert( obj->vt->clss->name->bytes );

                    return;
                }
            }
        }

        // No appropriate handler found, unwind the stack frame.
        if(jit_info->get_method()->is_synchronized()) {
            if(jit_info->get_method()->is_static()) {
                orp_monitor_exit((Java_java_lang_Object *)jit_info->get_method()->get_class());
            } else {
                void **p_this =
                    (void **)jit_info->get_jit()->get_address_of_this(jit_info->get_method(),
                                                                       context,
                                                                       is_first);
                orp_monitor_exit((Java_java_lang_Object *)*p_this);
            }
        }

        jit_info->get_jit()->unwind_stack_frame(jit_info->get_method(), context, is_first);
#ifdef USE_IA64_JIT
        uint64 *unwind_register_stack(Frame_Context *fc, uint64 *bsp_arg, uint64 pfs);
        assert(context);
        assert(context->p_ar_pfs);
        context->bsp = unwind_register_stack(context, context->bsp, *context->p_ar_pfs);
#endif
        orpct = orp_identify_eip((void *)*(context->p_eip));
        if(orpct == ORP_TYPE_JAVA) {
            jit_info = methods.find((void *)*(context->p_eip));
            assert(jit_info);
        } else {
            // Next stack frame belongs to a native method.
            // We have to:
            //  1. If LEO, create the exception object.
            //  2. Signal the exception in the orp_thread data structure
            //  3. Resume execution in the native code as if the Java method returned.

			//if (no_exc_obj)
			//	set_current_thread_exception(0);
			//else
			//	set_current_thread_exception( (void *)obj);

			//gc_safepoint_check(context, p_obj, no_exc_obj);

            printf("No Java handlers found\n");

            if(no_exc_obj) {
                obj = create_object_lazily(clss,
                                           false,
                                           &saved_context,
                                           saved_is_first,
                                           context,
                                           constructor,
                                           constructor_args);
            }
            assert( obj->vt->clss->name->bytes );
#if 0
            printf("*** %s\n", obj->vt->clss->name->bytes );
			//orp_print_stack_trace_debug(128);
			void print_stack_trace(FILE *f, Java_java_lang_Throwable *exc);
			print_stack_trace(NULL, (Java_java_lang_Throwable*)obj);
#endif			
            set_current_thread_exception((void *)obj);
            *p_obj = 0;
            return;
        }
    }
    assert(0);
} //orp_throw



// end corp exception throwing and catching functions
////////////////////////////////////////////////////////////////////////////



// end main entry points for throwing exceptions
////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////
// begin Target_Exception_Handler

Boolean Target_Exception_Handler::is_in_range(void *eip, Boolean is_first)
{
#ifdef POINTER64
    if(is_first) {
        return ((uint64)eip >= (uint64)_start_ip) &&
            ((uint64)eip < (uint64)_end_ip);
    } else {
        return ((uint64)eip > (uint64)_start_ip) &&
            ((uint64)eip <= (uint64)_end_ip);
    }
#else
    if(is_first) {
        return ((unsigned)eip >= (unsigned)_start_ip) &&
            ((unsigned)eip < (unsigned)_end_ip);
    } else {
        return ((unsigned)eip > (unsigned)_start_ip) &&
            ((unsigned)eip <= (unsigned)_end_ip);
    }
#endif
} //Target_Exception_Handler::is_in_range



Boolean Target_Exception_Handler::is_assignable(Class *exc)
{
    if(!_exc) {
        return TRUE;
    }
    for(; exc; exc = exc->super_class) {
        if(exc == _exc) {
            return TRUE;
        }
    }
    return FALSE;
} //Target_Exception_Handler::is_assignable



// end Target_Exception_Handler
////////////////////////////////////////////////////////////////////////////



void print_uncaught_exception_message(FILE *f, Java_java_lang_Throwable *exc)
{
    Global_Env *env = ORP_Global_State::loader_env;
    Class *clss = exc->vt->clss;
    //printf("Uncaught exception: %s: ", clss->name->bytes);
    printf("Uncaught exception:\n");
    // the following is unnecessary, classpath will print "message"
    /*
    bool real_exc_obj = clss != env->JavaLangClass_Class;
    if(real_exc_obj) {
        Java_java_lang_String *msg = get_java_lang_throwable_field_message(exc);
        if(msg) {
            int32 offset;
            int32 count;
            JavaArrayOfChar *value;
            get_java_lang_string_fields(msg, &value, &offset, &count);
            uint16 *chars = value->body;
            for(int i = 0; i < count; i++) {
                uint16 ch = chars[offset + i];
                char c;
                if(ch >= 32 && ch < 127) {
                    c = (char)ch;
                } else {
                    c = '.';
                }
                printf("%c", c);
            }
        }
    }
    printf("\n");
    */
    print_stack_trace(f, exc);
} //print_uncaught_exception_message



////////////////////////////////////////////////////////////////////////////
// begin throwing exception from native code


//
// WARNING! WARNING! WARNING! WARNING! WARNING! WARNING! WARNING! WARNING!
//
// This method only works when involved from native code that was entered
// using a stub which saved away some info which can be used to unwind the stack
// to the last Java frame.
//
void throw_java_exception(const char *exception_name)
{
    Global_Env *env = ORP_Global_State::loader_env;
    String *exc_str = env->string_pool.lookup(exception_name);
    Loader_Exception ld_exc;
    Class *exc_clss = class_load_verify_prepare(env, exc_str, &ld_exc);
    assert(exc_clss);
    Java_java_lang_Object *exc = 
        class_alloc_new_object_and_run_default_constructor(exc_clss);

    throw_java_exception_from_native(exc);

} //throw_java_exception



void throw_java_exception(const char *exception_name, const char *message)
{
    Global_Env *env = ORP_Global_State::loader_env;

    // Get the class
    String *exc_str = env->string_pool.lookup(exception_name);
    Loader_Exception ld_exc;
    Class *exc_clss = class_load_verify_prepare(env, exc_str, &ld_exc);
    assert(exc_clss);

    // Get the method for the constructor
    String *init_name = env->Init_String;
    String *init_descr = env->string_pool.lookup("(Ljava/lang/String;)V");
    Signature *init_sig = env->sig_table.lookup(init_name, init_descr);
    assert(init_sig);
    Method *exc_init = class_lookup_method(exc_clss, init_sig);
    assert(exc_init);

    volatile Java_java_lang_String *jls =
        create_java_lang_String_from_C_string(message);
    GC_Frame gcf_jls;
    orp_push_gc_frame(&gcf_jls, (void *)&jls, sizeof(volatile Java_java_lang_String *)); 

    volatile Java_java_lang_Throwable *exc_obj =
        (Java_java_lang_Throwable *)class_alloc_new_object(exc_clss);
    GC_Frame gcf_exc_obj;
    orp_push_gc_frame(&gcf_exc_obj,
                      (void *)&exc_obj,
                      sizeof(volatile Java_java_lang_Throwable *)); 

    J_Value args[2];
    args[0].r = (void *)exc_obj;
    args[1].r = (void *)jls;
    orp_execute_java_method_array(exc_init, 0, args);
    if(get_current_thread_exception()) {
        // Panic.
        assert(0);
    }

    orp_pop_gc_frame(&gcf_exc_obj);
    orp_pop_gc_frame(&gcf_jls);

    throw_java_exception_from_native((void *)exc_obj);

} //throw_java_exception


// end throwing exception from native code
////////////////////////////////////////////////////////////////////////////




////////////////////////////////////////////////////////////////////////////
// begin manipulation of the per-thread exception object.


void *get_current_thread_exception()
{
    return (void *)p_TLS_orpthread->p_exception_object;
} //get_current_thread_exception



void __stdcall set_current_thread_exception(void *obj)
{
    p_TLS_orpthread->p_exception_object = (Java_java_lang_Object *)obj;
} //set_current_thread_exception



void clear_current_thread_exception()
{
    p_TLS_orpthread->p_exception_object = 0;
} //clear_current_thread_exception



void rethrow_current_thread_exception()
{
    void *exc = get_current_thread_exception();
    assert(exc);
    clear_current_thread_exception();

    throw_java_exception_from_native(exc);

} //rethrow_current_thread_exception


void rethrow_current_thread_exception_if_pending()
{
    void *exc = get_current_thread_exception();
    if(exc) {
        clear_current_thread_exception();
        throw_java_exception_from_native(exc);
    }
} //rethrow_current_thread_exception_if_pending


// end manipulation of the per-thread exception object.
////////////////////////////////////////////////////////////////////////////
