// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_lang_Float.cpp,v 1.8 2002/01/09 03:45:03 gwu2 Exp $
//


#include "platform.h"
#include "jni.h"
#include "assert.h"
#include <math.h>
#include <string.h>
#include "java_lang_Float.h"

#include "include/gnu_classpath_jni_utils.h"


union my_union
{
	jfloat f;
	jint i;
};


/*
 * Class:     java_lang_Float
 * Method:    floatToIntBits
 * Signature: (F)I
 */

JNIEXPORT jint JNICALL Java_java_lang_Float_floatToIntBits
  (JNIEnv *jenv, jclass clazz, jfloat float_val)
{
	union my_union mu;
	mu.f = float_val;

	// The result is a representation of the floating point 
	// argument in the IEEE 754 floating point "single-format"
	// bit layout...

	// Bits 30-23 represent the exponent.
	jint e = mu.i & 0x7f800000;
	// Bits 22-0 represent the significand.
	jint s = mu.i & 0x007fffff;

	// If the argument is NaN the result will be 0x7fc00000.
	if ((s !=0) && (e == 0x7f800000))
		mu.i = 0x7fc00000;

	return mu.i;
} // Java_java_lang_Float_floatToIntBits


/*
 * Class:     java_lang_Float
 * Method:    intBitsToFloat
 * Signature: (I)F
 */

JNIEXPORT jfloat JNICALL Java_java_lang_Float_intBitsToFloat
  (JNIEnv *, jclass, jint int_val)
{
	// Argument is in IEEE 754 FP "single format", and the 
	// floating point value is returned as result.
	union my_union mu;
	mu.i = int_val;
	return mu.f;
} // Java_java_lang_Float_intBitsToFloat


/*
 * Class:     Float
 * Method:    parseFloat
 * Signature: (Ljava/lang/String;)F
 */
JNIEXPORT jfloat JNICALL Java_java_lang_Float_parseFloat
  (JNIEnv *env, jclass clazz, jstring str)
{
	jint count;
    jcharArray char_array;
	jint offset;
	get_string_specifics(env, str, &count, &offset, &char_array);
    
	unsigned strLength = (unsigned)count;
    unsigned i = 0;

	bool error = false;
    bool minus = false;
    bool digitsPresent = false;
    bool afterDecimalPoint = false;
	
	bool afterExponent = false;
	long exponent = 0;
	bool expminus = false;

    double val = 0; // The value to be returned.
    
	double fract = 0.1;
    
    jboolean isCopy;
    jchar *jchar_array = env->GetCharArrayElements(char_array, &isCopy);
	jchar_array += offset; //skip the offset

    char ch = (char)jchar_array[0];
    if(ch == '-' || ch == '+') {
        minus = ch == '-';
        i++;
    }

    for(; i < strLength; i++) {
        int ch = (int)jchar_array[i];
        if(ch == '.') {
            if(afterDecimalPoint || !digitsPresent) {
                assert(0);
            }
            afterDecimalPoint = true;
            continue;
        }
        if(ch >= '0' && ch <= '9') {
			ch -= '0';
			if(afterExponent){
				exponent = 10 * exponent + ch;
				continue;
			}
            if(afterDecimalPoint) {
                val = val + fract * ch;
                fract *= 0.1;
            } else {
                val = 10 * val + ch;
                digitsPresent = true;
            }
        } else {
			switch(ch){
			case 'e':
			case 'E':
				afterExponent = true;
				break;
			case 'f':
			case 'F':
			case 'd':
			case 'D':
				break;
			case '+':
				break;
			case '-':
				if(!afterExponent || 
					(jchar_array[i-1] != 'e' && jchar_array[i-1] != 'E') ){
					error = true;
					break;
				}
				expminus = true;
				break;
			default:
				error = true;
			}
			if (error)
				break;
		}
    }
    if(!digitsPresent) {
        error = true;
    }
    if(minus) {
        val *= -1;
    }
	if(expminus)
		exponent = -exponent;
	if(exponent != 0){
		val *= pow(10, exponent);
	}

    env->ReleaseCharArrayElements(char_array, jchar_array, JNI_ABORT);

    if(error) {
        jclass clazz = env->FindClass("java/lang/NumberFormatException");
        jint ok = env->ThrowNew(clazz, "from Java_java_lang_Double_doubleValueOf");
        return 0;
    }
    return (jfloat) val;
}



/*
 * Class:     Float
 * Method:    toString
 * Signature: (F)Ljava/lang/String;
 */

JNIEXPORT jstring JNICALL Java_java_lang_Float_toString
  (JNIEnv *jenv, jclass clazz, jfloat float_val)
{  
    // Use "sprintf" to decode to string format.   
    char *str = (char *) malloc(sizeof(char) * 1024);
    assert(str);
    sprintf(str, "%f", float_val); 
	//remove the tail '0', e.g, 2.800 -> 2.8 3.00->3.0
	char *p = str + strlen(str) - 1;
	while(*p == '0'){
		if( *(p-1) == '0' )
			*p = '\0';
		else{
			if(*(p-1) != '.')*p = '\0';
			break;
		}
		--p;
	}
  
    jstring ret_string = jenv->NewStringUTF(str);
    assert(ret_string);
    free(str);
    return ret_string;
} // Java_java_lang_Float_toString