// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/local_cse.cpp,v 1.2 2001/08/13 09:54:43 xhshi Exp $
//



#include "defines.h"
#include <iostream.h>
#include "ir.h"
#include "expression.h"
#include "local_cse.h"

//
// use type_map to map array types "BCDFIJLSARZV" to 0-12
//
char type_map['Z' - 'A'+1] = {
    // A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P, Q, R, S, T, U, V, W, X, Y, Z
       8, 0, 1, 2,-1, 3,-1,-1, 4, 5,-1, 6,-1,-1,-1,-1,-1, 9, 7,-1,12,11,-1,-1,-1,10
};

//
// bit vector manipulation routines
//
static unsigned bv_word_no(unsigned i) { return i >> 5;}

unsigned static bv_n_words(unsigned i) {
	if (i == 0)	return 0;
	return bv_word_no(i - 1) + 1;
}

unsigned static bv_mask(unsigned i) {
	// grab lower 5 bits
	unsigned m = i & 0x001f;
	m = 1 << m;
	return m;
}

Kill_Set::Kill_Set(Mem_Manager& mem, unsigned bv_size) : max_bv_size (bv_size){
    _array_type = _alias_across_call = 0;
    //
    // create bit vector and init it to NULL
    //
    unsigned n_word = bv_n_words(bv_size);
    _opnds_kill = (unsigned*)mem.alloc(n_word*sizeof(unsigned));
    unsigned i;
    for (i=0; i < n_word; i++)
        _opnds_kill[i] = 0;
}

//
// return non-zero if i-th bit of _vreg_kill is not set
//
bool Kill_Set::is_in_kill_set(unsigned i) {
    if( i < max_bv_size && (_opnds_kill[bv_word_no(i)] & bv_mask(i)))
        return true;
    return false;
}

//
// set i-th bit of _vreg_kill
//
void Kill_Set::add_to_kill(unsigned i) {
    assert(i < max_bv_size);
    unsigned word_no = bv_word_no(i);
    _opnds_kill[word_no] |= bv_mask(i);
}


//
// merge (OR) ks info into the current kill set
//
void Kill_Set::union_with(Kill_Set *ks) {
    if (ks == NULL) return;
    //
    // copy from ks to the current kill_set
    //
    unsigned n_word = bv_n_words(ks->max_bv_size);
	assert(bv_n_words(max_bv_size) >= n_word);
	//
	// union opnd_kill
	//
    unsigned i;
	for (i = 0; i < n_word; i++)
		_opnds_kill[i] |= ks->_opnds_kill[i];
	//
	// union array type and alias_across_call
	// 
	_array_type |= ks->_array_type;
	_alias_across_call |= ks->_alias_across_call;
}

void Live_LCSE::rm_lcse_elem(LCSE *elem, int& size) {
    elem->unlink();
    Inst *inst = elem->exp()->inst();
    //
    // t = x      "call foo" should kill "a0 = t" but not "t = x"
    // a0 = t     a0 expression carries inst "t = x" so we check
    // call foo   if "elem->exp() == inst->exp"
    //
    if (inst->is_assignment() && elem->exp() == inst->exp)
        ((Assign_Inst*)inst)->disable_pseudo_asgn();

    elem->exp()->set_inst(NULL);
    pool.free(elem);
    size--;
}

//
// remove a local cse whose exp->is_in_kill_set(id) == true from the list
//
void Live_LCSE::kill_lcse_contain(unsigned id) {
#if 0
    REMOVE_FROM_LCSE_LIST(_list_head,_list_size,is_in_kill_set(id));
#else // 0
    LCSE *lcse;
    for (lcse = _list_head.next(); lcse != &_list_head; ) { 
        LCSE *nxt = lcse->next(); 
        Exp *exp = lcse->exp();
        if (exp->is_in_kill_set(id))
            rm_lcse_elem(lcse,_list_size); 
        else if (exp != exp->inst()->exp &&
            exp->inst()->exp->is_in_kill_set(id))
            rm_lcse_elem(lcse,_list_size);
        lcse = nxt;
    }
    assert(_list_size >= 0); 
#endif // 0
    REMOVE_FROM_LCSE_LIST(_array_cmp_head,_cmp_size,is_in_kill_set(id));
}

//
// remove a local cse that contains an array of "type"
//
void Live_LCSE::kill_lcse_contain_array(O3_Jit_Type type) {
    REMOVE_FROM_LCSE_LIST(_list_head,_list_size,contain_of_array(type));
    REMOVE_FROM_LCSE_LIST(_array_cmp_head,_cmp_size,contain_of_array(type));
}

//
// remove a local cse whose expression is e
//
void Live_LCSE::kill_lcse_contain(Exp *e) {
    REMOVE_FROM_LCSE_LIST(_list_head,_list_size,is_same_exp(e));
    REMOVE_FROM_LCSE_LIST(_array_cmp_head,_cmp_size,is_same_exp(e));
}

//
// remove a local cse whose value may be killed by a call
//
void Live_LCSE::kill_lcse_alias_call() {
    REMOVE_FROM_LCSE_LIST(_list_head,_list_size,is_alias_across_call());
    REMOVE_FROM_LCSE_LIST(_array_cmp_head,_cmp_size,is_alias_across_call());
}

//
// insert l to head list and update size
//
void Live_LCSE::insert_lcse_to_list(LCSE& head, int max_size, int& size, LCSE *l) {
    if (size == max_size)
        rm_lcse_elem(head.prev(),size);
    l->insert_after(&head); // add to the beginning of list
    size++;
}
//
// add a local cse to _lcse_list
//
void Live_LCSE::add_lcse(Exp *e) {
    LCSE *l = pool.alloc_lcse_for(e);
    //
    // "cmp imm, rght" is inserted into _array_cmp_head list
    //
    if (e->op == Exp::Compare && ((Inst_Exp*)e)->left_child()->is_imm_exp())
        insert_lcse_to_list(_array_cmp_head,MAX_CMP_LIST_SIZE,_cmp_size,l);
    else 
        insert_lcse_to_list(_list_head,MAX_LIST_SIZE,_list_size,l);
}

//
// propagate lcse from this block to succ
//
void Live_LCSE::propagate_to(Live_LCSE &succ,
                             Exp *instanceof) {
    //
	// live cse of the current bb is the initial state of succ
    // copy live cse information
    //
    LCSE *lcse;
    for (lcse = _list_head.prev(); lcse != &_list_head; lcse = lcse->prev()) {
        if (lcse->exp()->op != Exp::Instanceof || 
            lcse->exp() == instanceof) {
		    lcse->enable_lcse(); 
            succ.add_lcse(lcse->exp());
        }
    }
    for (lcse = _array_cmp_head.prev(); lcse != &_array_cmp_head; lcse = lcse->prev()) {
		lcse->enable_lcse(); 
        succ.add_lcse(lcse->exp());
    }
}

//
// copy lcse->exp->inst() to lcse->live
// then set lcse->exp->set_inst(NULL) so that lcse->exp is no longer
// a cse
//
void Live_LCSE::snap_shot_of_lcse(){
    LCSE *lcse;
    for (lcse = _list_head.next(); lcse != &_list_head; lcse = lcse->next()) 
        lcse->snap_shot_of_lcse();
    for (lcse = _array_cmp_head.next(); lcse != &_array_cmp_head; lcse = lcse->next()) 
        lcse->snap_shot_of_lcse();
}

//
// left is a immediate 
// find out if "cmp imm, rght" whose imm >= left exists
//
Inst_Exp *Live_LCSE::array_cmp(Exp *left, Exp *rght){
    assert(left->is_imm_exp());
    Imm_Operand *left_imm = (Imm_Operand*)((Operand_Exp*)left)->opnd;
    LCSE *lcse;
    for (lcse = _array_cmp_head.next(); lcse != &_array_cmp_head; 
         lcse = lcse->next()) {
        assert(((Inst_Exp*)lcse->exp())->left_child()->is_imm_exp());
        if (((Inst_Exp*)lcse->exp())->rght_child() == rght) {
            Operand_Exp *l = (Operand_Exp*)((Inst_Exp*)lcse->exp())->left_child();
            if (((Imm_Operand*)l->opnd)->imm() >= left_imm->imm())
                return (Inst_Exp*)lcse->exp();
        }
    }
    return NULL;
}
