// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/stack.cpp,v 1.2 2001/08/13 10:00:42 xhshi Exp $
//

#include "defines.h"
#include <iostream.h>
#include <stdarg.h>
#include <assert.h>
#include "code_emitter.h"
#include "operand.h"
#include "stack.h"
#include "lazy_code_selector.h"
#include "bit_vector.h"
#include "cg_prepass.h"

Operand *Stack::pop() {
	Operand *opnd = _elems[--_top];
	if (_spill_mark > _top) _spill_mark = _top;
	if (_home_mark > _top ) _home_mark  = _top;
#ifdef _CSE
	cse->pop();
#endif // _CSE
    //
    // as soon as we pop a field, we assume that the field will be used 
    // immediately.  Therefore, we make sure that the precise exception
    // ordering.
    //
    if (opnd->kind == Operand::Field)
        maintain_precise_exception();

	return opnd;
}
void Stack::pop(unsigned n) {
	_top -= n;
	if (_spill_mark > _top) _spill_mark = _top;
	if (_home_mark > _top ) _home_mark  = _top;
#ifdef _CSE
	cse->pop(n);
	cse->give_up_cse(); // no common subexpression
#endif // _CSE
}

void Stack::pop64(Operand*& opnd_lo, Operand*& opnd_hi) {
	opnd_lo = _elems[--_top];
	opnd_hi = _elems[--_top];
	if (_spill_mark > _top) _spill_mark = _top;
	if (_home_mark > _top ) _home_mark  = _top;
#ifdef _CSE
	cse->pop();
	// 
	// expression info is kept in either opnd_lo or opnd_hi
	//
	cse->pop(1);
#endif // _CSE
    //
    // as soon as we pop a field, we assume that the field will be used 
    // immediately.  Therefore, we make sure that the precise exception
    // ordering.
    //
    if (opnd_lo->kind == Operand::Field)
        maintain_precise_exception();
}

void Stack::push(Operand *opnd) {
    assert(_top < size);
	_elems[_top++] = opnd;
	opnd->ty = Operand::T32bit;
#ifdef _CSE
	// At the entry of bb, _curr_bc  is NULL
	int cse_len = (!_curr_bc)? 0 : exp_len[*_curr_bc] + 1;
	combine_cse_exp(opnd,cse_len,0);
#endif // _CSE
}
void Stack::push64(Operand *opnd_lo, Operand *opnd_hi) {
    assert(_top < size-1);
	_elems[_top++] = opnd_hi;
	_elems[_top++] = opnd_lo;
	opnd_lo->ty = Operand::T64bit_lo;
	opnd_hi->ty = Operand::T64bit_hi;
#ifdef _CSE
	// At the entry of bb, _curr_bc  is NULL
	int cse_len = (!_curr_bc)? 0 : exp_len[*_curr_bc] + 1;
	combine_cse_exp(opnd_hi,cse_len,1);
	cse->dup(); // duplicate top of cse stack
#endif // _CSE
}

#ifdef _CSE
//
// don't combine_cse_exp
//
void Stack::push_cse(Operand *opnd,int cse_len) {
	_elems[_top++] = opnd;
	opnd->ty = Operand::T32bit;
	combine_cse_exp(opnd,cse_len,0);
}

void Stack::combine_cse_exp(Operand *dst,int len,unsigned is_long) {
	CSE_Exp exp;
	if (len == 0) {
		//
		// for those exp_len = -1, there is no cse
		//
		cse->give_up_cse();
	} else 
		exp.set_exp(_curr_bc - _first_bc, len);
	cse->combine(_first_bc,&exp,dst,reg_manager.local_regs(),is_long);	
}
#endif // _CSE

int Stack::spill_one(unsigned local_regs) {
	Operand *opnd;
	//
	// The first attempt is to spill one operand that does not require
	// an extra register
	//
	for (unsigned  i= _spill_mark; i < _top; i++) {
		opnd = _elems[i];
		//
		// opnd does not have a scratch reg so there is no point to
		// spill opnd
		//
		if (!opnd->hold_local_reg(local_regs)) {
			if (i == _spill_mark) _spill_mark++;
			continue;
		}
		if (_spilling_without_extra_reg(opnd,i))
			return 1;
	}
	return _home_one_opnd_with_caller_reg(local_regs);
}

int Stack::_home_one_opnd_with_caller_reg(unsigned local_regs) {
	//
	// homing one operand to free up registers
	//
	Operand *opnd;
	for (;_spill_mark < _top;_spill_mark++) {
		opnd = _elems[_spill_mark];
		if (opnd->hold_local_reg(local_regs)) {
			switch (opnd->ty) {
			case Operand::T32bit:
				opnd->home32(emitter,reg_manager,_frame,_spill_mark);
				_elems[_spill_mark] = op_pool.nth_stack(_spill_mark);
				_elems[_spill_mark]->ty = Operand::T32bit;
				return 1;
				break;
			case Operand::T64bit_hi:
				opnd->home64_hi(emitter,reg_manager,_frame,_spill_mark);
				_elems[_spill_mark] = op_pool.nth_stack(_spill_mark);
				_elems[_spill_mark]->ty = Operand::T64bit_hi;
				_spill_mark++;
				_elems[_spill_mark]->home64_lo(emitter,reg_manager,_frame,_spill_mark);
				_elems[_spill_mark] = op_pool.nth_stack(_spill_mark);
				_elems[_spill_mark]->ty = Operand::T64bit_lo;
				return 1;
				break;
			case Operand::T64bit_lo:
				assert(0); // should never happen
			}
		}
	}
	return 0;
}
int Stack::_spilling_without_extra_reg(Operand *opnd, unsigned entry) {
	//
	// opnd has one scratch reg.  Here we try to spill opnd to free up
	// the register.   If we can spill opnd, then the stack entry is 
	// replaced with a corresponding stack operand
	//
	switch (opnd->ty) {
	case Operand::T32bit:
		if (opnd->spill32(emitter,reg_manager,_frame,entry)) {
			_elems[entry] = op_pool.nth_stack(entry);
			_elems[entry]->ty = Operand::T32bit;
			return 1;
		}
		break;
	case Operand::T64bit_hi:
		//
		// spill lo and hi at the same time
		//
		if (opnd->spill64_hi(emitter,reg_manager,_frame,entry)) {
			_elems[entry] = op_pool.nth_stack(entry);
			_elems[entry]->ty = Operand::T64bit_hi;
			// 
			// At this point, we may spill a pair of registers.  However, this is 
			// necessay because we may pop 2 64-bit operands that use all 3 registers.
			// When we try to get_reg(), we are not able to get one.
			//
			_elems[entry+1]->spill64_lo(emitter,reg_manager,_frame,entry+1);
			_elems[entry+1] = op_pool.nth_stack(entry+1);
			_elems[entry+1]->ty = Operand::T64bit_lo;
			return 1;
		}
		break;
	case Operand::T64bit_lo: // skip it because the case has been dealt in T64bit_hi 
		break;
	}
	return 0;
}

void Stack::_home_opnd(Operand* opnd, unsigned entry) {
	switch (opnd->ty) {
	case Operand::T32bit:
		opnd->free_opnd(&reg_manager);
		opnd->home32(emitter,reg_manager,_frame,entry);
		break;
	case Operand::T64bit_hi:
		opnd->home64_hi(emitter,reg_manager,_frame,entry);
		break;
	case Operand::T64bit_lo:
		opnd->free_opnd(&reg_manager);
		opnd->home64_lo(emitter,reg_manager,_frame,entry);
		break;
	default: assert(0);
	}
	_elems[entry] = op_pool.nth_stack(entry);
	_elems[entry]->ty = opnd->ty;
}

void Stack::home_all() {
	//
	// get_reg() guarantees to find one available reg.  An operand on
	// the stack may be spilled to its home location to free up a register.
	// It is okay because all operands on the stack will be moved their 
	// home location eventually.
	//
	Reg_Operand *reg = reg_manager.get_reg();
	if (reg) reg->free_opnd(&reg_manager);
	//
	// after we free_opnd, we guarantee that there is at least one register
	// available for homing
	//



    if (fp_strict_mode)
    {
       	for (unsigned i = _top; i > _home_mark ; i--) 
		    _home_opnd(_elems[i-1],i-1);
        _home_mark = _top-1;
    }
    else
    {
	    Operand *opnd;
       	for (unsigned i = _top; i > _home_mark ; i--) {
            opnd = _elems[i-1];
            _home_opnd(opnd,i-1);
            
            if (opnd->kind == Operand::Fp) {
                if (opnd->ty == Operand::T32bit ||
                    opnd->ty == Operand::T64bit_lo) {
                    fp_dec_cnt();
                }
            }
        }
        _home_mark = _top-1;
        fp_reset_stack();
    }
	reg_manager.reset_reg_value(); // for LOAD_STORE
#ifdef _CSE
	cse->reset_cse();
#endif // _CSE
}

void Stack::call_home(unsigned n_args) {
	//
	// make sure that we have one reg for homing.  This may not be the best
	// choice.  The reasoning is that we may spill an outgoing argument that is
	// in a caller-save register when some other memory operands can be spilled.
	// However, it is the simplest one.
	//
	Reg_Operand *reg = reg_manager.get_reg();
	if (reg) reg->free_opnd(&reg_manager);
	//
	// Homing operands that can be potentially killed by a call
	// up to the first argument
	//
	Call_BV_List_Element *call_info = prepass.find_gc_sites_info(_curr_bc,_hint_for_call_site);
	unsigned been_searched = 0;
	Operand *opnd;
	unsigned i;

    if (fp_strict_mode)
    {
	    for ( i = _home_mark;i < _top - n_args;i++) {
		    opnd = _elems[i];
		    if (opnd->is_aliased_across_call()) { 
			    _home_opnd(opnd,i);
			    if (i == _home_mark) _home_mark++;
		    }
		    else if (call_info && opnd->is_callee_reg() && 
			    i < call_info->elem->numbits() && call_info->elem->is_set(i)) {
				    // don't enumerate opnd during GC
				    call_info->elem->clear(i);
				    call_info->stack_ref_in_local_callee |= (1<<((Reg_Operand*)opnd)->opnd.reg_no());
		    } 
	    }
    }
    else
    {
	    int save_home_mark = _home_mark;

        for ( i = _home_mark;i < _top - n_args;i++) {
            opnd = _elems[i];
            if (opnd->is_aliased_across_call()) { 
                if (opnd->kind != Operand::Fp) {
                    _home_opnd(opnd,i);
                    if (i == _home_mark) _home_mark++;
                }
            }
            else if (call_info && opnd->is_callee_reg() && 
                i < call_info->elem->numbits() && call_info->elem->is_set(i)) {
                // don't enumerate opnd during GC
                call_info->elem->clear(i);
                call_info->stack_ref_in_local_callee |= (1<<((Reg_Operand*)opnd)->opnd.reg_no());
            }
        }
        
        // spill the arguments.. no choice since args are
        // on fp stack
        for (int k = _top-1; k >= save_home_mark; k--) {
            opnd = _elems[k];
            if (opnd->kind == Operand::Fp) {
                _home_opnd(opnd,k);
                //  We spill starting from T64bit_lo
                //  We also spill 32 bit Fp operands
                //  See _home_opnd and operand.h
                //  Hence, the check..
                
                if (opnd->ty == Operand::T32bit ||
                    opnd->ty == Operand::T64bit_lo) {
                    fp_dec_cnt();
                }
            }
        }
        
        fp_reset_stack();
    }
	reg_manager.reset_reg_value(); // for LOAD_STORE
#ifdef _CSE
	cse->reset_cse();
#endif // _CSE

}
//
// the routine homes all operands that contain r.  The reasoning is that we may
// want to use one paricular register, e.g. eax_reg for returning.
//
void Stack::spill_opnds_contain(X86_Reg_No r) {
	// 
	// if r is free, then there is no operand containing r 
	//
#ifdef _CSE
	cse->reset_reg_cse(r);
#endif // _CSE
	if (reg_manager.is_free(r)) return;
	Operand *opnd;
	for (unsigned i = _home_mark; i < _top; i++) {
		opnd = _elems[i];
		if (opnd->contain(r)) {
			_home_opnd(opnd,i);
			if (i == _home_mark) _home_mark++;
		}
	}
	reg_manager.reset_reg_value(r); // for LOAD_STORE
}

// bytecode : iload_2
// bytecode : iinc 2 -1 
// The iinc bytecode decrements local var_2 by 1.  At this point, var_2 may be
// on the stack.  Hence we cannot delay code generation for var_2, otherwise,
// var_2 on the stack gets the decremented value.
//
void Stack::no_laziness(Frame& frame, unsigned var_no,
						X86_Reg_No reg_lo, X86_Reg_No reg_hi) {
	//
	// for long and double, we only need to check reg_lo because we assign
	// a pair of regs to a long/double
	//
	Operand *opnd;
	if (reg_lo != n_reg) { 
		for (unsigned i = _spill_mark; i < _top; i++) {
			opnd = _elems[i];
			if (opnd->kind != Operand::Reg) continue;
			X86_Reg_No reg =  ((Reg_Operand*)opnd)->opnd.reg_no();
			if (reg == reg_lo || reg == reg_hi) {
#ifdef _CSE
				cse->reset_cse_ith_entry(i);
#endif // _CSE
				_home_opnd(opnd,i);
				if (i == _spill_mark) _spill_mark++;
				if (i == _home_mark)  _home_mark++;
			}
		}
	} else {
		int var_off_lo = frame.var_offset(var_no);
		int var_off_hi = frame.var_offset(var_no+1);
		for (unsigned i = _spill_mark; i < _top; i++) {
			opnd = _elems[i];
			if (opnd->kind != Operand::Var) continue;
			int off = ((Mem_Var_Operand*)opnd)->opnd.off();
			if (var_off_lo == off || var_off_hi == off) {
#ifdef _CSE
				cse->reset_cse_ith_entry(i);
#endif // _CSE
				_home_opnd(opnd,i);
				if (i == _spill_mark) _spill_mark++;
				if (i == _home_mark)  _home_mark++;
			}
		}
	}
}
void Stack::no_laziness(Frame& frame, unsigned var_no,X86_Reg_No reg) 
{
	Operand *opnd;
	if (reg != n_reg) {
		for (unsigned i = _spill_mark; i < _top; i++) {
			opnd = _elems[i];
			if (opnd->kind == Operand::Reg &&
				((Reg_Operand*)opnd)->opnd.reg_no() == reg) {
#ifdef _CSE
					cse->reset_cse_ith_entry(i);
#endif // _CSE
				_home_opnd(opnd,i);
				if (i == _spill_mark) _spill_mark++;
				if (i == _home_mark)  _home_mark++;
			}
		}
	} else {
		int var_off = frame.var_offset(var_no);
		for (unsigned i = _spill_mark; i < _top; i++) {
			opnd = _elems[i];
			if (opnd->kind == Operand::Var && 
				((Mem_Var_Operand*)opnd)->opnd.off() == var_off) {
#ifdef _CSE
					cse->reset_cse_ith_entry(i);
#endif // _CSE
					_home_opnd(opnd,i);
					if (i == _spill_mark) _spill_mark++;
					if (i == _home_mark)  _home_mark++;
			}
		}
	}


}

// returns 0 and increments fp_cnt if there is still space on 
// the floating point stack
// otherwise, pop the top of stack and spill it 
// and return 1; also decrement the fp_cnt
bool Stack::fp_check_stack(
    unsigned is_dbl,
    Pre_Alloc_Operand_Pool &op_pool, 
    Code_Emitter &emitter) {

    if (fp_get_cnt() >= MAX_FP_STACK) {
        assert(0);
        Stack_Operand *dst = op_pool.nth_stack(depth());
        push(dst);
        emitter.emit_fst(dst->mem_opnd(),is_dbl,1);
        fp_dec_cnt();
        return TRUE;
    }
    else {
        fp_inc_cnt();
        return FALSE;
    }
}

//
// We are about to generate instruction that might cause exception.
// We ought to home all operands on the stack that might throw exceptions
// so as to maintain the precise exception ordering.
//
void Stack::maintain_precise_exception() {
    unsigned n_field = 0;
    bool need_reg  = false;
    unsigned i;
	for (i = _home_mark;i < _top;i++) {
        //
        // only field could cause problems
        //
        if (_elems[i]->kind == Operand::Field &&
            ((Field_Operand*)_elems[i])->may_throw_null_ptr_excp()) {
            n_field++;
            //
            // for 64-bit operand or opnd that has no local regs, we
            // need to reserve one reg for homing.
            //
            if (_elems[i]->ty == Operand::T64bit_hi ||
                !_elems[i]->hold_local_reg(reg_manager.local_regs()))
                need_reg = true;
        }
    }

    if (n_field == 0) return; // return if not found

    //
    // make sure one reg is available
    //
    Reg_Operand *reg = reg_manager.get_reg();
    reg->free_opnd(&reg_manager);

	for (i = _home_mark;i < _top;i++) {
        //
        // skip non field operands and fields whose bases are not null
        //
        if (_elems[i]->kind != Operand::Field ||
            !((Field_Operand*)_elems[i])->may_throw_null_ptr_excp()) 
            continue;

        n_field--;
        Operand *opnd = _elems[i];
        if (opnd->ty == Operand::T64bit_hi || 
            opnd->ty == Operand::T64bit_lo ||
            (n_field != 0 && !opnd->hold_local_reg(reg_manager.local_regs()))) {
            _home_opnd(opnd,i);
        }
        else {
            //
            // if opnd contains local reg, then we can use it to hold the
            // value of the operand.
            //
            opnd->free_opnd(&reg_manager);
            //
            // get_reg won't spill opnd because opnd either has freed regs or
            // does not have local_regs.
            //
            reg = reg_manager.get_reg();
            //
            // mov opnd's value to reg
            //
            opnd->emit_mov_to_reg(emitter,&reg->opnd);
            _elems[i]     = reg;
            _elems[i]->ty = Operand::T32bit;

        }
    }
}
