// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/register_manager.cpp,v 1.2 2001/08/13 10:00:42 xhshi Exp $
//

#include "defines.h"
#include <iostream.h>
#include <stdarg.h>
#include <assert.h>
#include "code_emitter.h"
#include "operand.h"
#include "register_manager.h"
#include "stack.h"

Register_Manager::Register_Manager(Mem_Manager& m) {
	for (unsigned i=0; i < n_reg; i++) {
		_x86_regs[i] = new(m) Reg_Operand((X86_Reg_No)i);
	}
	reset();
}

void Register_Manager::reset() {
	_next_free = ecx_reg;
	_free_bv = ALL_X86_CALLER_REGS;
	_available_local_regs = ALL_X86_CALLER_REGS;
	reset_reg_info();
}

void Register_Manager::update_reg_info(X86_Reg_No no) {
  reset_reg_value(no); // for LOAD_STORE
#ifndef NO_BOUNDS_CHECKING
	clr_newarray_in_reg(no);
#endif // NO_BOUNDS_CHECKING
#ifdef _CSE
	_stack->cse->reset_reg_cse(no);
#endif // _CSE
	_x86_regs[no]->clear_ref();
}
//
// find one register that is set in mask
//
Reg_Operand *Register_Manager::get_reg(unsigned mask) {
	if (!(_free_bv & mask)) {
		//
		// no free registers; spill a register holding stack value
		//
		if (_stack->spill_one(_available_local_regs & mask) == 0)
			return NULL;
	}
	//
	// a free register exists; find one using the next_free hint
	//
	while (!(_free_bv & (1 << _next_free) & mask)) {
		_next_free++;
		if (_next_free == n_reg)
			_next_free = 0;
	}
	Reg_Operand *reg = _x86_regs[_next_free];
	update_reg_info((X86_Reg_No)_next_free);
	// mark reg as busy (i.e., not free)
	//
	_free_bv &= ~(1 << _next_free);
	//
	// increment hint to next register
	//
	if (++_next_free == n_reg)
		_next_free = 0;
	return reg;
}

Reg_Operand *Register_Manager::get_reg(X86_Reg_No r) {
	//
	// not scratch regs, go ahead and return it
	//
	if (!(_available_local_regs & (1<<r))) {
		_x86_regs[r]->clear_ref();
		return _x86_regs[r];
	}
	assert(is_free(r));
	//
	// mark reg as busy (i.e., not free)
	//
	_free_bv &= ~(1 << r);

	update_reg_info(r);
	return _x86_regs[r];
}

#ifdef _CSE
Reg_Operand *Register_Manager::hold_cse_reg(X86_Reg_No r) {
	//
	// mark reg as busy (i.e., not free)
	//
	_free_bv &= ~(1 << r);
	return _x86_regs[r];
}
#endif // _CSE

void Register_Manager::push_reg(X86_Reg_No reg_no) {
	if (!(_available_local_regs & (1<<reg_no))) 
		return;
	//
	// mark register as free
	//
	_free_bv |= (1<<reg_no);
	//
	// next_free hint to reg so that reg can be
	// found without iterating thru free_bv
	//
	_next_free = reg_no;
}

void Register_Manager::free_reg(X86_Reg_No reg_no) {
	if (!(_available_local_regs & (1<<reg_no))) 
		return;
	//
	// the difference between push and free reg is that push_reg 
	// encourages the freed register to be used immediately again
	// by the next get_reg.  free_reg does not.
	//
	//
	// mark register as free
	//
	_free_bv |= (1<<reg_no);
	//
	// increment next_free hint to the register after reg so 
	// that its value sticks around for a while
	//
	_next_free = reg_no;
	if (++_next_free == n_reg) 
		_next_free = 0;
}

//
// find one callee-save reg that is not set in reg_use_bv
//
X86_Reg_No Register_Manager::find_one_available_callee(unsigned reg_use_bv){
	// look for ebx, esi, edi
	if (!(reg_use_bv & (1 << ebx_reg)))
		return ebx_reg;
	if (!(reg_use_bv & (1 << esi_reg)))
		return esi_reg;
	if (!(reg_use_bv & (1 << edi_reg)))
		return edi_reg;
	if (!(reg_use_bv & (1 << ebp_reg)))
		return ebp_reg;
	assert(0); return ebx_reg;
}
//
// make sure all caller-save regs are available at this point
//
void Register_Manager::assert_all_available() {
	assert(_free_bv == _available_local_regs);
}

// for LOAD_STORE
void Register_Manager::reg_value(Operand *src, int index) {
	assert(index >= 0);
	if (src->kind != Operand::Reg) return;
	X86_Reg_No reg_no = ((Reg_Operand*)src)->opnd.reg_no();

	if (!(_available_local_regs & (1<<reg_no))) return;

	reset_store(index);
	_reg_value[reg_no] = index;
}

// for LOAD_STORE
X86_Reg_No Register_Manager::find_reg_value(int index) {
	for (int i = 0; i < n_reg; i++)
		if (_reg_value[i] == index) {
			return (X86_Reg_No)i;
		}
	return n_reg;
}

// for LOAD_STORE
void Register_Manager::reset_store(int index) {
	for (int i = 0; i < n_reg; i++)
		if (_reg_value[i] == index) {
			_reg_value[i] = -1;
			return;
		}
}
