// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/cg_dup.cpp,v 1.2 2001/08/13 09:59:51 xhshi Exp $
//

#include "defines.h"
#include <assert.h>
#include <iostream.h>
#include "code_emitter.h"
#include "stack.h"
#include "operand.h"
#include "lazy_code_selector.h"
#include "cg_dup.h"

#ifndef NO_BOUNDS_CHECKING
#include "bounds_checking.h"
#endif // NO_BOUNDS_CHECKING

//
// mov a value from one spill loc to another
//
void gen_update_stack_pos(Code_Emitter& emitter,
						  Pre_Alloc_Operand_Pool& op_pool,
						  Reg_Operand *reg,  // scratch reg to hold temp value
						  Operand*& from,int depth) {
	assert(from->kind == Operand::Stk);
	Stack_Operand *to_spill_loc = op_pool.nth_stack(depth);
	from->emit_mov_to_reg(emitter,&reg->opnd);
	reg->emit_mov_to_mem(emitter,&to_spill_loc->opnd);
	to_spill_loc->ty = from->ty;
	from = to_spill_loc;
}

//
// duplicate the top element of the fp stack
//
static Operand *dup_fp_opnd(Mem_Manager& mm,
                            Code_Emitter& emitter,
                            Stack& stack,
                            Pre_Alloc_Operand_Pool& op_pool,
                            bool is_double)
{
    Fp_Operand *fp_res = new (mm) Fp_Operand(is_double);
    fp_res->fpstack_cnt = stack.fp_get_cnt();
    if (stack.fp_check_stack(is_double, op_pool, emitter))
        stack.fp_inc_cnt();
    emitter.emit_fld(0);
    return fp_res;
}

void gen_dup(Mem_Manager&  mem_manager,
             Code_Emitter& emitter, 
             Stack& stack,
             Pre_Alloc_Operand_Pool& op_pool
#ifndef NO_BOUNDS_CHECKING
             , Bounds_Checking& bounds
#endif // NO_BOUNDS_CHECKING
             ) {
	Operand *dup_opnd;
	Reg_Operand *reg;

	reg = stack.reg_manager.get_reg();
	Operand *src = stack.pop();
    if (src->kind == Operand::Fp) {
        assert(!((Fp_Operand*)src)->is_double);
        dup_opnd = dup_fp_opnd(mem_manager,emitter,stack,op_pool,false);
    } else {
	    src->emit_mov_to_reg(emitter,&reg->opnd);
	    dup_opnd = reg;
	    if (src->kind == Operand::Stk) {
		    dup_opnd = src;
		    src = reg;
	    }
    }
    // the following 2 lines were originally at the very end. XXX-JMS
	stack.push(dup_opnd);
	stack.push(src);
#ifndef NO_BOUNDS_CHECKING
	if (src->is_reg()) {
		X86_Reg_No src_no = ((Reg_Operand*)src)->opnd.reg_no();
		X86_Reg_No reg_no = reg->opnd.reg_no();
		if (stack.reg_manager.is_newarray_in_reg(src_no)) {
			//
			// propagate newarray info from src to reg
			//
			stack.reg_manager.set_newarray_in_reg(reg_no);
			bounds.propagate(reg_no,src_no);
		}
	}
#endif // NO_BOUNDS_CHECKING
}
//
// duplicate top operand stack word and push three down
// stack:  ... ,word2, word1 ==> ..., word1, word2, word1
//
void gen_dup_x1(Mem_Manager&            mem_manager, 
                Code_Emitter&           emitter, 
                Stack&                  stack,
				Pre_Alloc_Operand_Pool& op_pool) {
    //
    // if word1 and word2 are fp operands, the code for duplicating is 
    // complicated.  We call home_all() if word1 and word2 are fp operands.
    //
    if (stack.nth(0)->kind == Operand::Fp &&
        stack.nth(1)->kind == Operand::Fp)
        stack.home_all(); // force fp operands to be spilled back to memory

    Operand *word1, *word2, *dup_opnd;
	Reg_Operand *reg, *word2_reg = NULL;
	word1 = stack.pop();
	//
	// if word2 is a stack operand, then we need a register to mov word2
	// spill loc to another.  We grap one reg so that we are sure that
	// there is a reg when we need one.
	//
	if (stack.top()->kind == Operand::Stk) 
		word2_reg = stack.reg_manager.get_reg();

	int w1_depth = stack.depth();
	switch (word1->kind) {
	case Operand::Stk:
		//
		// Because the stack positions of operands change after perofrming 
		// dup_x1, we update their stack positions
		//
		reg = stack.reg_manager.get_reg();
		gen_update_stack_pos(emitter,op_pool,reg,word1,w1_depth+1);
		dup_opnd = reg;
		break;
	case Operand::Imm:
	case Operand::Static:
	case Operand::Var:
		dup_opnd = word1;
		break;
	case Operand::Reg:
	case Operand::Field:
	case Operand::Array:
		reg = stack.reg_manager.get_reg();
		if (reg != NULL) {
			word1->emit_mov_to_reg(emitter,&reg->opnd);
			//
			// don't swap word2 and word1, i.e. dup_opnd = word1; word1 = reg;
			// because word2 may overwrite word1's (e.g. field access) content.
			//
			dup_opnd = reg;
		} else {
			assert(word2_reg);
			Stack_Operand *loc = op_pool.nth_stack(w1_depth+1);
			word1->emit_mov_to_reg(emitter,&word2_reg->opnd);
			word2_reg->emit_mov_to_mem(emitter,&loc->opnd);
			dup_opnd = word1;
			word1 = loc;
		}
		break;
    case Operand::Fp:
        {
        assert(!((Fp_Operand*)word1)->is_double);
        Operand *fp_res = dup_fp_opnd(mem_manager,emitter,stack,op_pool,false);
        dup_opnd = word1;
        word1 = fp_res;
        }
        break;
	default: assert(0);
	}
	word2 = stack.pop();
	if (word2->kind == Operand::Stk) {
		gen_update_stack_pos(emitter,op_pool,word2_reg,word2,w1_depth);
		word2_reg->free_opnd(&stack.reg_manager);
	}
	stack.push(dup_opnd);
	stack.push(word2);
	stack.push(word1);
}

//
// duplicate top operand stack word and push three down
// stack:  ... ,word3, word2, word1 ==> ..., word1, word3, word2, word1
//
void gen_dup_x2(Code_Emitter& emitter, Stack& stack,
				Pre_Alloc_Operand_Pool& op_pool) {
    //
    // if word1 and word2/3 are fp operands, the code for duplicating is 
    // complicated.  We call home_all() if word1 is a fp operand.
    //
    if (stack.top()->kind == Operand::Fp)
        stack.home_all();

	Reg_Operand *reg = stack.reg_manager.get_reg();
	Operand *word1 = stack.pop();
	Operand *word2 = stack.pop();
	Operand *word3 = stack.pop();
	//
	// Because the stack positions of operands change after performing 
	// dup_x2, we update their stack positions
	//
	int curr_depth = stack.depth();
	if (word1->kind == Operand::Stk)
		gen_update_stack_pos(emitter,op_pool,reg,word1,curr_depth+3);
	if (word2->kind == Operand::Stk)
		gen_update_stack_pos(emitter,op_pool,reg,word2,curr_depth+2);
	if (word3->kind == Operand::Stk)
		gen_update_stack_pos(emitter,op_pool,reg,word3,curr_depth+1);
	//
	// duplicate word1
	//
	word1->emit_mov_to_reg(emitter,&reg->opnd);
	stack.push(reg);
	if (word3->ty == Operand::T32bit) {
		stack.push(word3);
		stack.push(word2);
	} else 
		stack.push64(word2,word3);
	stack.push(word1);
}

//
// duplicate top 2 operand stack words 
// stack:  ... ,word2, word1 ==> ..., word2, word1, word2, word1
//
void gen_dup2(Code_Emitter& emitter, Stack& stack,
			  Pre_Alloc_Operand_Pool& op_pool) {
    //
    // Peek the top two elements that are about to be duplicated.
    // If any of them is a fp operand, then using home_all to spill them
    // back to memory.
    //
    if (stack.nth(0)->kind == Operand::Fp ||
        stack.nth(1)->kind == Operand::Fp)
        stack.home_all();

	Reg_Operand *reg1, *reg2;
	Operand *word1 = stack.pop();
	Operand *word2 = stack.pop();
	int curr_depth = stack.depth();
	Stack_Operand *dup1_loc = op_pool.nth_stack(curr_depth+3);
	Stack_Operand *dup2_loc = op_pool.nth_stack(curr_depth+2);
	Stack_Operand *word1_loc = op_pool.nth_stack(curr_depth+1);
	Stack_Operand *word2_loc = op_pool.nth_stack(curr_depth);
	Operand *dup1 = NULL, *dup2 = NULL;
	if (word1->ty == Operand::T32bit) {
		// we try to load word1 & word2 into regs
		if (word1->kind != Operand::Imm &&
            word1->kind != Operand::Static)
			reg1 = stack.reg_manager.get_reg();
		else 
			dup1 = word1;
		if (word2->kind != Operand::Imm &&
            word2->kind != Operand::Static)
			reg2 = stack.reg_manager.get_reg();
		else 
			dup2 = word2;
		//
		// deal with word2 first
		//
		if (!dup2) {
			if (reg2) {
				word2->emit_mov_to_reg(emitter,&reg2->opnd);
				if (word2->kind == Operand::Reg) 
					dup2 = reg2;
                else {
					emitter.emit_mov(&dup2_loc->opnd,&reg2->opnd);
					dup2 = dup2_loc;
                    //
                    // reg2 is no longer needed because dup2 is dup2_loc.
                    //
                    reg2->free_opnd(&stack.reg_manager);
                } 
			} else {
				assert(!dup1);  // word2->kind != Operand::Imm
				// check if we can use reg1 to move value
				if (!reg1) {
				    word2->free_opnd(&stack.reg_manager);
					reg2 = stack.reg_manager.get_reg();
					assert(reg2);
                    reg2->free_opnd(&stack.reg_manager);
				    word2 = word2_loc;
                } else 
                    // reg1 cannot be freed at this point because dup1 may 
                    // still need reg1 to hold the value
                    reg2 = reg1;  
				word2->emit_mov_to_reg(emitter,&reg2->opnd);
                //
                // Because word2 is freed in this case, we need to move word2 
                // to word2_loc.
                //
				if (!reg1) 
                    emitter.emit_mov(&word2_loc->opnd,&reg2->opnd);
                emitter.emit_mov(&dup2_loc->opnd,&reg2->opnd);
				dup2 = dup2_loc;
            }
        }
		//
		// reg1 must not be NULL
		// (a). if word2 is imm, then at most word1 takes two regs (array operand). 
		//      Hence, there is at least one reg left.
		// (b). if word2 is not imm & reg2 is not NULL, then reg1 is not NULL as well
		//      because we get reg1 before reg2
		// (c). if word2 is not imm & reg2 is NULL, then we make sure that reg1 is not NULL by
		//      spill word2 if necessary
		//
		if (!dup1) {
			assert(reg1);
			word1->emit_mov_to_reg(emitter,&reg1->opnd);
			if (word1->kind == Operand::Reg) 
                dup1 = reg1;
            else {
				emitter.emit_mov(&dup1_loc->opnd,&reg1->opnd);
				dup1 = dup1_loc;
                reg1->free_opnd(&stack.reg_manager); // no longer need reg1
			}
		}
		stack.push(word2);
		stack.push(word1);
		stack.push(dup2);
		stack.push(dup1);
	} else { // long
		Reg_Operand *reg;
		if (word1->kind == Operand::Imm) {
			dup1 = word1;
			dup2 = word2;
		} else {
			reg = stack.reg_manager.get_reg();
			if (word1->kind != Operand::Stk) {
				word1->emit_mov_to_reg(emitter,&reg->opnd);
				emitter.emit_mov(&word1_loc->opnd,&reg->opnd);
				word2->emit_mov_to_reg(emitter,&reg->opnd);
				emitter.emit_mov(&word2_loc->opnd,&reg->opnd);
				dup1 = word1; dup2 = word2;
			} else {
				word1->emit_mov_to_reg(emitter,&reg->opnd);
				emitter.emit_mov(&dup1_loc->opnd,&reg->opnd);
				word2->emit_mov_to_reg(emitter,&reg->opnd);
				emitter.emit_mov(&dup2_loc->opnd,&reg->opnd);
				dup1 = dup1_loc; dup2 = dup2_loc;
			}
			word1 = word1_loc; word2 = word2_loc;
			reg->free_opnd(&stack.reg_manager);
		}
		stack.push64(word1,word2);
		stack.push64(dup1,dup2);
	}
}

//
// duplicate top 2 operand stack words and push three down
// stack:  ... ,word3, word2, word1 ==> ..., word2, word1, word3, word2, word1
//
void gen_dup2_x1(Code_Emitter& emitter, Stack& stack,
				Pre_Alloc_Operand_Pool& op_pool) {
	stack.home_all();
	Reg_Operand *reg1 = stack.reg_manager.get_reg();
	Reg_Operand *reg2 = stack.reg_manager.get_reg();
	Reg_Operand *reg3 = stack.reg_manager.get_reg();
	assert(reg1 && reg2 && reg3);
	Operand *word1 = stack.pop();
	Operand *word2 = stack.pop();
	Operand *word3 = stack.pop();
	//
	// Because the stack positions of operands change after perofrming 
	// dup_x1, we update their stack positions
	//
	int curr_depth = stack.depth();
	gen_update_stack_pos(emitter,op_pool,reg1,word1,curr_depth+4);
	gen_update_stack_pos(emitter,op_pool,reg2,word2,curr_depth+3);
	word3->emit_mov_to_reg(emitter,&reg3->opnd);
	if (word1->ty == Operand::T32bit) {
		stack.push(reg2);
		stack.push(reg1);
		stack.push(reg3);
		stack.push(word2);
		stack.push(word1);
	} else {
		stack.push64(reg1,reg2);
		stack.push(reg3);
		stack.push64(word1,word2);
	}
}

//
// duplicate top 2 operand stack words and push 4 down
// stack:  ... ,word4, word3, word2, word1 ==> ..., word2,word1,word4,word3,word2 word1
//
void gen_dup2_x2(Code_Emitter& emitter, Stack& stack,
				Pre_Alloc_Operand_Pool& op_pool) {
	stack.home_all();
	Reg_Operand *reg1 = stack.reg_manager.get_reg();
	Reg_Operand *reg2 = stack.reg_manager.get_reg();
	Reg_Operand *reg3 = stack.reg_manager.get_reg();
	assert(reg1 && reg2 && reg3);
	Operand *word1 = stack.pop();
	Operand *word2 = stack.pop();
	Operand *word3 = stack.pop();
	Operand *word4 = stack.pop();
	//
	// Because the stack positions of operands change after perofrming 
	// dup2_x2, we update their stack positions
	//
	int curr_depth = stack.depth();
	gen_update_stack_pos(emitter,op_pool,reg1,word1,curr_depth+5);
	gen_update_stack_pos(emitter,op_pool,reg2,word2,curr_depth+4);
	gen_update_stack_pos(emitter,op_pool,reg3,word3,curr_depth+3);
	gen_update_stack_pos(emitter,op_pool,reg3,word4,curr_depth+2);
	if (word1->ty == Operand::T32bit) {
		stack.push(reg2);
		stack.push(reg1);
		if (word3->ty == Operand::T32bit) {
			stack.push(word4);
			stack.push(word3);
		} else
			stack.push64(word3,word4);
		stack.push(word2);
		stack.push(word1);
	} else {
		stack.push64(reg1,reg2);
		if (word3->ty == Operand::T32bit) {
			stack.push(word4);
			stack.push(word3);
		} else
			stack.push64(word3,word4);
		stack.push64(word1,word2);
	}
	reg3->free_opnd(&stack.reg_manager);
}
