/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.server.uihandler;

import java.util.concurrent.TimeoutException;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import org.netbeans.modules.exceptions.entity.Logfile;
import java.util.concurrent.ExecutionException;
import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.persistence.EntityManager;
import javax.persistence.Query;
import javax.servlet.http.HttpSession;
import javax.servlet.jsp.PageContext;
import org.netbeans.lib.uihandler.LogRecords;
import org.netbeans.modules.exceptions.entity.LogfileParsed;
import org.netbeans.modules.exceptions.utils.PersistenceUtils;
import org.netbeans.server.uihandler.LogsManager.SessionInfo;
import org.openide.util.Exceptions;
import org.openide.util.Lookup;
import org.openide.util.NbCollections;

/**
 *
 * @author Jaroslav Tulach
 */
public class LogsManager extends Object implements Runnable {
    static final Logger LOG = Logger.getLogger(LogsManager.class.getName());
    private static Object logInfoLock = new Object();
    
    private File dir;
    private transient Future<?> initTask;
    private transient boolean closed;
    
    /** statistics to work on and their global values */
    private List<Value> statistics;
    
    /** flag to mark LogsManager as misconfigured */
    private boolean misconfigured;

    /** how many data has already been initialized */
    private List<Integer> initCounts = Collections.nCopies(2, 0);
    
    /** executor to use for changing data */
    private static final ExecutorService EXEC = Executors.newSingleThreadScheduledExecutor();
    
    private LogsManager(File userDir) {
        String msg = "Creating manager for " + userDir;
        LOG.log(Level.FINEST, msg, new Exception(msg));
        DEFAULT = this;
        this.dir = userDir;
        this.statistics = new ArrayList<LogsManager.Value>();
        for (Statistics<?> s : Lookup.getDefault().lookupAll(Statistics.class)) {
            addStatisticsData(this.statistics, s);
        }
        this.initTask = EXEC.submit(this);
    }
    /** For use from tests 
     * @param userDir the dir with log files
     * @return new instance of manager
     */
    static LogsManager createManager(File userDir) {
        return new LogsManager(userDir);
    }
    
    private static LogsManager DEFAULT;
    
    /** Getter for the default LogsManager. Location of directories it shall operate
     * on is looked up as "java:comp/env/uilogger/dir" from initial naming context.
     *
     * @return an instance of default manager
     * @throws IllegalStateException if the directory cannot be obtained
     */
    public static synchronized LogsManager getDefault() {
        if (DEFAULT == null) {
            String dir = Utils.getVariable("dir", String.class); // NOI18N
            if (dir == null) {
                try {
                    java.io.File tmp = java.io.File.createTempFile("uigestures", ".dir");
                    tmp.delete();
                    tmp.mkdirs();
                    if (tmp.isDirectory()) {
                        LogsManager lm = new LogsManager(tmp);
                        assert DEFAULT == lm;
                        DEFAULT.misconfigured = true;
                        return DEFAULT;
                    }
                } catch (IOException ex) {
                    Exceptions.printStackTrace(ex);
                }
                return new LogsManager(null);
            }
            DEFAULT = createManager(new java.io.File(dir));
        }
        return DEFAULT;
    }

    final void verify() {
        if (dir == null) {
            LOG.warning("Specify dir attribute, otherwise the server cannot work");
            return;
        }
        
        LOG.log(Level.FINE, "Checking status of {0}", dir);

        EntityManager em = PersistenceUtils.getInstance().createEntityManager();
        
        File[] files = dir.listFiles();
        Collections.shuffle(Arrays.asList(files));
        int cnt = 0;
        for (File f : files) {
            Integer[] counts = { cnt++, files.length };
            
            this.initCounts = Arrays.asList(counts);
            if (closed || DEFAULT != this) {
                LOG.log(Level.INFO, "Stopping processing at {0} files", cnt);
                break;
            }
            
            
            Logfile log = getLogInfo(f.getName());
            assert log != null;
            List<Value> recompute = statisticsToRecompute(log, em);
            if (recompute.isEmpty()) {
                LOG.log(Level.FINER, "File {0} is up to date.", f);
            } else {
                if (LOG.isLoggable(Level.FINER)) {
                    StringBuilder sb = new StringBuilder();
                    sb.append("File ");
                    sb.append(f.getPath());
                    sb.append(" needs reparse for ");
                    String sep = "";
                    for (Value<?> value : recompute) {
                        sb.append(sep);
                        sb.append(value.statistics.name);
                        sep = ", ";
                    }
                    LOG.finer(sb.toString());
                }
                handleAddNewLog(f, recompute, false, em);
            }
            

            LOG.log(Level.FINE, "Recomputing {0}", log.getFileName());
            for (Value<?> v : statistics) {
                addDataForLog(v, log, em);
            }
            // hopefully this will prevent the OutOfMemoryError
            em.clear();
        }

        this.initCounts = Collections.nCopies(2, getNumberOfLogs());
        
        LOG.log(Level.INFO, "Sessions created for {0}", dir);
        em.close();
        
    }
    
    private List<Value> statisticsToRecompute(Logfile log, EntityManager em) {
        List<Value> recompute = new ArrayList<Value>();
        
        for (Value<?> v : statistics) {
            boolean ok = false;
            int cnt = 0;
            Query q = em.createNamedQuery("LogfileParsed.findByStatistic"). // NOI18N
                setParameter("logfile", log). // NOI18N
                setParameter("statistic", v.statistics.name); // NOI18N
            for (Object record : q.getResultList()) {
                LogfileParsed p = (LogfileParsed)record;
                cnt++;
                if (p.getRevision() == v.statistics.revision) {
                    ok = true;
                    break;
                } else {
                    LOG.finer("  need reparse: " + v.statistics.name + " log: " + 
                        log.getFileName() + " stored: " + p.getRevision() + 
                        " != " + v.statistics.revision
                    );
                }
            }
            if (!ok) {
                if (cnt == 0) {
                    LOG.finer("  need reparse: " + v.statistics.name + " log: " + 
                        log.getFileName() + " no stored info"
                    );
                }
                recompute.add(v);
            }
        }
        return recompute;
    }
    
    /** does initialization */
    public void run() {
        long now = System.currentTimeMillis();
        verify();
        long diff = System.currentTimeMillis() - now;
        Logger.getLogger("PERFORMANCE").log(Level.SEVERE, "INICIALIZATION TOOK: " + diff);
    }
    
    private static String getIdSes(String fileName){
        String[] idSes = fileName.split("\\.");
        return idSes[0];
    }
    
    private static int getSessionNumber(String fileName){
        String[] idSes = fileName.split("\\.");
            /* ses == 0 - a file on disk without suffix like: NB2123128636
             ses != 0 - a file on disk without suffix like: NB2123128636.x
             * where ses = x+1
             */
        return idSes.length == 1 ? 0 : Integer.parseInt(idSes[1])+1;
    }
    
    private void handleAddNewLog(
        File f,
        List<Value> statistics,
        boolean initialParse,
        EntityManager em
    ) {
        boolean active = em.getTransaction().isActive();
        try {
            if (!active) {
                em.getTransaction().begin();
            }
            String idSes = getIdSes(f.getName());
            int ses = getSessionNumber(f.getName());
            
            LOG.log(Level.FINE, "handleAddNewLog for {0}", f);
            
            SessionInfo info = new SessionInfo(idSes, ses, f, getLogInfo(f.getName()), statistics);
            
            for (Value<?> v : info.getValues()) {
                finishUploadOpen(info, v, initialParse);
            }
    
            if (initialParse) {
                info.addValues(statistics);
            }

            LOG.log(Level.FINE, "Persisting data for {0}", f);
            
            info.persist(em);
            
            LOG.log(Level.FINE, "Finished processing {0}", f);
            
            if (!active) {
                em.getTransaction().commit();
                // hopefully this will prevent the OutOfMemoryError
                em.clear();
            }
        } catch (Exception ex) {
            LogRecord rec = new LogRecord(Level.WARNING, "Cannot process {0}");
            rec.setThrown(ex);
            rec.setParameters(new Object[] { f.getPath() });
            LOG.log(rec);
        }
    }
    
    private static <Data> void finishUploadOpen(SessionInfo info, Value<Data> v, boolean initialParse) {
        v.value = v.statistics.finishSessionUpload(info.id, info.session, initialParse, v.value);
    }
    
    /** Close the LogsManager, stop parsing, etc. get ready for shutdown.
     */
    public void close() {
        LOG.info("Shutdown");
        this.closed = true;
        boolean b = this.initTask.cancel(true);
        try {
            this.initTask.get(10, TimeUnit.SECONDS);
        } catch (Exception ex) {
            // ignore, we just need to wait a bit till the task finishes
        }
        LOG.log(java.util.logging.Level.INFO, "Cancel of init task: {0}", b);
        EXEC.shutdown();
        LOG.log(java.util.logging.Level.INFO, "is shutdown {0}", EXEC.isShutdown());
    }
    
    public void addLog(final File f, final String remoteHost) {
        if (dir == null) {
            LOG.warning("Specify dir attribute, otherwise the server cannot work");
            return;
        }
        
        class R implements Runnable {
            public void run() {
                if (closed) {
                    LOG.info("Processing cancelled");
                    return;
                }

                EntityManager em = PersistenceUtils.getInstance().createEntityManager();
                try {
                    LOG.info("processing parsing " + f);
                    Logfile info = getLogInfo(getIdSes(f.getName()), getSessionNumber(f.getName()), false);
                    info.setIpAddr(remoteHost);
                    Utils.getPersistenceUtils().merge(info);
                    LOG.fine("IP address stored as " + remoteHost + " for " + f);
                    handleAddNewLog(f, statistics, true, em);
                    initCounts = Collections.nCopies(2, getNumberOfLogs());
                    LOG.fine("done processing parsing " + f);
                    if (em.getTransaction().isActive()){
                        em.flush();
                    }
                } catch (Throwable t) {
                    LOG.log(Level.SEVERE, "failure during parsing " + f + "from " + remoteHost, t);
                    if (t instanceof ThreadDeath) {
                        throw (ThreadDeath)t;
                    }
                } finally {
                    em.close();
                }
            }
        }
        LOG.info("request for parsing " + f);
        R run = new R();
        initTask = EXEC.submit(run);
    }
    
    /**
     * Getter for number of all submitted logs.
     * @return number of log files in the logs directory.
     */
    public int getNumberOfLogs() {
        return dir == null ? -1 : dir.list().length;
    }
    
    /** Getter for the amount of parsed logs
     * @return 
     */
    public int getParsedLogs() {
        return initCounts.get(0);
    }
    
    /**
     * Getter to check whether the manager thinks everything is right or not
     * @return true if everything seems ok
     */
    public boolean isProperlyConfigured() {
        return !misconfigured;
    }
    
    /** Getter for info about log.
     * @param log name of the log
     * @return information about the log file
     */
    public Logfile getLogInfo(String log) {
        String userdir = getIdSes(log);
        int uploadnumber = getSessionNumber(log);
        return getLogInfo(userdir, uploadnumber);
    }
    
    /** Getter for info about log.
     * @param userdir userdir id
     * @param uploadnumber uploaded log number from this userdir id
     * @return information about the log file
     */
    public Logfile getLogInfo(String userdir, int uploadnumber) {
        return getLogInfo(userdir, uploadnumber, true);
    }
    final Logfile getLogInfo(String userdir, int uploadnumber, boolean wait) {
        if (wait) {
            DbInsertion.waitLogFileInsertFinished();
        }
        Map<String,Object> params = new HashMap<String, Object>(3);
        params.put("userdir", userdir);
        params.put("uploadnumber", uploadnumber);
        List res = null;
        /* synchronized because it can be called from db insertion and from add log,
         * but an insertion must be done just once
        */
        synchronized (logInfoLock){
            res = Utils.getPersistenceUtils().executeNamedQuery("Logfile.findByFile", params);
            if (res.size()==0){
                Logfile result = new Logfile();
                result.setUserdir(userdir);
                result.setUploadNumber(uploadnumber);
                Utils.getPersistenceUtils().persist(result);
                return result;
            }
        }
        return (Logfile) res.get(0);
    }
    
    /**
     * Takes a page context and fills it with informations about all statistics
     * known to the system.
     *
     * @param context the context to fill
     * @param statistics names of statistics to include in the page
     * @throws java.lang.InterruptedException
     * @throws java.util.concurrent.ExecutionException
     */
    public void preparePageContext(PageContext context, Set<String> statistics)
            throws InterruptedException, ExecutionException {
        ServletRequest request = context.getRequest();
        HttpSession session = context.getSession();
        
        String id = null;
        if (request != null) {
            Object o = request.getAttribute("id");
            if (o instanceof String) {
                id = (String)o;
            }
        }
        if (id == null && request != null) {
            id = request.getParameter("id");
        }
        if (id == null && session!= null) {
            id = (String)session.getAttribute("id");
        } else {
            if (id != null && session != null) {
                session.setAttribute("id", id);
            }
        }

        if (request != null) {
            Enumeration<String> en = NbCollections.checkedEnumerationByFilter(request.getParameterNames(), String.class, true);
            while (en.hasMoreElements()) {
                String name = en.nextElement();
                String value = request.getParameter(name);
                if (value == null || value.length() == 0) {
                    continue;
                }
                context.setAttribute(name, value, PageContext.REQUEST_SCOPE);
            }
        }
        
        preparePageContext(context, id, statistics);
    }

    /**
     * Takes a page context and fills it with informations about all statistics
     * known to the system.
     *
     * @param context the context to fill
     * @param id session ID identifying the user (and its userdir)
     * @param statistics names of statistics to include in the page
     * @throws java.lang.InterruptedException
     * @throws java.util.concurrent.ExecutionException
     */
    public void preparePageContext(PageContext context, String id, Set<String> statNames) 
    throws InterruptedException, ExecutionException {
        EntityManager em = PersistenceUtils.getInstance().createEntityManager();
        try {
            preparePageContext(em, context, id, statNames);
        } catch (RuntimeException ex) {
            Exceptions.printStackTrace(ex);
            throw ex;
        } finally {
            em.close();
        }
    }
    private void preparePageContext(EntityManager em, PageContext context, String id, Set<String> statNames) 
    throws InterruptedException, ExecutionException {
        context.setAttribute("manager", this, PageContext.REQUEST_SCOPE);
        
        if (dir == null) {
            LOG.warning("Specify dir attribute, otherwise the server cannot work");
            return;
        }
        if (!initTask.isDone()) {
            int timeout = Integer.getInteger("uihandlerserver.timeout", 30);
            try {
                initTask.get(timeout, TimeUnit.SECONDS);
            } catch (TimeoutException ex) {
                LOG.log(Level.WARNING, "Timeout (" + timeout + ") waiting for page context", ex); // NOI18N
                try {
                    //context.getSession().setAttribute("post.init.url", HttpUtils.getRequestURL(context.getRequest()));
                    context.getRequest().getRequestDispatcher("/initializing.jsp").forward(context.getRequest(), context.getResponse());
                    return;
                } catch (ServletException ex2) {
                    Exceptions.printStackTrace(ex2);
                } catch (IOException ex2) {
                    Exceptions.printStackTrace(ex2);
                }
            }
        }
        
        int cnt = 0;
        for (Value<?> value : this.statistics) {
            if (statNames != null && !statNames.contains(value.statistics.name)) {
                continue;
            }
            registerValue(context, value, id, "global");
            cnt++;
        }
        
        if (statNames != null && cnt < statNames.size()) {
            HashSet<String> unknown = new HashSet<String>(statNames);
            for (Value<?> value : this.statistics) {
                unknown.remove(value.statistics.name);
            }
            throw new IllegalArgumentException("Unknown statistic: " + unknown); // NOI18N`
        }
        
        if (id == null) {
            return;
        }
        
        
        List<Value> users = new ArrayList<Value>(statistics.size());
        for (Value<?> v : this.statistics) {
            addStatisticsData(users, v.statistics);
        }
        List<Value> lasts = new ArrayList<Value>(Collections.<Value>nCopies(statistics.size(), null));
        int lastLog = -1;
        
        Query q = em.createNamedQuery("Logfile.findByUserdir").setParameter("userdir", id); // NOI18N
        for (Object o : q.getResultList()) {
            Logfile log = (Logfile)o;

            boolean isLatest = lastLog < log.getUploadNumber();
            if (isLatest) {
                lastLog = log.getUploadNumber();
            }
            int index = 0;
            for (Value<?> value : users) {
                if (value == null) {
                    continue;
                }
                if (statNames != null && !statNames.contains(value.statistics.name)) {
                    continue;
                }
                Preferences prefs = DbPreferences.root(log, value.statistics, em);
                Value<?> readValue = readAndAddStatisticsData(value, prefs);
                if (isLatest) {
                    lasts.set(index, readValue);
                }
                index++;
            }
        }
        
        for (Value<?> value : lasts) {
            if (value == null) {
                continue;
            }
            if (statNames != null && !statNames.contains(value.statistics.name)) {
                continue;
            }
            if (value != null) {
                registerValue(context, value, id, "last");
            }
        }
        
        
        for (Value<?> value : users) {
            if (statNames != null && !statNames.contains(value.statistics.name)) {
                continue;
            }
            registerValue(context, value, id, "user");
        }
    }
    
    private <Data> void registerValue(PageContext context, Value<Data> data, String id, String prefix) {
        data.statistics.registerPageContext(context, prefix + data.statistics.name, data.value);
    }
    
    static <Data> void addStatisticsData(List<Value> toAdd, Statistics<Data> s) {
        toAdd.add(new Value<Data>(s));
    }
    static <Data> void addStatisticsData(List<Value> toAdd, Value<Data> s) {
        toAdd.add(new Value<Data>(s.statistics));
    }
    static <Data> void addDataForLog(Value<Data> value, Logfile file, EntityManager em) {
        try {
            Preferences prefs = DbPreferences.root(file, value.statistics, em);
            Data logData = value.statistics.read(prefs);
            value.value = value.statistics.join(value.value, logData);
        } catch (BackingStoreException ex) {
            LOG.log(Level.WARNING, "Cannot read data", ex);
        }
    }
    static <Data> void clearData(Value<Data> v) {
        v.value = v.statistics.newData();
    }
    
    static <Data> Value<Data> readAndAddStatisticsData(Value<Data> v, Preferences prefs) {
        Data d;
        try {
            d = v.statistics.read(prefs);
        } catch (BackingStoreException ex) {
            LOG.log(Level.WARNING, ex.getMessage(), ex);
            d = v.statistics.newData();
        }
        v.value = v.statistics.join(v.value, d);
        Value<Data> ret = new Value<Data>(v.statistics);
        ret.value = d;
        return ret;
    }
    
    static final class Value<Data> {
        final Statistics<Data> statistics;
        Data value;
        
        public Value(Statistics<Data> statistics) {
            this.statistics = statistics;
            this.value = statistics.newData();
        }

        private void persist(Logfile logfile, EntityManager em) {
            Preferences prefs = DbPreferences.root(logfile, statistics, em);
            try {
                statistics.write(prefs, value);
                LogfileParsed lfp = new LogfileParsed(statistics.name, logfile, statistics.revision);
                em.merge(lfp);
            } catch (BackingStoreException ex) {
                LOG.log(Level.WARNING, ex.getMessage(), ex);
            }
        }
    }
    
    static final class SessionInfo {
        private final String id;
        private final int session;
        private final File log;
        private final Logfile logfile;
        private volatile List<Value> values;
        private volatile boolean computed;
        
        SessionInfo(String id, int session, File log, Logfile logfile, Collection<Value> stats) {
            this.log = log;
            this.id = id;
            this.session = session;
            this.logfile = logfile;
            this.values = new ArrayList<Value>();
            for (Value<?> v : stats) {
                addStatisticsData(this.values, v);
            }
        }
        
        public List<Value> getValues() {
            if (computed) {
                return values;
            }
            
            class H extends Handler {
                public void publish(LogRecord rec) {
                    for (LogsManager.Value<?> value : values) {
                        addRecord(value, rec);
                    }
                }
                
                private <Data> void addRecord(Value<Data> value, LogRecord rec) {
                    Data d = value.statistics.process(rec);
                    value.value = value.statistics.join(value.value, d);
                }
                
                public void flush() {
                }
                
                public void close() throws SecurityException {
                }
            }
            H h = new H();
            
            InputStream is = null;
            try {
                is = new BufferedInputStream(new FileInputStream(log));
                LogRecords.scan(is, h);
            } catch (IOException ex) {
                LOG.log(Level.SEVERE, "Cannot read records: " + log, ex);
            } finally {
                if (is != null) {
                    try {
                        is.close();
                    } catch (IOException ex) {
                        LOG.log(Level.INFO, "Cannot close stream: " + log, ex);
                    }
                }
            }
            
            computed = true;
            
            return values;
        }
        
        @SuppressWarnings("unchecked")
        final void addValues(List<Value> sum) {
            assert sum.size() == getValues().size();
            
            for (int i = 0; i < sum.size(); i++) {
                Value ses = getValues().get(i);
                Value user = sum.get(i);
                assert ses.statistics == user.statistics;
                
                user.value = user.statistics.join(user.value, ses.value);
            }
            
        }
        
        public void persist(EntityManager em) {
            getValues();
            for (Value<?> value : values) {
                value.persist(logfile, em);
            }

        }
    } // end of SessionInfo
}
