/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.server.uihandler;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.modules.exceptions.entity.Exceptions;
import org.netbeans.modules.exceptions.entity.Hashcodes;
import org.netbeans.modules.exceptions.entity.Line;
import org.netbeans.modules.exceptions.entity.Stacktrace;
import org.netbeans.modules.exceptions.utils.LineComparator;

/**
 *
 * @author Jindrich Sedek
 */
public interface CheckingFilter {
    
    /**this function checks a stacktrace to duplicates
     *
     * @param thrown thrown is a stacktrace to check
     * @return Exception class that this is a duplicate of or null if unable to decide
     */
    public org.netbeans.modules.exceptions.entity.Exceptions check(Throwable thrown);
    
    
    public class WholeStackTraceFilter implements CheckingFilter{
        
        public WholeStackTraceFilter() {
        }
        
        public org.netbeans.modules.exceptions.entity.Exceptions check(Throwable thrown) {
            TreeMap<String, Object> params = new TreeMap<String, Object>();
            params.put("code", Utils.countHash(thrown.getStackTrace()));
            List<Hashcodes> results = Utils.getPersistenceUtils().executeNamedQuery("Hashcodes.findByCode", params);
            for (Hashcodes hashcodes : results) {
                if (verify(thrown, hashcodes.getExceptions())) return hashcodes.getExceptions();
            }
            return null;
        }
        
        // verify the same stacktraces - hashcode is not sure for it
        private boolean verify(Throwable thrown, org.netbeans.modules.exceptions.entity.Exceptions exception){
            StackTraceElement[] stackTrace = thrown.getStackTrace();
            Map<String, Object> params = Collections.singletonMap("stacktraceId", (Object)exception.getStacktrace().getId());
            List<Line> lines = Utils.getPersistenceUtils().executeNamedQuery("Line.findByStacktraceId", params);
            Collections.sort(lines, new LineComparator());
            if (stackTrace.length != lines.size()){// different lines count
                return false;
            }
            for (int i=0;i < lines.size();i++){
                String method = stackTrace[i].getClassName().concat(".").concat(stackTrace[i].getMethodName());
                if (!method.equals(lines.get(i).getMethod().getName())){//different lines
                    return false;
                }
            }
            return true;
        }
    }
    
    public class LinesCheckingFilter implements CheckingFilter{
        private final int depth;
        private static final String SELECT_STACKTRACE_BY_HASH_FROM_LINE = "SELECT l.stacktrace FROM Line l WHERE l.lineHashcode = :code";  //NOI18N
        
        public LinesCheckingFilter(int depth){
            this.depth = depth;
        }
        
        public org.netbeans.modules.exceptions.entity.Exceptions check(Throwable thrown) {
            // no exact matching found - try to go through lines
            TreeMap<String, Object> params = new TreeMap<String, Object>();
            String linesConcatenation = null, methodName;
            List<Stacktrace>  stacktracesResult = null;
            StackTraceElement element;
            Set<Stacktrace> stackTraceIds = null, nextIds = null;
            Stacktrace[] last = null;
            Stacktrace result = null;
            for (int i = 0; i < thrown.getStackTrace().length; i++) {
                element = thrown.getStackTrace()[i];
                methodName = DbInsertion.lineConcat(element);// NOI18N
                if (linesConcatenation == null){
                    linesConcatenation = methodName;
                }else{
                    linesConcatenation = linesConcatenation.concat(methodName);
                }
                params.put("code", linesConcatenation.hashCode());//rewrite previous code
                stacktracesResult = Utils.getPersistenceUtils().executeQuery(SELECT_STACKTRACE_BY_HASH_FROM_LINE, params);
                if (stackTraceIds == null){//fill all results of first line matching
                    stackTraceIds = new HashSet<Stacktrace>(stacktracesResult);
                    removeWrongClasses(thrown, stackTraceIds);
                    if (thrown.getCause() != null){//#112883
                        removeExceptionsWithWrongCause(thrown, stackTraceIds);
                    }
                }else{
                    nextIds = new HashSet<Stacktrace>(stacktracesResult);
                }
                if (nextIds != null) {
                    stackTraceIds.retainAll(nextIds); //in stackTraceIds is now an intersection with nextIds
                }
                if (stackTraceIds.size() == 0) {
                    if ((last != null) && (last.length > 0)) {
                        //last contains the most simillar stacktraces to this one
                        if ((last.length == 1) && (i >= depth)) {
                            // only one possibility and depth lines matching
                            result = last[0];
                        }
                        if (i > depth + 2) {
                            //more lines matching - let's get the first issue
                            result = last[0];
                        }
                    }
                    break; // break allways since stackTraceIds.size() == 0 !
                }
                if (stackTraceIds.size() > 0){
                    if (last != null) last = stackTraceIds.toArray(last);
                    else last = stackTraceIds.toArray(new Stacktrace[0]);
                }
            }
            return getRootExceptions(result);
        }
        
        public static Exceptions getRootExceptions(Stacktrace stacktrace){
            if (stacktrace == null) return null;
            /* go to root stacktrace */
           while (stacktrace.getStacktrace()!= null) stacktrace = stacktrace.getStacktrace();
           Map<String, Object> params = Collections.singletonMap("stack", (Object)stacktrace);
           List<org.netbeans.modules.exceptions.entity.Exceptions> list = Utils.getPersistenceUtils().executeQuery("SELECT e from Exceptions e where e.stacktrace = :stack", params);
           if ((list != null)&&(list.size()>0)) return list.get(0);
           return null;
        }
        
        private void removeExceptionsWithWrongCause(Throwable thrown, Set<Stacktrace> results) {
            Throwable cause = thrown.getCause();
            String message = cause.getMessage();
            if ((message == null) ||(results == null)){
                return ;
            }
            Set<Stacktrace> removed = new HashSet<Stacktrace>();
            for (Stacktrace stacktrace : results) {
                Stacktrace annotation = stacktrace.getAnnotation();
                if (annotation == null){
                    removed.add(stacktrace);//there should be some annotation
                    continue;
                }
                String annotationMessage = annotation.getMessage();
                if (annotationMessage == null){
                    removed.add(stacktrace);//there should be some message
                    continue;
                }
                if (!sameMessages(cause, annotation)){
                    removed.add(stacktrace);
                }
            }
            results.removeAll(removed);
        }
        
        private void removeWrongClasses(Throwable thrown, Set<Stacktrace> results) {
            String message = thrown.getMessage();
            if ((message == null) || (results == null)) {
                return;
            }
            Set<Stacktrace> removed = new HashSet<Stacktrace>();
            for (Stacktrace stacktrace : results) {
                //remove all stacktraces with different message
                if (!sameMessages(thrown, stacktrace)) {
                    removed.add(stacktrace);
                }
            }
            results.removeAll(removed);
        }

        private boolean sameMessages(Throwable thrown, Stacktrace stacktrace) {
            String message = thrown.getMessage();
            String classFromMessage = getClass(message);
            if (classFromMessage.contains("ClassNotFoundException")) {
                message = removeLoader(message);
                String stackTraceMessage = stacktrace.getMessage();
                if (stackTraceMessage != null) {
                    stackTraceMessage = removeLoader(stackTraceMessage);
                    if (!message.equals(stackTraceMessage)) {
                        return false;
                    }
                }
            } else {
                String stackTraceMessage = stacktrace.getMessage();
                if (stackTraceMessage != null) {
                    String stackTraceClass = getClass(stackTraceMessage);
                    if (!classFromMessage.equals(stackTraceClass)) {
                        return false;
                    }
                }
            }
            return true;
        }

        public static String getClass(String message) {
            int index = message.indexOf(':');
            /*there is no message - just exception class*/
            if (index == -1) {
                index = message.length();
            }
            return message.substring(0, index);
        }

        private String removeLoader(String message){
        int begin = message.indexOf('[');
            if (begin != -1){
            return message.substring(0, begin);
        }
        return message;
            
    }

    }
    
    public class StackOverflowCheckingFilter implements CheckingFilter{
        private static String SELECT_STACKTRACE_BY_NAME_FROM_LINE = 
                "SELECT l.stacktrace FROM Line l, Method m WHERE l.linePK.methodId = m.id AND m.name = :name";  //NOI18N;
        public Exceptions check(Throwable thrown) {
            if (thrown.getMessage() == null) return null;
            String className = LinesCheckingFilter.getClass(thrown.getMessage());
            if (className.contains("StackOverflowError")){
                StackTraceElement[] elements = thrown.getStackTrace();
                int centeralLine = elements.length /2;
                boolean found = false;
                StackTraceElement firstLine = elements[centeralLine];
                int cycleEnd;
                for (cycleEnd = centeralLine + 1; cycleEnd < elements.length; cycleEnd++){
                    if (firstLine.equals(elements[cycleEnd])){
                        found = true;
                        break;
                    }
                }
                if (found==false){
                    Logger.getLogger(StackOverflowCheckingFilter.class.getName())
                            .log(Level.SEVERE, "NO RECURSION FOUND", thrown);
                    return null;
                }
                String shortestLine = DbInsertion.lineConcat(elements[centeralLine]);
                int shortestLineIndex = centeralLine;
                for (int j = centeralLine + 1; j < cycleEnd; j++){
                    String concatenation = DbInsertion.lineConcat(elements[j]);
                    if (shortestLine.compareTo(concatenation) < 0){
                        shortestLine = concatenation;
                        shortestLineIndex = j;
                    }
                }
                Map<String, Object> params = Collections.singletonMap("name", (Object)shortestLine);
                List<Stacktrace>  stacktracesResult;//kandidates for duplicates
                stacktracesResult = Utils.getPersistenceUtils().executeQuery(
                        SELECT_STACKTRACE_BY_NAME_FROM_LINE, params);
                for (Stacktrace stacktrace : stacktracesResult) {
                    Collection<Line> lines = stacktrace.getLineCollection();
                    Iterator<Line> iterator = lines.iterator();
                    while (iterator.hasNext()){
                        Line line = iterator.next();
                        int j = 0;
                        while (line.equals(elements[shortestLineIndex+j])){
                            line = iterator.next();
                            j++;
                        }
                        if (j >= centeralLine - cycleEnd){
                            return LinesCheckingFilter.getRootExceptions(stacktrace);
                        }
                    }
                }
            }
            return null;
        }
    } 
}

