/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.sql.framework.ui.view;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.LinkedHashMap;
import java.util.Properties;
import java.sql.Connection;
import java.sql.Statement;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import org.netbeans.modules.jdbc.builder.DBMetaData;
import org.netbeans.modules.jdbc.builder.KeyColumn;
import org.netbeans.modules.sql.framework.common.utils.DBExplorerConnectionUtil;
import org.netbeans.modules.sql.framework.ui.utils.BinaryToStringConverter;
import org.openide.util.NbBundle;

import com.sun.sql.framework.exception.DBSQLException;
import com.sun.sql.framework.utils.Logger;

/**
 * Renders rows and columns of an arbitrary ResultSet via JTable.
 *
 * @author Jonathan Giron
 * @version $Revision: 1.3 $
 */
public class ResultSetTablePanel extends JPanel {
    
    private boolean isEditable = true;
    
    private Properties dbProp;
    
    private String sql;
    
    private String catalog;
    
    private String schema;
    
    private String tblName;
    
    private boolean isDirty = false;
    
    private DataOutputPanel dataPanel;
    
    private Map changes = new LinkedHashMap();
    
    private class DataTableModel extends DefaultTableModel {
        /**
         * Returns true regardless of parameter values.
         *
         * @param row the row whose value is to be queried
         * @param column the column whose value is to be queried
         * @return true
         * @see #setValueAt
         */
        public boolean isCellEditable(int row, int column) {
            return isEditable;
        }
        
        public void setValueAt(Object value, int row, int col) {
            ResultSet rs = null;
            Statement stmt = null;
            Connection conn = null;
            String whereClause = " SET ";
            try {
                conn = DBExplorerConnectionUtil.createConnection(dbProp);
                stmt = conn.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
                rs = stmt.executeQuery(sql);
                ResultSetMetaData meta = rs.getMetaData();
                if(meta.getColumnType(col + 1) == java.sql.Types.DATE) {
                    value = "{d'" + value.toString() + "'}";
                } else if(meta.getColumnType(col + 1) == java.sql.Types.TIME) {
                    value = "{t'" + value.toString() + "'}";
                } else if(meta.getColumnClassName(col + 1).equals("java.lang.String")) {
                    value = "'" + value.toString() + "'";
                } else if(meta.getColumnClassName(col + 1).equals("java.math.BigDecimal")){
                    value = value.toString();
                } else if(meta.getColumnClassName(col + 1).equals("java.math.BigInteger")){
                    value = value.toString();
                } else {
                    value = "'" + value.toString() + "'";
                }
                whereClause = whereClause + meta.getColumnName(col + 1) + " = " + value +" WHERE ";
                int colCount = meta.getColumnCount();
                Iterator it = getPrimaryKeys(conn, catalog, schema, tblName).iterator();
                if(it.hasNext()) {
                    for(int j = 0; it.hasNext(); j++) {
                        if(j != 0) {
                            whereClause = whereClause + " AND ";
                        }
                        KeyColumn key = (KeyColumn)it.next();
                        String keyName = key.getColumnName();
                        String val = "";
                        for(int i = 0; i < table.getColumnCount(); i++) {
                            if(table.getColumnModel().getColumn(i).getHeaderValue().equals(keyName)) {
                                if(this.getValueAt(row, i) != null) {
                                    if(meta.getColumnClassName(i + 1).equals("java.lang.String")) {
                                        whereClause = whereClause + keyName + " = '" + this.getValueAt(row, i) + "'";
                                    } else if(meta.getColumnType(i + 1) == java.sql.Types.DATE) {
                                        whereClause = whereClause + keyName +" = {d'" + this.getValueAt(row, i) + "'}";
                                    } else if(meta.getColumnType(i + 1) == java.sql.Types.TIME) {
                                        whereClause = whereClause + keyName +" = {t'" + this.getValueAt(row, i) +"'}";
                                    } else if(meta.getColumnClassName(i + 1).equals("java.math.BigDecimal")){
                                        whereClause = whereClause + keyName + " = " + this.getValueAt(row, i);
                                    } else if(meta.getColumnClassName(i + 1).equals("java.math.BigInteger")){
                                        whereClause = whereClause + keyName + " = " + this.getValueAt(row, i);
                                    } else {
                                        whereClause = whereClause + keyName + " = '" + this.getValueAt(row, i) + "'";
                                    }
                                    break;
                                }
                            }
                        }
                    }
                } else {
                    for(int i = 1; i <= colCount; i++) {
                        if(this.getValueAt(row, i - 1) != null) {
                            String columnName = meta.getColumnName(i);
                            if(i != 1) {
                                if(!whereClause.endsWith("AND ")) {
                                    whereClause = whereClause + " AND ";
                                }
                            }
                            if(meta.getColumnClassName(i).equals("java.lang.String")) {
                                whereClause = whereClause + columnName + " = '" + this.getValueAt(row, i - 1) + "'";
                            } else if(meta.getColumnType(i) == java.sql.Types.DATE) {
                                whereClause = whereClause + columnName + " = {d'" + this.getValueAt(row, i - 1) + "'}";
                            } else if(meta.getColumnType(i) == java.sql.Types.TIME) {
                                whereClause = whereClause + columnName + " = {t'" + this.getValueAt(row, i - 1) + "'}";
                            } else if(meta.getColumnClassName(i).equals("java.math.BigDecimal")){
                                whereClause = whereClause + columnName + " = " + this.getValueAt(row, i - 1);
                            } else if(meta.getColumnClassName(i).equals("java.math.BigInteger")){
                                whereClause = whereClause + columnName + " = " + this.getValueAt(row, i - 1);
                            } else if (meta.getColumnType(i) == java.sql.Types.BINARY) {
                                continue;
                            } else if (meta.getColumnType(i) == java.sql.Types.BLOB) {
                                continue;
                            } else if (meta.getColumnType(i) == java.sql.Types.CLOB) {
                                continue;
                            } else {
                                whereClause = whereClause + columnName + " = '" + this.getValueAt(row, i - 1) + "'";
                            }
                        }
                    }
                }
                super.setValueAt(value, row, col);
                isDirty = true;
                String changeData = (row + 1) + ";" + (col + 1);
                changes.put(changeData, whereClause);
                whereClause = "";
                dataPanel.commit.setEnabled(true);
            } catch (Exception ex) {
                Logger.printThrowable(Logger.ERROR, ResultSetTablePanel.class.getName(),
                        null, "Error connection to database", ex);
            } finally {
                try {
                    conn.close();
                    fireTableDataChanged();
                    table.revalidate();
                    table.repaint();
                    table.updateUI();
                } catch (Exception ex) {
                    //ignore
                }
            }
        }
    }
    
    public Properties getConnectionProperties() {
        return dbProp;
    }
    
    public Map getChanges() {
        return changes;
    }
    
    public void fireTableModelChange() {
        
    }
    
    public String[] getDeleteRowsSQL() {
        int[] rows = this.table.getSelectedRows();
        String[] sqls = new String[rows.length];
        Connection conn = null;
        Statement stmt = null;
        ResultSet rs = null;
        try {
            conn = DBExplorerConnectionUtil.createConnection(dbProp);
            stmt = conn.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
            rs = stmt.executeQuery(this.sql);
            ResultSetMetaData meta = rs.getMetaData();            
            for(int row = 0; row < rows.length; row++) {
                int r = rows[row];
                Iterator it = getPrimaryKeys(conn, catalog, schema, tblName).iterator();
                if(it.hasNext()) {
                    for(int j = 0; it.hasNext(); j++) {
                        if(j != 0) {
                            sqls[row] = sqls[row] + " and ";
                        } else {
                            sqls[row] = "";
                        }
                        KeyColumn key = (KeyColumn)it.next();
                        String keyName = key.getColumnName();
                        for(int i = 0; i < table.getColumnCount(); i++) {
                            if(table.getColumnModel().getColumn(i).getHeaderValue().equals(keyName)) {
                                if(model.getValueAt(r, i) != null) {
                                    if(meta.getColumnClassName(i + 1).equals("java.lang.String")) {
                                        sqls[row] = sqls[row] + keyName + " = '" + model.getValueAt(r, i) + "'";
                                    } else if(meta.getColumnType(i + 1) == java.sql.Types.DATE) {
                                        sqls[row] = sqls[row] + keyName +" = {d'" + model.getValueAt(r, i) + "'}";
                                    } else if(meta.getColumnType(i + 1) == java.sql.Types.TIME) {
                                        sqls[row] = sqls[row] + keyName +" = {t'" + model.getValueAt(r, i) +"'}";
                                    } else if(meta.getColumnClassName(i + 1).equals("java.math.BigDecimal")){
                                        sqls[row] = sqls[row] + keyName + " = " + model.getValueAt(r, i);
                                    } else if(meta.getColumnClassName(i + 1).equals("java.math.BigInteger")){
                                        sqls[row] = sqls[row] + keyName + " = " + model.getValueAt(r, i);
                                    } else {
                                        sqls[row] = sqls[row] + keyName + " = '" + model.getValueAt(r, i) + "'";
                                    }
                                    break;
                                }
                            }
                        }
                    }
                } else {
                    for(int col = 0; col < this.table.getColumnCount(); col++) {
                        StringBuilder value = new StringBuilder();
                        if (sqls[row] != null) {
                            sqls[row] = sqls[row] + " and ";
                        } else {
                            sqls[row] = "";
                        }
                        value.append(meta.getColumnName(col + 1) + "=");
                        if(meta.getColumnType(col + 1) == java.sql.Types.DATE) {
                            value.append("= {d'" + model.getValueAt(r, col).toString() + "'}");
                        } else if(meta.getColumnType(col + 1) == java.sql.Types.TIME) {
                            value.append("{t'" + model.getValueAt(r, col).toString() + "'}");
                        } else if(meta.getColumnClassName(col + 1).equals("java.lang.String")) {
                            value.append("'" + model.getValueAt(r, col).toString() + "'");
                        } else if(meta.getColumnClassName(col + 1).equals("java.math.BigDecimal")){
                            value.append(model.getValueAt(r, col).toString());
                        } else if(meta.getColumnClassName(col + 1).equals("java.math.BigInteger")){
                            value.append(model.getValueAt(r, col).toString());
                        } else {
                            value.append("'" + model.getValueAt(r, col).toString() + "'");
                        }
                        sqls[row] = sqls[row] + value.toString();
                    }
                }
            }
        } catch (DBSQLException ex) {
            //ignore
        } catch (SQLException ex) {
            Logger.print(Logger.ERROR, LOG_CATEGORY, ex.getMessage());
        } finally {
            try {
                rs.close();
                stmt.close();
                conn.close();
            } catch (SQLException ex) {
                //ignore
            }
        }
        return sqls;
    }
    
    public void setEditable(boolean edit) {
        this.isEditable = edit;
        this.table.setRowSelectionAllowed(edit);
    }
    
    public void setSql(String query) {
        this.sql = query;
    }
    
    public String getSql() {
        return this.sql;
    }
    
    public void setTableName(String tbl) {
        this.tblName = tbl;
    }
    
    public void setCatalog(String tcatalog) {
        this.catalog = tcatalog;
    }
    
    public void setSchema(String tschema) {
        this.schema = tschema;
    }
    
    public boolean isDirty() {
        if(!isDirty) {
            changes.clear();
        }
        return isDirty;
    }
    
    public void setDirtyStatus(boolean dirty) {
        isDirty = dirty;
        if(!isDirty) {
            changes.clear();
        }
    }
    
    private List getPrimaryKeys(Connection conn, String catalog, String schema, String tableName) {
        List lst = null;
        try {
            DBMetaData meta = new DBMetaData();
            meta.connectDB(conn);
            lst = meta.getPrimaryKeys(catalog, schema, tableName);
        } catch (Exception ex) {
            //ignore
        }
        return lst;
    }
    
    private static class NullObjectCellRenderer extends DefaultTableCellRenderer {
        static final String NULL_LABEL = NbBundle.getMessage(ResultSetTablePanel.class, "LBL_showdata_null");
        
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            Component c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            
            setValue(NULL_LABEL);
            c.setForeground(Color.GRAY);
            
            return c;
        }
    }
    
    private static class ResultSetCellRenderer extends DefaultTableCellRenderer {
        static final TableCellRenderer NULL_RENDERER = new NullObjectCellRenderer();
        
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            return (null == value) ? NULL_RENDERER.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column) : super.getTableCellRendererComponent(
                    table, value, isSelected, hasFocus, row, column);
        }
    }
    
    private static final String data = "WE WILL EITHER FIND A WAY, OR MAKE ONE.";
    
    /* Log4J category string */
    private static final String LOG_CATEGORY = ResultSetTablePanel.class.getName();
    
    int MAX_COLUMN_WIDTH = 50;
    
    /* TableModel containing contents of result set */
    private TableModel model;
    
    private final int multiplier;
    
    /* JTable displaying contents of TableModel */
    private JTable table;
    
    /**
     * Constructs empty instance of SQLResultSetTableView. Call setResultSet(ResultSet) to
     * display the contents of a given ResultSet.
     *
     * @see #setResultSet
     */
    public ResultSetTablePanel() {
        this.setLayout(new BorderLayout());
        table = new JTable();
        table.setRowSelectionAllowed(false);
        table.setColumnSelectionAllowed(false);
        table.setCellSelectionEnabled(true);
        table.setDefaultRenderer(Object.class, new ResultSetCellRenderer());
        
        table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        JScrollPane sp = new JScrollPane(table);
        this.add(sp, BorderLayout.CENTER);
        
        multiplier = table.getFontMetrics(table.getFont()).stringWidth(data) / data.length();
        
        // Set the header with sort button
        ButtonTableHeader bHeader = new ButtonTableHeader();
        bHeader.setReorderingAllowed(false);
        table.setTableHeader(bHeader);
    }
    
    public ResultSetTablePanel(DataOutputPanel panel) {
        this();
        this.dataPanel = panel;
    }
    
    /**
     * Constructs instance of SQLResultSetTableView to display the result sets contained
     * in the given Map.
     *
     * @param rsMap Map of ResultSets; possibly empty
     */
    public ResultSetTablePanel(Map rsMap) {
        this();
        setResultSet(rsMap);
    }
    
    /**
     * Constructs instance of SQLResultSetTableView to display the given result set.
     *
     * @param rs ResultSet
     */
    public ResultSetTablePanel(ResultSet rs) {
        this();
        setResultSet(rs);
    }
    
    public void clearView() {
        DataTableModel dtm = new DataTableModel();
        final SortableTableModel sortModel = new SortableTableModel(dtm);
        Runnable run = new Runnable() {
            public void run() {
                table.setModel(sortModel);
            }
        };
        SwingUtilities.invokeLater(run);
    }
    
    /**
     * Updates this view's data model with the results sets contained in the given Map.
     *
     * @param rsMap Map of ResultSets; possibly empty
     */
    public void setResultSet(Map rsMap) {
        if (rsMap == null) {
            throw new IllegalArgumentException("Must supply non-null Map reference for rsMap");
        }
        
        model = createModelFrom(rsMap);
        final TableModel tempModel = model;
        Runnable run = new Runnable() {
            public void run() {
                table.setModel(tempModel);
            }
        };
        SwingUtilities.invokeLater(run);
    }
    
    
    /**
     * Updates this view's data model to display the contents of the given ResultSet.
     *
     * @param rs new ResultSet to be displayed.
     */
    public void setResultSet(ResultSet rs) {
        this.setResultSet(rs, -1);
    }
    
    /**
     * Updates this view's data model to display the contents of the given ResultSet.
     *
     * @param rs new ResultSet to be displayed.
     */
    public void setResultSet(ResultSet rs, int maxRowsToShow) {
        if (rs == null) {
            throw new IllegalArgumentException("Must supply non-null ResultSet reference for rs");
        }
        // Get RSMD before rs is iterated - DB2Connect Universal driver closes ResultSet
        // after rs.next() is advanced to end.
        ResultSetMetaData rsmd = null;
        try {
            rsmd = rs.getMetaData();
        } catch (SQLException ignore) {
            // Could not obtain metadata - headers will not be displayed.
        }
        
        model = createModelFrom(rs, maxRowsToShow);
        final TableModel tempModel = model;
        Runnable run = new Runnable() {
            public void run() {
                table.setModel(tempModel);
            }
        };
        SwingUtilities.invokeLater(run);
        
        if (rsmd != null) {
            setHeader(table, rsmd);
        }
    }
    
    /**
     * Updates this view's data model to display the contents of the given ResultSet.
     *
     * @param rs new ResultSet to be displayed.
     */
    public void setResultSet(ResultSet rs, Properties connProp, int maxRowsToShow, int startFrom) {
        if (rs == null) {
            throw new IllegalArgumentException("Must supply non-null ResultSet reference for rs");
        }
        this.dbProp = connProp;
        
        // Get RSMD before rs is iterated - DB2Connect Universal driver closes ResultSet
        // after rs.next() is advanced to end.
        ResultSetMetaData rsmd = null;
        try {
            rsmd = rs.getMetaData();
        } catch (SQLException ignore) {
            // Could not obtain metadata - headers will not be displayed.
        }
        
        model = createModelFrom(rs, maxRowsToShow, startFrom);
        final TableModel tempModel = model;
        Runnable run = new Runnable() {
            public void run() {
                table.setModel(tempModel);
            }
        };
        SwingUtilities.invokeLater(run);
        
        if (rsmd != null) {
            setHeader(table, rsmd);
        }
    }
    
    /**
     * create a table model
     *
     * @param rs resultset
     * @return TableModel
     */
    TableModel createModelFrom(ResultSet rs, int maxRowsToShow, int startFrom) {
        DataTableModel dtm = new DataTableModel();
        SortableTableModel sortModel = new SortableTableModel(dtm);
        
        try {
            ResultSetMetaData md = rs.getMetaData();
            
            int colCt = md.getColumnCount();
            int[] colType = new int[colCt + 1];
            // Obtain display name
            for (int i = 1; i <= colCt; i++) {
                dtm.addColumn(md.getColumnLabel(i));
                int columnType = md.getColumnType(i);
                colType[i] = columnType;
            }
            
            Object[] row = new Object[colCt];
            int rowCnt = 0;
            while(rs.getRow() < (startFrom - 1)) {
                rs.next();
            }
            while (((maxRowsToShow == -1) || (maxRowsToShow > rowCnt)) && rs.next()) {
                for (int i = 0; i < colCt; i++) {
                    row[i] = readResultSet(rs, colType[i + 1], i + 1);
                }
                dtm.addRow(row);
                rowCnt++;
            }
        } catch (SQLException e) {
            Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, "Failed to set up table model", e);
        } catch (Exception e) {
            Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, "Failed to set up table model", e);
        }
        
        return sortModel;
    }
    
    /**
     * create a table model
     *
     * @param rs resultset
     * @return TableModel
     */
    TableModel createModelFrom(ResultSet rs, int maxRowsToShow) {
        DataTableModel dtm = new DataTableModel();
        SortableTableModel sortModel = new SortableTableModel(dtm);
        
        try {
            ResultSetMetaData md = rs.getMetaData();
            
            int colCt = md.getColumnCount();
            int[] colType = new int[colCt + 1];
            // Obtain display name
            for (int i = 1; i <= colCt; i++) {
                dtm.addColumn(md.getColumnLabel(i));
                int columnType = md.getColumnType(i);
                colType[i] = columnType;
            }
            
            Object[] row = new Object[colCt];
            int rowCnt = 0;
            while (((maxRowsToShow == -1) || (maxRowsToShow > rowCnt)) && rs.next()) {
                for (int i = 0; i < colCt; i++) {
                    row[i] = readResultSet(rs, colType[i + 1], i + 1);
                }
                dtm.addRow(row);
                rowCnt++;
            }
        } catch (SQLException e) {
            Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, "Failed to set up table model", e);
        } catch (Exception e) {
            Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, "Failed to set up table model", e);
        }
        
        return sortModel;
    }
    
    private Object readResultSet(ResultSet rs, int colType, int index) throws SQLException {
        switch (colType) {
            case Types.BIT:
            case Types.BOOLEAN: {
                boolean bdata = rs.getBoolean(index);
                if (rs.wasNull()) {
                    return null;
                } else {
                    return new Boolean(bdata);
                }
            }
            
            case Types.TIME: {
                Time tdata = rs.getTime(index);
                if (rs.wasNull())
                    return null;
                else
                    return tdata;
            }
            
            case Types.DATE: {
                Date ddata = rs.getDate(index);
                if (rs.wasNull())
                    return null;
                else
                    return ddata;
            }
            
            case Types.TIMESTAMP:
            case -100: // -100 = Oracle timestamp
            {
                Timestamp tsdata = rs.getTimestamp(index);
                if (rs.wasNull())
                    return null;
                else
                    return tsdata;
            }
            
            case Types.BIGINT: {
                long ldata = rs.getLong(index);
                if (rs.wasNull())
                    return null;
                else
                    return new Long(ldata);
            }
            
            case Types.DOUBLE:
            case Types.FLOAT: {
                double fdata = rs.getDouble(index);
                if (rs.wasNull())
                    return null;
                else
                    return new Double(fdata);
            }
            
            case Types.REAL: {
                float rdata = rs.getFloat(index);
                if (rs.wasNull())
                    return null;
                else
                    return new Float(rdata);
            }
            
            case Types.DECIMAL:
            case Types.NUMERIC: {
                BigDecimal bddata = rs.getBigDecimal(index);
                if (rs.wasNull())
                    return null;
                else
                    return bddata;
            }
            
            case Types.INTEGER: {
                int idata = rs.getInt(index);
                if (rs.wasNull())
                    return null;
                else
                    return new Integer(idata);
            }
            
            case Types.SMALLINT: {
                short sidata = rs.getShort(index);
                if (rs.wasNull())
                    return null;
                else
                    return new Short(sidata);
            }
            
            case Types.TINYINT: {
                byte tidata = rs.getByte(index);
                if (rs.wasNull())
                    return null;
                else
                    return new Byte(tidata);
            }
            
            // JDBC/ODBC bridge JDK1.4 brings back -9 for nvarchar columns in
            // MS SQL Server tables.
            // -8 is ROWID in Oracle.
            case Types.CHAR:
            case Types.VARCHAR:
            case Types.LONGVARCHAR:
            case -9:
            case -8: {
                String sdata = rs.getString(index);
                if (rs.wasNull())
                    return null;
                else
                    return sdata;
            }
            
            case Types.BINARY:
            case Types.VARBINARY:
            case Types.LONGVARBINARY: {
                byte[] bdata = rs.getBytes(index);
                if (rs.wasNull())
                    return null;
                else {
                    Byte[] internal = new Byte[bdata.length];
                    for (int i = 0; i < bdata.length; i++)
                        internal[i] = new Byte(bdata[i]);
                    return BinaryToStringConverter.convertToString(internal, BinaryToStringConverter.HEX, false);
                }
            }
            
            case Types.BLOB: {
                // We always get the BLOB, even when we are not reading the contents.
                // Since the BLOB is just a pointer to the BLOB data rather than the
                // data itself, this operation should not take much time (as opposed
                // to getting all of the data in the blob).
                Blob blob = rs.getBlob(index);
                
                if (rs.wasNull())
                    return null;
                
                // BLOB exists, so try to read the data from it
                byte[] blobData = null;
                if (blob != null) {
                    blobData = blob.getBytes(1, 255);
                }
                Byte[] internal = new Byte[blobData.length];
                for (int i = 0; i < blobData.length; i++)
                    internal[i] = new Byte(blobData[i]);
                return BinaryToStringConverter.convertToString(internal, BinaryToStringConverter.HEX, false);
                
            }
            
            case Types.CLOB: {
                // We always get the CLOB, even when we are not reading the contents.
                // Since the CLOB is just a pointer to the CLOB data rather than the
                // data itself, this operation should not take much time (as opposed
                // to getting all of the data in the clob).
                Clob clob = rs.getClob(index);
                
                if (rs.wasNull())
                    return null;
                
                // CLOB exists, so try to read the data from it
                if (clob != null) {
                    return clob.getSubString(1, 255);
                }
            }
            
            case Types.OTHER:
            default:
                return rs.getObject(index);
        }
    }
    
    void setHeader(JTable table, ResultSetMetaData md) {
        try {
            TableColumnModel cModel = table.getColumnModel();
            
            for (int i = 0; i < md.getColumnCount(); i++) {
                int fieldWidth = md.getColumnDisplaySize(i + 1);
                int labelWidth = md.getColumnLabel(i + 1).length();
                int colWidth = Math.max(fieldWidth, labelWidth) * multiplier;
                if (colWidth > MAX_COLUMN_WIDTH * multiplier) {
                    colWidth = MAX_COLUMN_WIDTH * multiplier;
                }
                
                TableColumn column = cModel.getColumn(i);
                column.setPreferredWidth(colWidth);
            }
            
            table.getTableHeader().setColumnModel(cModel);
        } catch (SQLException e) {
            System.err.println("SQLException: " + e.getMessage());
            Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, "Failed to set the size of the table headers", e);
        } catch (Exception e) {
            System.err.println("Unknown Exception: " + e.getMessage());
            Logger.printThrowable(Logger.ERROR, LOG_CATEGORY, this, "Failed to set the size of the table headers", e);
        }
    }
    
    private TableModel createModelFrom(Map rs) {
        DataTableModel dtm = new DataTableModel();
        SortableTableModel sortModel = new SortableTableModel(dtm);
        Object[] row = null;
        
        java.util.Iterator it = rs.keySet().iterator();
        while (it.hasNext()) {
            String key = (String) it.next();
            String value = (String) rs.get(key);
            dtm.addColumn(key);
            row = new Object[value.length()];
        }
        
        java.util.Iterator it1 = rs.values().iterator();
        int i = 0;
        while (it1.hasNext()) {
            if (row != null) {
                row = new Object[rs.size()];
                row[i++] = it1.next();
                dtm.addRow(row);
            }
        }
        
        return sortModel;
    }
}

