/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.bpel.properties.editors.controls.filter;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import org.netbeans.modules.bpel.model.api.ActivityHolder;
import org.netbeans.modules.bpel.model.api.BaseScope;
import org.netbeans.modules.bpel.model.api.Process;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.model.api.CompositeActivity;
import org.netbeans.modules.bpel.model.api.ExtendableActivity;
import org.netbeans.modules.bpel.model.api.Scope;
import org.netbeans.modules.bpel.model.api.VariableDeclarationScope;
import org.netbeans.modules.bpel.model.spi.FindHelper;
import org.openide.util.Lookup;

/**
 * Describes a visibility scope for a bpel element. 
 * There are 2 different kind of visibility scope: for activities and for variables. 
 * Activities are visible inside of Scope elements. 
 * Variables are visible inside of VariableDeclarationScope elements. 
 * The visibility scope is the chain of such elements. This chain is calculated 
 * relative the certain target BPEL element and defines the containers 
 * with elements visible to the target element.  
 * <p>
 * The Scopes has particular order in the chain.
 * The first element of visibility zone is the Process.
 * The most nested scope is at the end of the chain.
 *
 * ATTENTION!
 * TODO: It necessary to check if using of this class is safe. 
 * The matter is that the scope list is cached here and can be chenged after it. 
 * So the cached value can be obsolete.
 *
 * @author nk160297
 */
public class VisibilityScope {
    
    /**
     * This model element has been used to construct the current instance
     */
    private BpelEntity baseModelElement;
    
    private List<BaseScope> myScopeList;
    private List<VariableDeclarationScope> myVarScopeList;
    
    public VisibilityScope(BpelEntity modelElement, Lookup lookup) {
        baseModelElement = modelElement;
    }

    private void calculateScopeList(final BpelEntity modelElement) {
        myScopeList = new LinkedList<BaseScope>();
        FindHelper helper =
                (FindHelper)Lookup.getDefault().lookup(FindHelper.class);
        Iterator<BaseScope> itr = helper.scopeIterator(modelElement);
        while (itr.hasNext()) {
            BaseScope baseScope = itr.next();
            // Put new element to the top of list to provide the sorting contract.
            myScopeList.add(0, baseScope);
        }
        //
        if (modelElement instanceof Process && myScopeList.isEmpty()) {
            myScopeList.add((BaseScope)modelElement);
        }
    }
    
    private void calculateVarScopeList(final BpelEntity modelElement) {
        myVarScopeList = new LinkedList<VariableDeclarationScope>();
        FindHelper helper =
                (FindHelper)Lookup.getDefault().lookup(FindHelper.class);
        Iterator<VariableDeclarationScope> itr =
                helper.varaibleDeclarationScopes(modelElement);
        while (itr.hasNext()) {
            VariableDeclarationScope vdScope = itr.next();
            // Put new element to the top of list to provide the sorting contract.
            myVarScopeList.add(0, vdScope);
        }
        //
        if (modelElement instanceof Process && myVarScopeList.isEmpty()) {
            myVarScopeList.add((BaseScope)modelElement);
        }
    }
    
    /**
     * Returns the Bpel element relative to which the visibility scope was constructed.
     */
    public BpelEntity getBaseModelElement() {
        return baseModelElement;
    }
    
    public List<BaseScope> getScopeChain() {
        if (myScopeList == null) {
            calculateScopeList(baseModelElement);
        }
        return myScopeList;
    }
    
    public BaseScope getClosestScope() {
        List<BaseScope> scopeList = getScopeChain();
        assert !scopeList.isEmpty() : "The VisibilityScope should never be empty"; 
        return scopeList.get(scopeList.size() - 1);
    }
    
    public List<VariableDeclarationScope> getVarScopeChain() {
        if (myVarScopeList == null) {
            calculateVarScopeList(baseModelElement);
        }
        return myVarScopeList;
    }
    
    public VariableDeclarationScope getClosestVarScope() {
        List<VariableDeclarationScope> varScopeList = getVarScopeChain();
        assert !varScopeList.isEmpty() : "The VisibilityScope should never be empty";
        return varScopeList.get(varScopeList.size() - 1);
    }
    
    public static class Utils {
        
        public static List<Scope> getNestedScopes(BpelEntity startFrom) {
            List<Scope> scopes = new ArrayList<Scope>();
            addNestedScopes(startFrom, scopes);
            return scopes;
        }
        
        public static List<VariableDeclarationScope> getNestedVarScopes(
                BpelEntity startFrom) {
            List<VariableDeclarationScope> scopes = 
                    new ArrayList<VariableDeclarationScope>();
            addNestedVarScopes(startFrom, scopes);
            return scopes;
        }
        
        /**
         * Iterate recursively over the BPEL Object model and looking for
         * nested Scope elements. Searching goes on only to the nearest Scopes.
         */
        private static void addNestedScopes(
                BpelEntity startFrom, List<Scope> scopeList) {
            if (startFrom == null || scopeList == null) {
                return;
            } else if (startFrom instanceof Scope) {
                // Check to Scope should be the first to catch only the nearest Scopes!!!
                scopeList.add((Scope)startFrom);
            } else if (startFrom instanceof ActivityHolder) {
                addNestedScopes(
                        ((ActivityHolder)startFrom).getActivity(), scopeList);
            } else if (startFrom instanceof CompositeActivity) {
                ExtendableActivity[] actArr =
                        ((CompositeActivity)startFrom).getActivities();
                for (ExtendableActivity act : actArr) {
                    addNestedScopes(act, scopeList);
                }
            }
        }
        
        /**
         * Iterate recursively over the BPEL Object model and looking for
         * nested VariableDeclarationScope elements. 
         * Searching goes on only to the nearest VariableDeclarationScopes.
         */
        private static void addNestedVarScopes(
                BpelEntity startFrom, List<VariableDeclarationScope> scopeList) {
            if (startFrom == null || scopeList == null) {
                return;
            } else if (startFrom instanceof VariableDeclarationScope) {
                // Check to VariableDeclarationScope should be the first 
                // in order to catch only the nearest Scopes!!!
                scopeList.add((VariableDeclarationScope)startFrom);
            } else if (startFrom instanceof ActivityHolder) {
                addNestedVarScopes(
                        ((ActivityHolder)startFrom).getActivity(), scopeList);
            } else if (startFrom instanceof CompositeActivity) {
                ExtendableActivity[] actArr =
                        ((CompositeActivity)startFrom).getActivities();
                for (ExtendableActivity act : actArr) {
                    addNestedVarScopes(act, scopeList);
                }
            }
        }
        
    }
    
}
