/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */


package org.netbeans.modules.bpel.debugger.ui.process;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.ref.WeakReference;
import java.util.Vector;

import javax.swing.SwingUtilities;

import org.netbeans.modules.bpel.debugger.api.BpelDebugger;
import org.netbeans.modules.bpel.debugger.api.ProcessInstance;
import org.netbeans.modules.bpel.debugger.api.ProcessInstancesModel;
import org.netbeans.spi.debugger.ContextProvider;
import org.netbeans.spi.viewmodel.ModelEvent;
import org.netbeans.spi.viewmodel.ModelListener;
import org.netbeans.spi.viewmodel.TreeModel;
import org.netbeans.spi.viewmodel.UnknownTypeException;
import org.openide.util.RequestProcessor;


/**
 * Represents a process instance table that shows all known
 * process instances and their current known state.
 * This is built to plug into the SOA debugger UI module
 * process view mechanism as seemlessly as possible.
 * 
 * @author Sun Microsystems
 * @author Sun Microsystems
 */
public class ProcessInstancesTreeModel implements TreeModel {

    private BpelDebugger    myDebugger;
    private Listener            myListener;
    private Vector              myListeners = new Vector ();
    
    /**
     * Creates a new instance of ProcessInstancesTreeModel.
     *
     * @param lookupProvider debugger context
     */
    public ProcessInstancesTreeModel(ContextProvider lookupProvider) {
        myDebugger = (BpelDebugger)lookupProvider.
            lookupFirst (null, BpelDebugger.class);
    }
    
    public Object getRoot () {
        return ROOT;
    }
    
    public Object[] getChildren (Object o, int from, int to) 
            throws UnknownTypeException 
    {
        if (o.equals (ROOT)) {
            Object[] os = getProcessInstances(from, to);
            return os;
        } else
        if (o instanceof ProcessInstance) {
            return new Object[0];
        } else {
            throw new UnknownTypeException (o);
        }
    }
    
    /**
     * Returns number of children for given node.
     * 
     * @param   node the parent node
     * @throws  UnknownTypeException if this TreeModel implementation is not
     *          able to resolve children for given node type
     *
     * @return  true if node is leaf
     */
    public int getChildrenCount (Object node) throws UnknownTypeException {
        if (node.equals (ROOT)) {
            return getProcessInstancesCount();
        } else
        if (node instanceof ProcessInstance) {
            return 0;
        } else {
            throw new UnknownTypeException (node);
        }
    }
    
    public boolean isLeaf (Object o) throws UnknownTypeException {
        if (o.equals (ROOT))
            return false;
        if (o instanceof ProcessInstance)
            return true;
        if (o.equals ("NoInfo")) // NOI18N
            return true;
        throw new UnknownTypeException (o);
    }

    public void addModelListener (ModelListener l) {
        myListeners.add (l);
        if (myListener == null && myDebugger != null) {
            myListener = new Listener (this, myDebugger);
        }
    }

    public void removeModelListener (ModelListener l) {
        myListeners.remove (l);
        if (myListeners.size () == 0) {
            myListener.destroy ();
            myListener = null;
        }
    }
    
    void fireTreeChanged () {
        Runnable runner = new Runnable() {
            public void run() {
                Vector v = (Vector) myListeners.clone ();
                int i, k = v.size ();
                for (i = 0; i < k; i++)
                    ((ModelListener) v.get (i)).modelChanged (
                        new ModelEvent.TreeChanged (this)
                    );
            }
        };
        runInDispatch(runner);
    }
    
    void fireNodeChanged (final Object node) {
        Runnable runner = new Runnable() {
            public void run() {
                Vector v = (Vector) myListeners.clone ();
                int i, k = v.size ();
                for (i = 0; i < k; i++)
                    ((ModelListener) v.get (i)).modelChanged (
                        new ModelEvent.NodeChanged(this, node)
                    );
            }
        };
        runInDispatch(runner);
    }
    
    void fireTableValueChanged (final Object node, final String propertyName) {
        Runnable runner = new Runnable() {
            public void run() {
                Vector v = (Vector) myListeners.clone ();
                int i, k = v.size ();
                for (i = 0; i < k; i++)
                    ((ModelListener) v.get (i)).modelChanged (
                        new ModelEvent.TableValueChanged (this, node, propertyName)
                    );
            }
        };
        runInDispatch(runner);
    }
    
    // private methods .........................................................
    
    private void runInDispatch(Runnable runner) {
        if (!SwingUtilities.isEventDispatchThread()) {
            SwingUtilities.invokeLater(runner);
        } else {
            runner.run();
        }
    }
    
    private BpelDebugger getDebugger() {
        return myDebugger;
    }

    private int getProcessInstancesCount() {
        if (myDebugger != null) {
            Object[] pis = myDebugger.getProcessInstancesModel().getProcessInstances();
            if (pis != null) {
                return pis.length;
            }
        }
        return 0;
    }

    private Object[] getProcessInstances(int from, int to) {
        if (myDebugger != null) {
            Object[] pis = myDebugger.getProcessInstancesModel().getProcessInstances();
            Object[] fpis = new Object[to - from];
            System.arraycopy(pis, from, fpis, 0, to - from);
            return fpis;
        }
        return new Object[0];
    }
    
    
    
    // innerclasses ............................................................
    
    private static class Listener 
    implements ProcessInstancesModel.Listener, PropertyChangeListener {
        
        private BpelDebugger myDebugger;
        private WeakReference myModel;
        
        private Listener(ProcessInstancesTreeModel tm, BpelDebugger debugger) {
            myDebugger = debugger;
            myModel = new WeakReference (tm);
            debugger.addPropertyChangeListener(this);
            debugger.getProcessInstancesModel().addListener(this);
        }
        
        void destroy () {
            myDebugger.getProcessInstancesModel().removeListener(this);
            myDebugger.removePropertyChangeListener(this);
            if (task != null) {
                // cancel old task
                task.cancel ();
                task = null;
            }
        }
        
        private ProcessInstancesTreeModel getModel () {
            ProcessInstancesTreeModel tm =
                    (ProcessInstancesTreeModel) myModel.get ();
            if (tm == null) {
                destroy ();
            }
            return tm;
        }
        
        // currently waiting / running refresh task
        // there is at most one
        private RequestProcessor.Task task;
        
        public void propertyChange (PropertyChangeEvent e) {
            if (e.getPropertyName() == BpelDebugger.PROP_CURRENT_PROCESS_INSTANCE) {
                //TODO:get this code back as soon as the bug with the html
                //markup in debugger views is fixed.
                //Making full tree update for now...
                
//                ProcessInstancesTreeModel pitm = getModel();
//                if (pitm == null) return;
//                pitm.fireNodeChanged(e.getOldValue());
//                pitm.fireNodeChanged(e.getNewValue());
                
                asyncFireTreeChanged();
                
            } else if (e.getPropertyName() == BpelDebugger.PROP_STATE) {
                if (myDebugger.getState () == BpelDebugger.STATE_DISCONNECTED) {
                    destroy();
                    asyncFireTreeChanged();
                } else if (myDebugger.getState() == BpelDebugger.STATE_RUNNING) {
                    asyncFireTreeChanged();
                }
            }
        }

        public void processInstanceRemoved(ProcessInstance processInstance) {
            asyncFireTreeChanged();
        }

        public void processInstanceAdded(ProcessInstance processInstance) {
            asyncFireTreeChanged();
        }

        public void processInstanceStateChanged(
                ProcessInstance processInstance, int oldState, int newState)
        {
            //TODO:get this code back as soon as the bug with the html
            //markup in debugger views is fixed.
            //Making full tree update for now...
            
//            ProcessInstancesTreeModel pitm = getModel();
//            if (pitm == null) return;
//            pitm.fireTableValueChanged(processInstance, null);
            
            asyncFireTreeChanged();
        }

        private void asyncFireTreeChanged() {
            final ProcessInstancesTreeModel pitm = getModel();
            if (pitm == null) return;
            if (task != null) {
                // cancel old task
                task.cancel();
                task = null;
            }
            task = RequestProcessor.getDefault().post(new Runnable () {
                public void run () {
                    pitm.fireTreeChanged ();
                }
            }, 500);
        }
    }
}
