/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.bpel.model.validation.xpath;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.LinkedList;
import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import org.netbeans.modules.bpel.model.api.BaseScope;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.model.api.Process;
import org.netbeans.modules.bpel.model.api.BpelModel;
import org.netbeans.modules.bpel.model.api.ContentElement;
import org.netbeans.modules.bpel.model.api.Import;
import org.netbeans.modules.bpel.model.api.VariableContainer;
import org.netbeans.modules.bpel.model.api.VariableDeclaration;
import org.netbeans.modules.bpel.model.api.VariableDeclarationScope;
import org.netbeans.modules.bpel.model.api.references.SchemaReference;
import org.netbeans.modules.bpel.model.api.references.WSDLReference;
import org.netbeans.modules.bpel.model.api.support.ExNamespaceContext;
import org.netbeans.modules.xml.schema.model.Attribute;
import org.netbeans.modules.xml.schema.model.Element;
import org.netbeans.modules.xml.schema.model.Form;
import org.netbeans.modules.xml.schema.model.GlobalAttribute;
import org.netbeans.modules.xml.schema.model.GlobalElement;
import org.netbeans.modules.xml.schema.model.GlobalType;
import org.netbeans.modules.xml.schema.model.LocalAttribute;
import org.netbeans.modules.xml.schema.model.LocalElement;
import org.netbeans.modules.xml.schema.model.SchemaComponent;
import org.netbeans.modules.xml.schema.model.SchemaModel;
import org.netbeans.modules.xml.schema.model.TypeContainer;
import org.netbeans.modules.xml.wsdl.model.Message;
import org.netbeans.modules.xml.wsdl.model.Part;
import org.netbeans.modules.xml.xam.Model.State;
import org.netbeans.modules.xml.xam.Named;
import org.netbeans.modules.xml.xam.dom.AbstractDocumentComponent;
import org.netbeans.modules.xml.xam.dom.NamedComponentReference;
import org.netbeans.modules.xml.xam.spi.Validator.ResultItem;
import org.netbeans.modules.xml.xam.spi.Validator.ResultType;
import org.netbeans.modules.xml.xpath.LocationStep;
import org.netbeans.modules.xml.xpath.StepNodeNameTest;
import org.netbeans.modules.xml.xpath.StepNodeTest;
import org.netbeans.modules.xml.xpath.StepNodeTypeTest;
import org.netbeans.modules.xml.xpath.XPathCoreFunction;
import org.netbeans.modules.xml.xpath.XPathCoreOperation;
import org.netbeans.modules.xml.xpath.XPathExpression;
import org.netbeans.modules.xml.xpath.XPathExpressionPath;
import org.netbeans.modules.xml.xpath.XPathExtensionFunction;
import org.netbeans.modules.xml.xpath.XPathLocationPath;
import org.netbeans.modules.xml.xpath.XPathPredicateExpression;
import org.netbeans.modules.xml.xpath.XPathVariableReference;
import org.netbeans.modules.xml.xpath.visitor.AbstractXPathVisitor;
import org.openide.util.NbBundle;

/**
 * This visitor is intended to validate semantics of single XPath.
 * It can check reference integrity of variables, parts and types.
 *
 * @author nk160297
 */
public class PathValidatorVisitor extends AbstractXPathVisitor {

    private XPathExpression myInitialExpression = null;

    private PathValidationContext myContext;

    private VariableDeclaration myVariable;
    private Part myPart;

    private transient SchemaComponent parentComponent;
    private transient boolean stopPathValidation = false;
    private transient boolean lookForGlobalObject = false;

    private static EntityTypeNameVisitor etnv = new EntityTypeNameVisitor();

    public PathValidatorVisitor(PathValidationContext context) {
        myContext = context;
    }

    public PathValidationContext getContext() {
        return myContext;
    }

    /**
     * This method returns the type of the last element of the path.
     * It represents the type of a value which can be get after
     * execution of the XPath.
     * Method can return null value.
     * It is implied that the type is global because of only
     * global type can be used by another schema object,
     * to which the xpath can be assigned or with which
     * it can be compared.
     * The method returns correct result just after the validator
     * has completely processed.
     */
    public GlobalType getPathResultType() {
        if (parentComponent != null) {
            return getComponentType(parentComponent);
        }
        return null;
    }

    //========================================================
    // Do standard processing
    
    public void visit(XPathCoreFunction coreFunction) {
        visitChildren( coreFunction );
    }

    public void visit(XPathCoreOperation coreOperation) {
        visitChildren( coreOperation );
    }

    public void visit(XPathExtensionFunction extensionFunction) {
        visitChildren( extensionFunction );
    }

    //========================================================
    
    public void visit(XPathLocationPath locationPath) {
        if (myInitialExpression == null) {
            myInitialExpression = locationPath;
        } else {
            // Delegate processing of a new path expression to a new path validator
            PathValidationContext newContext = myContext.clone();
            newContext.setSchemaContextComponent(parentComponent);
            PathValidatorVisitor newPVVisitor = new PathValidatorVisitor(newContext);
            locationPath.accept(newPVVisitor);
            return;
        }
        //
        if (locationPath.getAbsolute()) {
            //
            // Process the first step of an absolute location path.
            //
            SchemaModel contextModel = myContext.getSchemaContextModel();
            SchemaComponent rootComp = myContext.getSchemaContextComponent();
            //
            if (rootComp instanceof GlobalType) {
                // Error. The location path must not be absolute if the global type is used.
                addResultItem(ResultType.ERROR, "ABSOLUTE_XPATH_WITH_TYPE"); // NOI18N
            }
            //
            assert rootComp.getModel() == contextModel;
            lookForGlobalObject = true;
            parentComponent = contextModel.getRootComponent();
        } else {
            parentComponent = myContext.getSchemaContextComponent();
        }
        //
        LocationStep[] steps = locationPath.getSteps();
        if ( steps != null ){
            for (LocationStep step : steps) {
                visit(step);
                if (stopPathValidation) {
                    break;
                }
            }
        }
    }

    public void visit(XPathExpressionPath expressionPath) {
        if (myInitialExpression == null) {
            myInitialExpression = expressionPath;
        } else {
            // Delegate processing of a new path expression to a new path validator
            PathValidationContext newContext = myContext.clone();
            newContext.setSchemaContextComponent(parentComponent);
            PathValidatorVisitor newPVVisitor = new PathValidatorVisitor(newContext);
            expressionPath.accept(newPVVisitor);
            return;
        }
        //
        XPathExpression expression = expressionPath.getRootExpression();
        if ( !expressionPath.equals( expression ) ) {
            expression.accept( this );
        }
        if (stopPathValidation) {
            return;
        }
        //
        GlobalElement gElement = null;
        GlobalType gType = null;
        boolean varTypeResolved = false;
        //
        if (myVariable != null) {
            if (myPart == null) {
                SchemaReference<GlobalElement> elementRef = myVariable.getElement();
                if (elementRef != null) {
                    gElement = elementRef.get();
                }
                if (gElement == null) {
                    SchemaReference<GlobalType> typeRef = myVariable.getType();
                    if (typeRef != null) {
                        gType = typeRef.get();
                    }
                }
            } else {
                NamedComponentReference<GlobalElement> elementRef = myPart.getElement();
                if (elementRef != null) {
                    gElement = elementRef.get();
                }
                if (gElement == null) {
                    NamedComponentReference<GlobalType> typeRef = myPart.getType();
                    if (typeRef != null) {
                        gType = typeRef.get();
                    }
                }
            }
            //
            // Check if the type of the variable is resolved
            varTypeResolved = gElement != null || gType != null;
            if (!varTypeResolved) {
                // Error. Type of the variable can not be resolved
                String varName = myVariable.getVariableName();
                if (myPart != null) {
                    String partName = myPart.getName();
                    varName = varName + "." + partName;
                }
                addResultItem(ResultType.ERROR, "UNRESOLVED_VARIABLE_TYPE", varName); // NOI18N
                stopPathValidation = true;
                return;
            }
        }
        //
        if (gElement != null) {
            parentComponent = gElement;
        } else {
            parentComponent = gType;
        }
        //
        if (parentComponent == null) {
            // Error has to be provided already.
            stopPathValidation = true;
            return;
        }
        //
        if (stopPathValidation) {
            return;
        }
        //
        LocationStep[] steps = expressionPath.getSteps();
        if ( steps != null ){
            for (LocationStep step : steps) {
                step.accept( this );
                if (stopPathValidation) {
                    break;
                }
            }
        }
    }

    public void visit(LocationStep locationStep) {
        //
        // Check if the location step is a of XPath location path,
        // which is started from a variable name.
        if (parentComponent == null) {
            addResultItem(ResultType.ERROR, "MISSING_XPATH_LEADING_VARIABLE"); // NOI18N
            stopPathValidation = true;
            return;
        }
        //
        boolean isAttribute = locationStep.getAxis() == LocationStep.AXIS_ATTRIBUTE;
        if (!isAttribute && locationStep.getAxis() != LocationStep.AXIS_CHILD) {
            // The usage of any axis except the attribute or child can result in
            // loss of type context. It doesn't matter to check schema types any more.
            addResultItem(ResultType.WARNING, 
                    "UNSUPPORTED_AXIS", locationStep.getString()); // NOI18N
            stopPathValidation = true;
            return;
        }
        //
        StepNodeTest nodeTest = locationStep.getNodeTest();
        assert nodeTest != null;
        if (nodeTest instanceof StepNodeNameTest) {
            // get the text of the step
            String nodeName = ((StepNodeNameTest)nodeTest).getNodeName();
            //
            // Extract namespace prefix
            String nsPrefix = null;
            int colonIndex = nodeName.indexOf(':');
            if (colonIndex != -1) {
                nsPrefix = nodeName.substring(0, colonIndex);
                nodeName = nodeName.substring(colonIndex + 1);
            }
            //
            // Obtain the namespace URI by the prefix
            // The absence of prefix means that the XPath step is unqualified.
            // The default namespace can't be used by XPath in BPEL!
            String nsUri = null;
            if (nsPrefix == null) {
                //
                // If the prefix isn't specified for the global object 
                // then the checkNsPrefixes should show warning
                // 
                // If the prefix isn't specified then the step component can
                // be considered as an unqualified schema object.
                // ATTENTION! The namaspace is indefinite in such case. 
                // It doesn't related to the namespace of the parent component 
                // because the child component can be defined in other schema 
                // with other target namespace! It can't be considered as 
                // default namespace as for global elements in such case. 
                // The child element has to be found among all children 
                // by name only. 
                nsUri = null; 
            } else {
                ContentElement contentElement = myContext.getXPathContentElement();
                assert contentElement instanceof AbstractDocumentComponent;
                nsUri = ((AbstractDocumentComponent) contentElement).
                        lookupNamespaceURI(nsPrefix, true);
                //
                if (nsUri == null) {
                    addResultItem(ResultType.WARNING, 
                            "UNKNOWN_NAMESPACE_PREFIX", nsPrefix); // NOI18N
                    stopPathValidation = true;
                    return;
                }
            }
            //
            SchemaComponent foundComponent = null;
            if (lookForGlobalObject) {
                SchemaModel contextModel = myContext.getSchemaContextModel();
                SchemaComponent rootComp = myContext.getSchemaContextComponent();
                //
                String name = null;
                String namespace = contextModel.getEffectiveNamespace(rootComp);
                //
                if (rootComp instanceof GlobalElement) {
                    name = ((GlobalElement)rootComp).getName();
                } else if (rootComp instanceof GlobalAttribute) {
                    name = ((GlobalAttribute)rootComp).getName();
                } else {
                    assert false : "The root component of an absolute " +
                            "location path has to be either GlobalElement " +
                            "or GlobalAttribute"; // NOI18N
                    stopPathValidation = true;
                    return;
                }
                //
                boolean isSuitableType = false;
                if (nsUri == null) {
                    isSuitableType = nodeName.equals(name);
                } else {
                    isSuitableType = (nsUri.equals(namespace) && nodeName.equals(name));
                }
                if (isSuitableType) {
                    foundComponent = rootComp;
                } else {
                    // Error. The XPath has to be started from another global object
                    String correctRootName =
                            (nsPrefix == null ? "" : nsPrefix + ":") + name;
                    if (isAttribute) {
                        addResultItem(ResultType.ERROR,
                                "WRONG_GLOBAL_ATTRIBUTE", correctRootName); // NOI18N
                    } else {
                        addResultItem(ResultType.ERROR,
                                "WRONG_GLOBAL_ELEMENT", correctRootName); // NOI18N
                    }
                }
                //
                // Look for local object next time.
                lookForGlobalObject = false;
            } else {
                // look for local object here.
                FindChildSchemaVisitor visitor =
                        new FindChildSchemaVisitor(nodeName, nsUri, isAttribute);
                visitor.lookForSubcomponent(parentComponent);
                //
                if (visitor.isChildFound()) {
                    foundComponent = visitor.getFound();
                } else {
                    // Error. The child with the specified name isn't found
                    if (isAttribute) {
                        addResultItem(ResultType.ERROR,
                                "UNKNOWN_ATTRIBUTE", nodeName, nsUri); // NOI18N
                    } else {
                        addResultItem(ResultType.ERROR, 
                                "UNKNOWN_ELEMENT", nodeName, nsUri); // NOI18N
                    }
                    //
                    // It doesn't matter to check schema types any more
                    stopPathValidation = true;
                    return;
                }
            }
            //
            assert foundComponent instanceof GlobalElement ||
                    foundComponent instanceof LocalElement ||
                    foundComponent instanceof Attribute;
            //
            checkNsPrefixes(foundComponent, nsPrefix, nsUri);
            //
            parentComponent = foundComponent;
            //
        } else if (nodeTest instanceof StepNodeTypeTest) {
            // It doesn't matter to check schema types any more
            stopPathValidation = true;
            return;
        }
        //
        // Process nested predicates
        // IMPORTANT! This code has to be here because of it requires that
        // the current step element has already calculated.
        // The parentComponent variable points to it.
        XPathPredicateExpression[] expressions = locationStep.getPredicates();
        if ( expressions!= null ){
            for (XPathPredicateExpression expression : expressions) {
                expression.accept( this );
            }
        }
    }

    public void visit(XPathVariableReference vReference) {
        QName varQName = vReference.getVariableName();
        //
        // Check the namespace and prefix of the variable
        // The namespace has to be the same as the
        String namespaceUri = varQName.getNamespaceURI();
        if (namespaceUri != null && namespaceUri.length() != 0) {
            // Error: The XPath doesn't allow to use URI
            addResultItem(ResultType.ERROR, 
                    "NAMESPACE_URI_IN_XPATH", namespaceUri); // NOI18N
        }
        //
        String prefix = varQName.getPrefix();
        if (prefix != null && prefix.length() != 0) {
            BpelModel bpelModel = myContext.getBpelContextActivity().getBpelModel();
            if (bpelModel == null || bpelModel.getState() != State.VALID) {
                // Error: invalid bpel model
                stopPathValidation = true;
                return;
            }
            Process process = bpelModel.getProcess();
            if (process == null) {
                // Error: invalid process
                stopPathValidation = true;
                return;
            }
            ExNamespaceContext nsContext = getNsContext();
            namespaceUri = nsContext.getNamespaceURI(prefix);
            if (namespaceUri == null) {
                // Error: Unknown prefix
                addResultItem(ResultType.ERROR, 
                        "UNKNOWN_NAMESPACE_PREFIX", prefix); // NOI18N
                stopPathValidation = true;
                return;
            }
            //
            String processNsUri = process.getTargetNamespace();
            assert processNsUri != null;
            //
            if (!processNsUri.equals(namespaceUri)) {
                // Error: the variable is defined in a wrong namespace.
                String correctPrefix = nsContext.getPrefix(processNsUri);
                String varName = varQName.getLocalPart();
                addResultItem(ResultType.ERROR,
                        "WRONG_VARIABLE_PREFIX", varName, prefix, correctPrefix); // NOI18N
                stopPathValidation = true;
                return;
            }
        }
        //
        // Extract part
        String varName = varQName.getLocalPart();
        int partIndex = varName.lastIndexOf('.');
        String partName = null;
        if (partIndex != -1) {
            partName = varName.substring(partIndex + 1);
            varName = varName.substring(0, partIndex);
        }
        //
        // Try to find the variable in the model.
        // Visibility rules are taken into consideration!
        myVariable = lookForVariable(varName);
        if (myVariable == null) {
            // Error: Can't find the variable
            //
            // Don't show this error because of it is already detected by another
            // validator: org.netbeans.modules.bpel.model.validation.references.ReferencesValidator
            // See: ReferencesValidator.FIX_VARIABLE
            // addResultItem(ResultType.ERROR, "UNKNOWN_VARIABLE", varName); // NOI18N
            stopPathValidation = true;
            return;
        } else if (partName != null && partName.length() != 0) {
            WSDLReference<Message> messageTypeRef = myVariable.getMessageType();
            if (messageTypeRef == null) {
                // Error: if the part is specified then variable has to be of a message type
                addResultItem(ResultType.ERROR, "VARIABLE_MESSAGE_TYPE_REQUIRED",
                        varName); // NOI18N
                stopPathValidation = true;
                return;
            } else {
                Message message = messageTypeRef.get();
                if (message == null) {
                    // Error: impossible to get message by ref
                    addResultItem(ResultType.ERROR, "CANT_RETRIEVE_MESSAGE_TYPE",
                            messageTypeRef.getRefString(), varName); // NOI18N
                    stopPathValidation = true;
                    return;
                } else {
                    Collection<Part> parts = message.getParts();
                    for (Part part : parts) {
                        if (part.getName().equals(partName)) {
                            myPart = part;
                            break;
                        }
                    }
                }
                //
                if (myPart == null) {
                    // Error: the specified part is not found
                    addResultItem(ResultType.ERROR, "UNKNOWN_MESSAGE_PART",
                            partName, varName); // NOI18N
                    stopPathValidation = true;
                    return;
                }
            }
        }
    }

    //========================================================
    
    /**
     * Obtains the type of the schema component.
     * It works only with components which can have a type.
     */
    private GlobalType getComponentType(SchemaComponent comp) {
        NamedComponentReference<? extends GlobalType> gTypeRef = null;
        if (comp instanceof TypeContainer) {
            gTypeRef = ((TypeContainer)comp).getType();
        } else if (comp instanceof LocalAttribute) {
            gTypeRef = ((LocalAttribute)comp).getType();
        } else if (comp instanceof GlobalAttribute) {
            gTypeRef = ((GlobalAttribute)comp).getType();
        } else {
            // Error. Can not resolve type of the last location path element.
            addResultItem(ResultType.ERROR, "UNRESOLVED_XPATH_TAIL",
                    myInitialExpression.getExpressionString()); // NOI18N
            return null;
        }
        //
        if (gTypeRef == null) {
            // Error. A global type has to be specified for the last element (attribute)
            // of the Location path.
            String lastElementName = ((Named)comp).getName();
            addResultItem(ResultType.ERROR, "XPATH_TAIL_NOT_GLOBAL_TYPE",
                    lastElementName); // NOI18N
            return null;
        } else {
            GlobalType gType = gTypeRef.get();
            if (gType == null) {
                // Error. Can not resolve the global type
                addResultItem(ResultType.ERROR, "UNRESOLVED_GLOBAL_TYPE",
                        gTypeRef.getRefString()); // NOI18N
            }
            //
            return gType;
        }
    }

    /**
     * Checks if the prefix required or redundant.
     * Check if the prefix is correct.
     * Check if the external schema is imported and prefix is defined.
     */
    private void checkNsPrefixes(SchemaComponent sComp, String nsPrefix, String nsUri) {
        Form form = null;
        if (sComp instanceof LocalElement){
            form = ((LocalElement) sComp).getFormEffective();
        } else if (sComp instanceof LocalAttribute){
            form = ((LocalAttribute) sComp).getFormEffective();
        } else {
            form = Form.QUALIFIED; // by default for global components
        }
        //
        if (Form.UNQUALIFIED.equals(form) && nsPrefix != null) {
            // Error. It should be without a prefix
            if (sComp instanceof LocalElement){
                String elementName = ((LocalElement)sComp).getName();
                addResultItem(ResultType.WARNING,
                        "ELEMENT_UNNECESSARY_PREFIX", elementName); // NOI18N
            } else if (sComp instanceof LocalAttribute){
                String attrName = ((LocalAttribute)sComp).getName();
                addResultItem(ResultType.WARNING,
                        "ATTRIBUTE_UNNECESSARY_PREFIX", attrName); // NOI18N
            }
        } else if (Form.QUALIFIED.equals(form) && nsPrefix == null) {
            // Error. It should be qualified.
            //
            // Check if the prefix is declared for the namespace URI
            String preferredPrefix = null;
            if (nsUri != null && nsPrefix == null) {
                preferredPrefix = getPrefixByNsUri(nsUri);
                //
                if (preferredPrefix == null) {
                    // Error. The required prefix isn't declared
                    addResultItem(ResultType.WARNING,
                            "MISSING_NAMESPACE_PREFIX", nsUri); // NOI18N
                }
            }
            //
            // Check if the schema is imported, where current step is defined.
            if (nsUri != null && !isSchemaImported(sComp)) {
                // Error. The required prefix isn't declared
                addResultItem(ResultType.WARNING,
                        "MISSING_SCHEMA_IMPORT", nsUri); // NOI18N
            }
            //
            String name = ((Named)sComp).getName();
            if (sComp instanceof Element){
                if (preferredPrefix == null) {
                    addResultItem(ResultType.WARNING,
                            "ELEMENT_PREFIX_REQUIRED", name); // NOI18N
                } else {
                    addResultItem(ResultType.WARNING,
                            "ELEMENT_SPECIFIC_PREFIX_REQUIRED",
                            name, preferredPrefix); // NOI18N
                }
            } else if (sComp instanceof Attribute){
                if (preferredPrefix == null) {
                    addResultItem(ResultType.WARNING,
                            "ATTRIBUTE_PREFIX_REQUIRED", name); // NOI18N
                } else {
                    addResultItem(ResultType.WARNING,
                            "ATTRIBUTE_SPECIFIC_PREFIX_REQUIRED",
                            name, preferredPrefix); // NOI18N
                }
            }
        }
    }

    private synchronized VariableDeclaration lookForVariable(String varName) {
        LinkedList<VariableDeclarationScope> varDeclScopeList =
                myContext.getVarDeclScopeList();
        //
        VariableDeclaration result = null;
        //
        for (VariableDeclarationScope varScope : varDeclScopeList) {
            result = findVariableByNameInScope(varName, varScope);
            if (result != null) {
                break;
            }
        }
        //
        return result;
    }

    /**
     * This method is taken from the
     * org.netbeans.modules.bpel.properties.choosers.VariableChooserPanel;
     * It's a candidate to a common utility class.
     */
    private VariableDeclaration findVariableByNameInScope(
            String targetVarName, VariableDeclarationScope vdScope) {
        if (vdScope == null || targetVarName == null ||
                targetVarName.length() == 0) {
            return null;
        }
        //
        if (vdScope instanceof BaseScope) {
            VariableContainer vc = ((BaseScope)vdScope).getVariableContainer();
            if (vc != null) {
                VariableDeclaration[] varArr =
                        (VariableDeclaration[])vc.getVariables();
                //
                for (VariableDeclaration varDecl : varArr) {
                    String varName = varDecl.getVariableName();
                    if (targetVarName.equals(varName)) {
                        return varDecl;
                    }
                }
            }
        } else if (vdScope instanceof VariableDeclaration) {
            VariableDeclaration varDecl = (VariableDeclaration)vdScope;
            String varName = varDecl.getVariableName();
            if (targetVarName.equals(varName)) {
                return varDecl;
            }
        }
        //
        return null;
    }

    private void addResultItem(ResultType resultType, String bundleKey,
            Object... values){
        //
        String str = NbBundle.getMessage(BpelXpathValidator.class, bundleKey);
        if (values != null && values.length > 0) {
            str = MessageFormat.format(str, values);
        }
        //
        ContentElement ce = myContext.getXPathContentElement();
        String ceTypeName = etnv.getTypeName((BpelEntity)ce);
        str = ceTypeName + ": " + str;
        //
        if (myInitialExpression != null) {
            str = str + " Expression: \"" + myInitialExpression + "\"";
        }
        //
        ResultItem resultItem = new ResultItem(
                myContext.getValidator(),
                resultType,
                (BpelEntity)ce, 
                // myContext.getBpelContextActivity(),
                str);
        myContext.getVVisitor().getResultItems().add(resultItem);
    }

    private String trimAndCheck(String text) {
        if (text == null && text.length() == 0) {
            return text;
        }
        //
        String trimmed = text.trim();
        if (trimmed.length() != text.length()) {
            // Error. Whitespace characters are not allowed in the XPath
        }
        //
        return trimmed;
    }

    private String getPrefixByNsUri(String nsUri) {
        ExNamespaceContext nsContext = getNsContext();
        String nsPrefix = nsContext.getPrefix(nsUri);
        return nsPrefix;
    }

    private ExNamespaceContext getNsContext() {
        ContentElement cElement = myContext.getXPathContentElement();
        ExNamespaceContext nsContext = ((BpelEntity)cElement).getNamespaceContext();
        return nsContext;
    }

    private boolean isSchemaImported(SchemaComponent sc) {
        String soughtNamspace = sc.getModel().getEffectiveNamespace(sc);
        assert soughtNamspace != null;
        //
        BpelModel model = myContext.getBpelContextActivity().getBpelModel();
        if (model.getState() == State.VALID) {
            for (Import anImport : model.getProcess().getImports()) {
                if (Import.SCHEMA_IMPORT_TYPE.equals(anImport.getImportType())) {
                    if (soughtNamspace.equals(anImport.getNamespace())) {
                        return true;
                    }
                }
            }
        }
        //
        return false;
    }
}
