/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.bpel.model.impl.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.apache.commons.jxpath.ri.compiler.VariableReference;
import org.netbeans.modules.bpel.model.api.Variable;
import org.netbeans.modules.xml.xam.Named;
import org.netbeans.modules.xml.xpath.AbstractXPathModelHelper;
import org.netbeans.modules.xml.xpath.LocationStep;
import org.netbeans.modules.xml.xpath.XPathCoreFunction;
import org.netbeans.modules.xml.xpath.XPathCoreOperation;
import org.netbeans.modules.xml.xpath.XPathException;
import org.netbeans.modules.xml.xpath.XPathExpression;
import org.netbeans.modules.xml.xpath.XPathExpressionPath;
import org.netbeans.modules.xml.xpath.XPathExtensionFunction;
import org.netbeans.modules.xml.xpath.XPathLocationPath;
import org.netbeans.modules.xml.xpath.XPathModel;
import org.netbeans.modules.xml.xpath.XPathNumericLiteral;
import org.netbeans.modules.xml.xpath.XPathOperationOrFuntion;
import org.netbeans.modules.xml.xpath.XPathPredicateExpression;
import org.netbeans.modules.xml.xpath.XPathStringLiteral;
import org.netbeans.modules.xml.xpath.XPathVariableReference;
import org.netbeans.modules.xml.xpath.visitor.AbstractXPathVisitor;
import org.netbeans.modules.xml.xpath.visitor.XPathVisitor;

/**
 * @author ads
 */
public final class ExpressionUpdater {
    
    public static class ExpressionException extends Exception {

        private static final long serialVersionUID = -6309073089869606561L;

        /**
         * {@inheritDoc}
         */
        public ExpressionException() {
            super();
        }

        /**
         * {@inheritDoc}
         */
        public ExpressionException( String message ) {
            super(message);
        }

        /**
         * {@inheritDoc}
         */
        public ExpressionException( String message, Throwable cause ) {
            super(message, cause);
        }

        /**
         * {@inheritDoc}
         */
        public ExpressionException( Throwable cause ) {
            super(cause);
        }
        
    }
    
    public static class InvalidExpressionException extends ExpressionException {

        private static final long serialVersionUID = -461547631006192178L;

        /**
         * {@inheritDoc}
         */
        public InvalidExpressionException() {
            super();
        }

        /**
         * {@inheritDoc}
         */
        public InvalidExpressionException( String message ) {
            super(message);
        }

        /**
         * {@inheritDoc}
         */
        public InvalidExpressionException( String message, Throwable cause ) {
            super(message, cause);
        }

        /**
         * {@inheritDoc}
         */
        public InvalidExpressionException( Throwable cause ) {
            super(cause);
        }
        
    }
    
    private ExpressionUpdater(){
        myFactories.add( new VariableReferenceFactory() );
        //myFactories.add( new PartReferenceFactory() );
    }
    
    public static ExpressionUpdater getInstance(){
        return INSTANCE;
    }

    /**
     * Method returns true if <code>component</code> is present in 
     * <code>expression</code>.
     * @param expression Subject expression.
     * @param component Component that will be trying to find.
     * @return true if component is found in expression.
     */
    public boolean isPresent( String expression, Named component ) {
        if ( expression == null || expression.length() == 0 ){
            return false;
        }
        for( RefactoringReferenceFactory factory : myFactories){
            if ( factory.isApplicable( component )){
                return factory.isPresent( expression , component );
            }
        }
        return false;
    }

    /**
     * Update entrance of <code>component</code> in <code>expression</code>
     * with new <code>component</code> name.
     * @param expression Subject expression.
     * @param component Component that will be trying to find for update.
     * @param newName New name of component.
     * @return Updated expression or null if it was not updated.
     */
    public String update( String expression, Named component, String newName ) {
        if ( expression == null || expression.length() == 0 ){
            return null;
        }
        for( RefactoringReferenceFactory factory : myFactories){
            if ( factory.isApplicable( component )){
                return factory.update( expression , component , newName );
            }
        }
        return null;
    }
    /**
     * @param expression Subject expression.
     * @return collection names of found variables.
     */
    @SuppressWarnings("unchecked")
    public Collection<String> getUsedVariables( String expression ) {
        if ( expression == null || expression.length() == 0 ){
            return Collections.EMPTY_LIST;
        }
        else {
            try {
                XPathModel model = AbstractXPathModelHelper.getInstance().newXPathModel();
                XPathExpression exp = model.parseExpression( expression );
                FindVaribleVisitor visitor = new FindVaribleVisitor();
                exp.accept( visitor );
                return visitor.getVariables();
            }
            catch (XPathException e) {
                return Collections.EMPTY_LIST;
            }
        }
    }
    
    private static ExpressionUpdater INSTANCE = new ExpressionUpdater(); 
    
    private Set<RefactoringReferenceFactory> myFactories = 
        new HashSet<RefactoringReferenceFactory>();

}

interface RefactoringReferenceFactory {
    
    boolean isApplicable( Named component );
    
    boolean isPresent(String expression, Named component);
    
    String update( String expression, Named component, String newName );

}

class VariableReferenceFactory implements RefactoringReferenceFactory {

    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.impl.RefactoringReferenceFactory#isApplicable(org.netbeans.modules.xml.xam.Named)
     */
    public boolean isApplicable( Named component ) {
        return component instanceof Variable || component instanceof Named;//todo
    }

    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.impl.RefactoringReferenceFactory#isPresent(java.lang.String, org.netbeans.modules.xml.xam.Named)
     */
    public boolean isPresent( String expression, Named component ) {
        XPathModel model = AbstractXPathModelHelper.getInstance().newXPathModel();
        if ( expression == null || component.getName() == null ){
            return false;
        }
        try {
            XPathExpression exp = model.parseExpression( expression );

            FindVaribleVisitor visitor = 
                new FindVaribleVisitor( component.getName() ); 
            
            exp.accept( visitor );
            
            return visitor.isFound(); 
        }
        catch (XPathException e) {
            // in the case when we cannot parse expression we return false
            return false;
        }
    }

    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.impl.RefactoringReferenceFactory#update(java.lang.String, org.netbeans.modules.xml.xam.Named, java.lang.String)
     */
    public String update( String expression, Named component, String newName ) {
        XPathModel model = AbstractXPathModelHelper.getInstance().newXPathModel();
        if ( expression == null || component.getName() == null ){
            return null;
        }
        try {
            XPathExpression exp = model.parseExpression( expression );
            UpdateVaribleVisitor visitor = 
                new UpdateVaribleVisitor( component.getName() , newName , exp ); 
            exp.accept( visitor );
            // FIX for #IZ80076
            String ret = visitor.getExpressionString();
            if ( expression.trim().equals( ret ) ){
                // we don't want to update not changed expression.
                return null;
            }
            return ret;
        }
        catch (XPathException e) {
            // in the case when we cannot parse expression we return false
            return null;
        }
    }
    
}

/**
 * It determine presence for either just variable or variable and part in expression. 
 * 
 * @author ads
 *
 */
class FindVaribleVisitor extends XPathVisitorAdaptor {
    
    /**
     * Visitor for finding all alvailible variables. 
     */
    FindVaribleVisitor( ){
        myFoundVars = new LinkedList<String>();
    }
    
    /**
     * Visitor for finding just variable with specified name.
     */
    FindVaribleVisitor( String name ){
        this();
        myName = name;
    }
    
    /**
     * Visitor for finding variable and part . 
     */
    FindVaribleVisitor( String varName , String partName) {
        this( varName );
        assert partName!= null;
        myPartName = partName;
    }
        
    @Override
    public void visit(XPathVariableReference variableRef ) {
        QName qName = variableRef.getVariableName();
        if ( qName != null   
                //&& BpelEntity.BUSINESS_PROCESS_NS_URI.equals( qName.getNamespaceURI()) TODO !
                )
        {
            String local = qName.getLocalPart();

            if ( local == null ){
                return;
            }
            int index = local.indexOf("."); 
            /*
             * trying to devide variable in two part : bpel variable name and part.
             * ( the part could be absent )
             */ 

            if ( index <0 ){
                myFoundVars.add( local );           // FIX for IZ79861
                isFound = local.equals( myName );
                if ( isFound && myPartName != null) {
                    isFound = false; // string doesn't contain "." so there is no part
                }
            }
            else {
                String varName = local.substring( 0, index );
                myFoundVars.add( varName );         // FIX for IZ79861
                String part =(index<(local.length()-1)) ? 
                        local.substring( index+1 ):null;
                isFound = varName.equals( myName );
                
                if ( isFound && myPartName != null) {
                    isFound = myPartName.equals(part); 
                }
            }
        }
    }
    
    public Collection<String> getVariables() {
        return myFoundVars;
    }
    
    boolean isFound(){
        return isFound;
    }
    
    private String myName;
    
    private String myPartName;
    
    private boolean isFound; 
    
    private Collection<String> myFoundVars;
}

/**
 * It updates expression with new value for either just variable or part with 
 *  specified variable. 
 * @author ads
 *
 */
class UpdateVaribleVisitor extends AbstractXPathVisitor {
    
    UpdateVaribleVisitor( String name , String newName , 
            XPathExpression expression )
    {
        myName = name;
        myNewName = newName;
        myMap = new IdentityHashMap<XPathExpression,XPathExpression>();
        myExpression = expression;
        myUpdater = new ExpressionUpdaterVisitor( myMap , expression );
    }
    
    UpdateVaribleVisitor( String varName , String partName, String newName , 
            XPathExpression expression ) 
    {
        this( varName , newName , expression );
        assert partName!= null;
        myPartName = partName;
    }
    
    public void visit(LocationStep locationStep) {
        XPathPredicateExpression[] expressions = locationStep.getPredicates();
        if ( expressions!= null ){
            for (XPathPredicateExpression expression : expressions) {
                expression.accept( this );
            }
        }
        updateExpression( locationStep );
    }

    public void visit(XPathCoreFunction coreFunction) {
        visitChildren( coreFunction );
        updateExpression( coreFunction );
    }

    public void visit(XPathCoreOperation coreOperation) {
        visitChildren( coreOperation );
        updateExpression( coreOperation );
    }

    public void visit(XPathExpressionPath expressionPath) {
        XPathExpression expression = expressionPath.getRootExpression();
        if ( !expressionPath.equals( expression ) ) {
            expression.accept( this );
        }
        LocationStep[] steps = expressionPath.getSteps();
        if ( steps != null ){
            for (LocationStep step : steps) {
                step.accept( this );
            }
        }
        updateExpression( expressionPath );
    }

    public void visit(XPathExtensionFunction extensionFunction) {
        visitChildren( extensionFunction );
        updateExpression( extensionFunction );
    }

    public void visit(XPathLocationPath locationPath) {
        LocationStep[] steps = locationPath.getSteps();
        if ( steps != null ){
            for (LocationStep step : steps) {
                step.accept( this );
            }
        }
        updateExpression( locationPath );
    }
    
    @Override
    public void visit(XPathVariableReference variableRef ) {
        QName qName = variableRef.getVariableName();
        if ( qName != null   
                //&& BpelEntity.BUSINESS_PROCESS_NS_URI.equals( qName.getNamespaceURI()) TODO !
                )
        {
            String local = qName.getLocalPart();
            if ( local == null ){
                return;
            }
            int index = local.indexOf("."); 
            /*
             * trying to devide variable in two part : bpel variable name and part.
             * ( the part could be absent )
             */ 
            if ( index <0 ){
                if ( local.equals( myName ) && myPartName == null) {
                    XPathVariableReference newRef = AbstractXPathModelHelper.
                        getInstance().newXPathVariableReference( 
                                new VariableReference( 
                                        new org.apache.commons.jxpath.ri.QName(
                                                myNewName)));
                    myMap.put( variableRef , newRef );
                }
            }
            else {
                String varName = local.substring( 0, index );
                String part =(index<(local.length()-1)) ? 
                        local.substring( index+1 ):null;
                
                if ( !varName.equals( myName ) ){
                    return;
                }
                
                String newName;
                if ( myPartName != null) {
                    newName = myName+"."+myNewName;
                }
                else {
                    newName = myNewName+"."+part;
                }
                XPathVariableReference newRef = AbstractXPathModelHelper.
                getInstance().newXPathVariableReference( 
                        new VariableReference( 
                                new org.apache.commons.jxpath.ri.QName(
                                        newName)));
                myMap.put( variableRef , newRef );
            }
        }
        updateExpression( variableRef );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xpath.visitor.XPathVisitor#visit(org.netbeans.modules.xml.xpath.XPathStringLiteral)
     */
    public void visit( XPathStringLiteral stringLiteral ) {
        updateExpression( stringLiteral );        
    }

    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xpath.visitor.XPathVisitor#visit(org.netbeans.modules.xml.xpath.XPathNumericLiteral)
     */
    public void visit( XPathNumericLiteral numericLiteral ) {
        updateExpression( numericLiteral );         
    }

    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xpath.visitor.XPathVisitor#visit(org.netbeans.modules.xml.xpath.XPathPredicateExpression)
     */
    public void visit( XPathPredicateExpression predicateExpression ) {
        updateExpression( predicateExpression );         
    }
    
    // FIX for #IZ80076
    String getExpressionString() {
        if ( myExpression == null ) {
            return null;
        }
        return myExpression.getExpressionString();
    }
    
    // FIX for #IZ80076
    private void updateExpression( XPathExpression expression){
        expression.accept( myUpdater );
        myExpression = myUpdater.getExpression();
    }
    
    private String myName;
    
    private String myPartName;
    
    private String myNewName;
    
    private ExpressionUpdaterVisitor myUpdater;
    
    private Map<XPathExpression,XPathExpression> myMap;
    
    private XPathExpression myExpression;

}

/**
 * This is generic children updater.
 * It is constracted with Map that contains in key old expression, 
 * in value - new expression . This visitor will change 
 * old expression in parent to new expression and remove key from Map. 
 * 
 * @author ads
 *
 */
class ExpressionUpdaterVisitor implements XPathVisitor {
    
    ExpressionUpdaterVisitor( Map<XPathExpression,XPathExpression> oldNewMap ,
            XPathExpression expression )
    {
        myMap = oldNewMap;
        myExpression = expression;
    }
    
    public void visit(LocationStep locationStep) {
        updateExpression( locationStep );
        XPathPredicateExpression[] expressions = locationStep.getPredicates();
        boolean hasChanges = false;
        if ( expressions!= null ){
            int i = 0 ;
            for (XPathPredicateExpression expression : expressions) {
                XPathExpression expr = myMap.remove( expression );
                if ( expr!= null ){
                    hasChanges = true;
                    expressions[i]=(XPathPredicateExpression)expr;
                }
                i++;
            }
        }
        if ( hasChanges ){
            locationStep.setPredicates( expressions );
        }
    }
    
    public void visit(XPathCoreFunction coreFunction) {
        updateExpression( coreFunction );
        visitChildren( coreFunction );
    }
    
    public void visit(XPathCoreOperation coreOperation) {
        updateExpression( coreOperation );
        visitChildren( coreOperation );
    }
    
    public void visit(XPathExpressionPath expressionPath) {
        updateExpression( expressionPath );
        XPathExpression expression = expressionPath.getRootExpression();
        XPathExpression expr =  myMap.remove( expression );
        if ( expr!=null ){
            expressionPath.setRootExpression( expr );
        }

        boolean hasChanges = false;
        LocationStep[] steps = expressionPath.getSteps();
        if ( steps != null ){
            int i = 0 ;
            for (LocationStep step : steps) {
                expr = myMap.remove( step );
                if ( expr!= null ){
                    hasChanges = true;
                    steps[i]= (LocationStep)expr;
                }
                i++;
            }
        }
        if ( hasChanges ){
            expressionPath.setSteps( steps );
        }
    }

    public void visit(XPathExtensionFunction extensionFunction) {
        updateExpression( extensionFunction );
        visitChildren( extensionFunction );
    }

    public void visit(XPathLocationPath locationPath) {
        updateExpression( locationPath );
        LocationStep[] steps = locationPath.getSteps();
        boolean hasChanges = false;
        if ( steps != null ){
            int i = 0 ;
            for (LocationStep step : steps) {
                XPathExpression expr = myMap.remove( step );
                if ( expr!= null ){
                    hasChanges = true;
                    steps[i]= (LocationStep)expr;
                }
                i++;
            }
        }
        if ( hasChanges ){
            locationPath.setSteps( steps );
        }
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xpath.visitor.XPathVisitor#visit(org.netbeans.modules.xml.xpath.XPathStringLiteral)
     */
    public void visit( XPathStringLiteral stringLiteral ) {
        /*XPathExpression expression = myMap.remove( stringLiteral );
        if  (expression != null ) {
            stringLiteral.setValue( ((XPathStringLiteral)expression).getValue() );
        }*/
        updateExpression( stringLiteral );
    }

    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xpath.visitor.XPathVisitor#visit(org.netbeans.modules.xml.xpath.XPathNumericLiteral)
     */
    public void visit( XPathNumericLiteral numericLiteral ) {
        /*XPathExpression expression = myMap.remove( numericLiteral );
        if  (expression != null ) {
            numericLiteral.setValue( ((XPathNumericLiteral)expression).getValue() );
        }*/
        updateExpression( numericLiteral );
    }

    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xpath.visitor.XPathVisitor#visit(org.netbeans.modules.xml.xpath.XPathVariableReference)
     */
    public void visit( XPathVariableReference variableReference ) {
        updateExpression(variableReference);
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xpath.visitor.XPathVisitor#visit(org.netbeans.modules.xml.xpath.XPathPredicateExpression)
     */
    public void visit( XPathPredicateExpression predicateExpression ) {
        updateExpression( predicateExpression );
    }
    
    XPathExpression getExpression() {
        return myExpression;
    }
    
    protected void visitChildren(XPathOperationOrFuntion expr) {
         Collection children = expr.getChildren();
         Collection<XPathExpression> newChildren = null;
         boolean hasChanges = false;
         if(children != null) {
             newChildren = new ArrayList<XPathExpression>( children.size() );
             Iterator it = children.iterator();
             while(it.hasNext()) {
                 XPathExpression child = (XPathExpression) it.next();
                 XPathExpression expression = myMap.remove( child );
                 if ( expression!= null ){
                     hasChanges = true;
                     newChildren.add(expression); // Fix for IZ#80079
                 }
                 else {
                     newChildren.add( child );
                 }
             }
         }
         if ( hasChanges ){
             expr.clearChildren();
             for( XPathExpression expression : newChildren ){
                 expr.addChild( expression );
             }
         }
    }
    
    private void updateExpression( XPathExpression expression ) {
        XPathExpression updated = myMap.get( expression );
        /*
         *  only when ALL expression should be changed ( because
         *  it doesn't have mutation method ) we replace
         *  this expression to new from map. This could 
         *  be applicable ONLY for expression that 
         *  is oroginal expression from which we start.
         *  Any subexpression will be handled as child in
         *  appropriate container.
         *  This is FIX for #IZ80076
         */ 
        if ( updated != null && myExpression == expression ) {
            myExpression = updated;
        }
    }
    
    private Map<XPathExpression,XPathExpression> myMap = 
        new IdentityHashMap<XPathExpression,XPathExpression>();
    
    // this need when expression itself needs to be changed.
    private XPathExpression myExpression;

}
