/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.cnd.modelimpl.csm;

import antlr.collections.AST;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.netbeans.modules.cnd.api.model.*;
import org.netbeans.modules.cnd.api.model.util.CsmKindUtilities;
import org.netbeans.modules.cnd.apt.utils.TextCache;
import org.netbeans.modules.cnd.modelimpl.csm.core.*;
import org.netbeans.modules.cnd.modelimpl.debug.TraceFlags;
import org.netbeans.modules.cnd.modelimpl.repository.PersistentUtils;
import org.netbeans.modules.cnd.modelimpl.repository.RepositoryUtils;
import org.netbeans.modules.cnd.modelimpl.textcache.QualifiedNameCache;
import org.netbeans.modules.cnd.modelimpl.uid.UIDCsmConverter;
import org.netbeans.modules.cnd.modelimpl.uid.UIDObjectFactory;

/**
 * Common ancestor for ClassImpl and EnumImpl
 * @author Vladimir Kvashin
 */
public abstract class ClassEnumBase<T> extends OffsetableDeclarationBase<T> implements Disposable, CsmCompoundClassifier<T>, CsmMember<T> {
    
    private final String name;
    
    private /*final*/ String qualifiedName;
    
    // only one of scopeRef/scopeAccessor must be used (based on USE_REPOSITORY)
    private CsmScope scopeRef;// can be set in onDispose or contstructor only
    private CsmUID<CsmScope> scopeUID;
    
    private boolean isValid = true;

    private boolean _static = false;
    private CsmVisibility visibility = CsmVisibility.PRIVATE;
    private final List<CsmUID<CsmTypedef>> enclosingTypdefs = Collections.synchronizedList(new ArrayList<CsmUID<CsmTypedef>>());
    
    protected ClassEnumBase(String name, CsmFile file, AST ast) {
        super(ast, file);
        this.name = (name == null) ? "" : name;
    }
    
    public String getName() {
        return name;
    }
    
    /**
     * Initialization method. 
     * Should be called immediately after object creation. 
     *
     * Descendants may override it; in this case it's a descendant's responsibility
     * to call super.init()
     */
    protected void init(CsmScope scope, AST ast) {
        if (TraceFlags.USE_REPOSITORY && TraceFlags.UID_CONTAINER_MARKER) {
            this.scopeUID = UIDCsmConverter.scopeToUID(scope);
	    assert (this.scopeUID != null || scope == null) : "null UID for class scope " + scope;
            this.scopeRef = null;
        } else {
            this.scopeRef = scope;
            this.scopeUID = null;
        }   
	
	String qualifiedNamePostfix = getQualifiedNamePostfix();
        if(  CsmKindUtilities.isNamespace(scope) ) {
            qualifiedName = Utils.getQualifiedName(qualifiedNamePostfix, (CsmNamespace) scope);
        }
	else if( CsmKindUtilities.isClass(scope) ) {
            qualifiedName = ((CsmClass) scope).getQualifiedName() + "::" + qualifiedNamePostfix; // NOI18N
	}
        else  {
	    qualifiedName = qualifiedNamePostfix;
        }
        // can't register here, because descendant class' constructor hasn't yet finished!
        // so registering is a descendant class' responsibility
    }
    
    abstract public Kind getKind();
    
    protected void register(CsmScope scope) {
        if (TraceFlags.USE_REPOSITORY) {
            RepositoryUtils.put(this);
        }
        if( ProjectBase.canRegisterDeclaration(this) ) {
            registerInProject();
	    
	    
	    if( getContainingClass() == null ) {
		if(  CsmKindUtilities.isNamespace(scope) ) {
		    ((NamespaceImpl) scope).addDeclaration(this);
		}
	    }
        }
    }
    
    private void registerInProject() {
        ((ProjectBase) getContainingFile().getProject()).registerDeclaration(this);
    }
    
    private void unregisterInProject() {
        ((ProjectBase) getContainingFile().getProject()).unregisterDeclaration(this);
        this.cleanUID();
    }
    
    public NamespaceImpl getContainingNamespaceImpl() {
	CsmScope scope = getScope();
	return (scope instanceof NamespaceImpl) ? (NamespaceImpl) scope : null;
    }
    
    public String getQualifiedName() {
        return qualifiedName;
    }

  
    public CsmScope getScope() {
        CsmScope scope = this.scopeRef;
        if (scope == null) {
            if (TraceFlags.USE_REPOSITORY) {
                scope = UIDCsmConverter.UIDtoScope(this.scopeUID);
                assert (scope != null || this.scopeUID == null) : "null object for UID " + this.scopeUID;
            }
        }
        return scope;
    }

    public void dispose() {
        super.dispose();
        onDispose();
        if (getContainingNamespaceImpl() != null) {
            getContainingNamespaceImpl().removeDeclaration(this);
        }
	unregisterInProject();
        isValid = false;
    }
        
    private void onDispose() {
        if (TraceFlags.RESTORE_CONTAINER_FROM_UID) {
            // restore container from it's UID
            this.scopeRef = UIDCsmConverter.UIDtoScope(this.scopeUID);
            assert (this.scopeRef != null || this.scopeUID == null) : "empty scope for UID " + this.scopeUID;
        }
    }
    
    public boolean isValid() {
        return isValid && getContainingFile().isValid();
    }

    public CsmClass getContainingClass() {
	CsmScope scope = getScope();
	return CsmKindUtilities.isClass(scope) ? (CsmClass) scope : null;
    }
    
//    private void setContainingClass(CsmClass cls) {
//        containingClassRef = cls;
//        qualifiedName = cls.getQualifiedName() + "::" + getName();
//    }
    
    public CsmVisibility getVisibility() {
        return visibility;
    }
    
    public void setVisibility(CsmVisibility visibility) {
        this.visibility = visibility;
    }
    
    public boolean isStatic() {
        return _static;
    }
    
    public void setStatic(boolean _static) {
        this._static = _static;
    }

    ////////////////////////////////////////////////////////////////////////////
    // impl of SelfPersistent
    
    public void write(DataOutput output) throws IOException {
        super.write(output); 
        output.writeBoolean(this.isValid);
	
        assert this.name != null;
        output.writeUTF(this.name);
	
        assert this.qualifiedName != null;
        output.writeUTF(this.qualifiedName);
        
	UIDObjectFactory.getDefaultFactory().writeUID(this.scopeUID, output);
        
        output.writeBoolean(this._static);
	
        assert this.visibility != null;
        PersistentUtils.writeVisibility(this.visibility, output);
        
        // write UID for unnamed classifier
        if (getName().length() == 0) {
            super.writeUID(output);
        }
        UIDObjectFactory.getDefaultFactory().writeUIDCollection(enclosingTypdefs, output, true);
    }  
    
    protected ClassEnumBase(DataInput input) throws IOException {
        super(input);
        this.isValid = input.readBoolean();
	
        this.name = TextCache.getString(input.readUTF());
        assert this.name != null;
	
        this.qualifiedName = QualifiedNameCache.getString(input.readUTF());
        assert this.qualifiedName != null;
	
        this.scopeUID = UIDObjectFactory.getDefaultFactory().readUID(input);
        this.scopeRef = null;
	
        this._static = input.readBoolean();
	
        this.visibility = PersistentUtils.readVisibility(input);
        assert this.visibility != null;
                
        assert TraceFlags.USE_REPOSITORY;
        
        // restore UID for unnamed classifier
        if (getName().length() == 0) {
            super.readUID(input);
        }
        UIDObjectFactory.getDefaultFactory().readUIDCollection(enclosingTypdefs, input);
    }

    public Collection<CsmTypedef> getEnclosingTypedefs() {
        return UIDCsmConverter.UIDsToDeclarations(enclosingTypdefs);
    }

    public void addEnclosingTypedef(CsmTypedef typedef) {
        enclosingTypdefs.add(typedef.getUID());
    }
}
