/**************************************************************
*   
*   Creation Date: <1999-02-03 19:30:07 samuel>
*   Time-stamp: <2001/09/29 22:34:24 samuel>
*   
*	<main.c>
*	
*	
*   
*   Copyright (C) 1997, 1999, 2000, 2001 Samuel Rydh (samuel@ibrium.se)
*   
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation
*   
**************************************************************/

#include "mol_config.h"

#include <sys/mman.h>
#include <unistd.h>
#include <signal.h>
#include <pthread.h>
#include <semaphore.h>

#include "memory.h"
#include "wrapper.h"
#include "promif.h"
#include "io.h"
#include "res_manager.h" 
#include "debugger.h"
#include "thread.h"
#include "timer.h"
#include "version.h"
#include "os_interface.h"
#include "booter.h"
#include "async.h"
#include "session.h"
#include "keycodes.h"
#include "molcpu.h"
#include "rvec.h"
#include "../drivers/include/console.h"
#include "../drivers/include/osi_driver.h"

#include <ucontext.h>

static void	exit_hook( void );
static void	do_cleanup( void );
static void	set_signals( void );
static void	map_mregs( void );
static void	unmap_mregs( void );
static int	create_lockfile( void );
static void	remove_lockfile( void );
static int 	open_session( void );


/* GLOBALS */
int 		in_security_mode = 1;
ulong		g_session_magic;
mac_regs_t	*mregs = NULL;

_syscall5( int, multiplexer, int, selector, int, arg1, int, arg2, int, arg3, int, arg4 );

/************************************************************************/
/*	M A I N								*/
/************************************************************************/

int
main( int argc, char **argv )
{
	int kvers;

	printm("Version %d.%d.%d \n", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL );
	/* printm("(C) 1997-2000 Samuel Rydh, <samuel@ibrium.se>\n"); */

	if( getuid() == 0 )
		in_security_mode = 0;

	/* If we are not root, disable "unsecure features"
	 * like the debugger and force the use of the /etc/molrc file.
	 */
	if( seteuid(0) != 0 ){
                fprintf( stderr,"Mac-on-Linux must be run as root!\n");
                exit(1);
	}
	/* printm("in_security_mode %d\n", in_security_mode ); */

	/* 
	 * It is necessary to run in a new session (for the console/FB stuff).
	 * The fork is necessary since setsid() usually fails otherwise.
	 */
	if( fork() != 0 )
		return 0;
	if( setsid() < 0 )
		perrorm("setsid failed!\n");

	/* Make sure the kernel module is up to date... */
	kvers = _get_mol_mod_version();
	if( kvers < 0 ){
		fprintf( stderr, "Error: The MOL kernel module is not loaded\n" );
		exit(1);
	}
	if( kvers != MOL_VERSION ){
		printm("The MOL kernel module version %d.%d.%d does not match the binary.\n",
		       (kvers>>16), (kvers>>8)&0xff, kvers&0xff );
		printm("Please unload the kernel module (rmmod mol) and try again\n");
		exit(1);
	}
	if( is_a_601() ){
		printm("Mac-on-Linux no longer supports the PowerPC 601 processor\n");
		exit(1);
	}

	/* Initialize res_mamager (and g_session_id) */
	res_manager_init( 1, argc, argv );

	/* Initialize session */
	if( create_lockfile() )
		exit(1);
	if( open_session() ) {
		remove_lockfile();
		exit(1);
	}

	atexit( exit_hook );

	/* We must be careful with dependencies here. */
	map_mregs();			/* We want mregs early on */

	set_signals();
	threadpool_init();		/* Provides threads */
	async_init();			/* Provides async IO capabilities */
	debugger_init();		/* Provides logging and debugger */
	mainloop_init();		/* Provides set_rvector */
	os_interface_init();		/* Provides register_osi_call */

	session_init();
	booter_init();			/* Provides gPE.xxx */

	promif_init();			/* Most drivers depend on this */
	timer_init();			/* Provides timer services */
	osi_driver_services_init();	/* Provides osi driver services */

	mem_init();			/* Memory subsystem */
	molcpu_init();			/* CPU / emulation */
	ioports_init();			/* All drivers */
	booter_startup();		/* Boot loader */

	session_is_loaded();

	if( !is_main_thread() )
		printm("**************** is_main_thread not working ****************\n");
	mainloop_start();

	/* cleanup through exit_hook */
	return 0;
}

static void 
do_cleanup( void ) 
{
	printm("cleaning up...\n");

	console_make_safe();

	ioports_cleanup();
	osi_driver_services_cleanup();
	timer_cleanup();
	molcpu_cleanup();
	booter_cleanup();
	mem_cleanup();
	promif_cleanup();

	debugger_cleanup();
	async_cleanup();
	threadpool_cleanup();
	session_cleanup();

	mainloop_cleanup();
	os_interface_cleanup();
	res_manager_cleanup();

	unmap_mregs();
        _kernel_cleanup();

}

static void 
exit_hook( void )
{
	if( !is_main_thread() ) {
		/* Umm... one of the sub-threads has probably segfaulted.
		 * We better kill -9 the main thread and then tries cleanup
		 * things from this thread.
		 */ 
		printm("Exiting due to a crashed thread\n");
		pthread_kill(get_main_th(), 9 );
	}

	/* reset signal handlers */
	signal( SIGILL, SIG_DFL );
	signal( SIGSEGV, SIG_DFL );
	signal( SIGINT, SIG_DFL );
	signal( SIGTERM, SIG_DFL );
	signal( SIGSTKFLT, SIG_DFL );
	
	do_cleanup();

	remove_lockfile();

	printm( "DONE\n");
	/* execlp("/etc/mol.sh", "mol.sh", "stop", NULL ); */
}

static void 
signal_handler( int sig_num, siginfo_t *sinfo, struct ucontext *puc, ulong rt_sf )
{
	static time_t warntime = 0;
	time_t cur;

	if( sig_num != SIGINT ) {
		async_print("***** SIGNAL %d [%s] in thread %s *****\n", 
			    sig_num, strsignal(sig_num), get_thread_name() );
	}
	switch( sig_num ) {
	case SIGINT:
		if( !is_main_thread() )
			return;

		time( &cur );
		if( !warntime || cur-warntime>1 ) {
			printm("Signal INT\nOne more to kill emulator\n");
			warntime = cur;
			/* break emulation */
			stop_emulation();      
			return;
		}
		break;
	case SIGPIPE:
		/* Usually the debugger... */
		return;
		break;
	case SIGHUP:
		async_print("HUP: _pid %d, _uid %d. Last RVEC %ld\n", 
		       sinfo->si_pid, sinfo->si_uid, mregs->dbg_last_rvec );
		// if( !is_main_thread() )
		//	return;
		break;
	case SIGSEGV:
	case SIGILL:
	case SIGBUS: {
		struct pt_regs *regs;
		async_print("   si_signo = %d, si_errno %d, si_code %08X\n", 
		       sinfo->si_signo, sinfo->si_errno, sinfo->si_code );
		async_print("   si_addr %p. Last RVEC: 0x%lx (%ld)\n", 
		       sinfo->si_addr, mregs->dbg_last_rvec, (mregs->dbg_last_rvec & RVEC_MASK) );
		regs = puc->uc_mcontext.regs;
		async_print("   NIP %08lx\n", regs->nip );
		break;
	}
	default:
		break;
	}
	exit(1);
}

static void 
set_signals( void )
{
	struct sigaction act;
	act.sa_handler = (void*)signal_handler;
	sigemptyset(&act.sa_mask);
	act.sa_flags = SA_RESTART | SA_SIGINFO;
	act.sa_restorer = NULL;
	
	sigaction( SIGINT, &act, NULL );
	sigaction( SIGHUP, &act, NULL );
	sigaction( SIGILL, &act, NULL );
	sigaction( SIGBUS, &act, NULL );
	sigaction( SIGSEGV, &act, NULL );
	sigaction( SIGTERM, &act, NULL );
	sigaction( SIGSTKFLT, &act, NULL );
	sigaction( SIGTRAP, &act, NULL );
	sigaction( SIGPIPE, &act, NULL );
	/* NOTE: SIGUSR1 and SIGUSR2 are reserved by pthread (still true?). */
}


/************************************************************************/
/*	Lockfile							*/
/************************************************************************/

static int
create_lockfile( void )
{
	FILE *f;
	char *name = get_lockfile();

	printm("Session %d. Lockfile '%s'\n", g_session_id, name );
	
 	if( (f=fopen( name, "r")) != NULL ){
		fclose(f);
		printm("MOL lockfile detected! Make sure MOL isn't running\n"		
		       "before deleting the lockfile /var/lock/mol\n");
		return 1;
	}
	if( (f=fopen( name, "w+") ) == NULL ) {
		printm("Warning: could not create the lockfile %s\n", name);
	} else {
		fprintf(f,"%d\n",getpid() );
		fclose( f );
	}
	return 0;
}

static void
remove_lockfile( void )
{
	FILE *f;
	char *name = get_lockfile();
	
	if( (f=fopen(name, "r")) != NULL ){
		ulong pid;
		fscanf(f, "%ld", &pid );
		fclose(f);
		if( pid == getpid() )
			unlink( name );
	}
}



/************************************************************************/
/*	Create MOL session						*/
/************************************************************************/

static int 
open_session( void )
{
	struct timeval tv;
	int err;

	gettimeofday( &tv, NULL );
	srandom( tv.tv_usec );
	g_session_magic = random();


	if( (err=_kernel_init( g_session_magic )) ) {
		if( err == errSessionInUse ) {
			printm("---> Removing unreleased MOL session\n");
			_kernel_cleanup();
			err = _kernel_init( g_session_magic );
		}
		if( err == errSessionOutOfBounds )
			printm("The session number (%d) is out of bounds\n", g_session_id );
		else if( err )
			printm("_kernel_init returned an error %d\n", err);
	}
	return err;	
}



/************************************************************************/
/*	Mregs Initialization						*/
/************************************************************************/

static void
map_mregs( void )
{
	mregs = (mac_regs_t*) map_phys_mem( 0, _get_mregs_phys(), sizeof(mac_regs_t), 
					    PROT_READ | PROT_WRITE );
	if( !mregs ) {
		perrorm("map_kmem:");
		exit(1);
	}
	memset( mregs, 0, sizeof(mac_regs_t) );
}

static void
unmap_mregs( void )
{
	if( mregs )
		unmap_mem( (char*)mregs, sizeof(mac_regs_t) );
}
