package com.ibm.bsf.engines.jacl;

import java.util.*;
import java.io.*;

import tcl.lang.*;

import com.ibm.bsf.*;
import com.ibm.bsf.util.*;

/**
 * This is the interface to Scriptics's Jacl (Tcl) from the
 * Bean Scripting Framework.
 * <p>
 *
 * @author   Sanjiva Weerawarana
 */

public class JaclEngine extends BSFEngineImpl {
  /* the Jacl interpretor object */
  private Interp interp;

  /**
   * Initialize the engine.
   */
  public void initialize (BSFManager mgr, String lang,
			  Vector declaredBeans) throws BSFException {
    super.initialize (mgr, lang, declaredBeans);

    // create interpreter
    interp = new Interp();

    // register the extension that user's can use to get at objects
    // registered by the app
    interp.createCommand ("bsf", new BSFCommand (mgr, this));

    int size = declaredBeans.size ();
    for (int i = 0; i < size; i++) {
      declareBean ((BSFDeclaredBean) declaredBeans.elementAt (i));
    }
  }

  /**
   * This is used by an application to evaluate a string containing
   * some expression.
   */
  public Object eval (String source, int lineNo, int columnNo, 
		      Object oscript) throws BSFException {
    String script = oscript.toString ();
    try {
      interp.eval (script);
      TclObject result = interp.getResult();
      Object internalRep = result.getInternalRep();

      // if the object has a corresponding Java type, unwrap it
      if (internalRep instanceof ReflectObject)
        return ReflectObject.get(interp,result);
      if (internalRep instanceof TclString)
        return result.toString();
      if (internalRep instanceof TclDouble)
        return new Double(TclDouble.get(interp,result));
      if (internalRep instanceof TclInteger)
        return new Integer(TclInteger.get(interp,result));

      return result;
    } catch (TclException e) { 
      throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
			      "error while eval'ing Jacl expression: " + 
			      interp.getResult (), e);
    }
  }

  /**
   * 
   * @param method The name of the method to call.
   * @param args an array of arguments to be
   * passed to the extension, which may be either
   * Vectors of Nodes, or Strings.
   */
  public Object call (Object obj, String method, Object[] args) 
                                                        throws BSFException {
    StringBuffer tclScript = new StringBuffer (method);
    if (args != null) {
      for( int i = 0 ; i < args.length ; i++ ) {
	tclScript.append (" ");
	tclScript.append (args[i].toString ());
      }
    }
    return eval ("<function call>", 0, 0, tclScript.toString ());
  }

  /**
   * Declare a bean
   */
  public void declareBean (BSFDeclaredBean bean) throws BSFException {
    String expr = "set " + bean.name + " [bsf lookupBean \"" + bean.name +
      "\"]";
    eval ("<declare bean>", 0, 0, expr);
  }

  /**
   * Undeclare a previously declared bean.
   */
  public void undeclareBean (BSFDeclaredBean bean) throws BSFException {
    eval ("<undeclare bean>", 0, 0, "set " + bean.name + " \"\"");
  }
}

// class used to add "bsf" command to the Jacl runtime
class BSFCommand implements Command {
  BSFManager mgr;
  BSFEngine jengine;

  BSFCommand (BSFManager mgr, BSFEngine jengine) {
    this.mgr = mgr;
    this.jengine = jengine;
  }

  public void cmdProc (Interp interp, 
		       TclObject argv[]) throws TclException {
    if (argv.length < 2) {
      interp.setResult ("invalid # of args; usage: bsf " +
        "lookupBean|registerBean|unregisterBean|addEventListener args");
      throw new TclException (TCL.ERROR);
    }

    String op = argv[1].toString ();

    if (op.equals ("lookupBean")) {
      if (argv.length != 3) {
	interp.setResult ("invalid # of args; usage: bsf " +
			  "lookupBean name-of-bean");
	throw new TclException (TCL.ERROR);
      }

      String beanName = argv[2].toString ();
      Object bean = mgr.lookupBean (beanName);
      if (bean == null) {
	interp.setResult ("unknown object: " + beanName);
	throw new TclException (TCL.ERROR);
      }
      interp.setResult (ReflectObject.newInstance (interp, bean.getClass (), 
						   bean));

    } else if (op.equals ("registerBean")) {
      if (argv.length != 4) {
	interp.setResult ("invalid # of args; usage: bsf " +
			  "registerBean name-of-bean bean");
	throw new TclException (TCL.ERROR);
      }
      mgr.registerBean (argv[2].toString (), 
			ReflectObject.get (interp, argv[3]));
      interp.setResult ("");

    } else if (op.equals ("unregisterBean")) {
      if (argv.length != 3) {
	interp.setResult ("invalid # of args; usage: bsf " +
			  "unregisterBean name-of-bean");
	throw new TclException (TCL.ERROR);
      }
      mgr.unregisterBean (argv[2].toString ());
      interp.setResult ("");

    } else if (op.equals ("addEventListener")) {
      if (argv.length != 6) {
	interp.setResult ("invalid # of args; usage: bsf " +
			  "addEventListener object event-set-name filter " +
			  "script-to-run");
	throw new TclException (TCL.ERROR);
      }
      try {
	// usage: bsf addEventListener object event-set filter script
	String filter = argv[4].toString ();
	filter = filter.equals ("") ? null : filter;
	EngineUtils.addEventListener (ReflectObject.get (interp, argv[2]),
				      argv[3].toString (), filter,
				      jengine, mgr, "<event-script>", 0, 0,
				      argv[5].toString ());
      } catch (BSFException e) {
	e.printStackTrace ();
	interp.setResult ("got BSF exception: " + e.getMessage ());
	throw new TclException (TCL.ERROR);
      }
    }
  }
}
