/***************************************************************************
 *   Copyright (C) 2003-2005 by Kevin Hessels                              *
 *   khessels@shaw.ca                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
 
#include <qcheckbox.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qregexp.h>

#include <kcombobox.h>
#include <kdebug.h>
#include <kfile.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knuminput.h>
#include <kstandarddirs.h>
#include <kurlrequester.h>
 
#include "kfoldingconfigdata.h"
#include "kfoldingsettingsdialog.h"
#include "kfoldingsettingswidget.h"

#define PREVIEW_HEIGHT 46

kfoldingSettingsDialogue::kfoldingSettingsDialogue( QWidget *parent, const char *name, kfoldingConfigData* cfg ) 
	: KDialogBase( Plain, i18n( "Settings" ), Apply | Ok | Cancel, Ok,  parent, name ),
	  _configData( cfg ),
	  _inputError( false )
{
	setIcon( SmallIcon( "kfolding" ) );

	setupSettingsWidget();
	
	connect( _settingsWidget->workingDir, SIGNAL( textChanged( const QString& ) ), this, SLOT( slotWorkingDirChanged( const QString& ) ) );
	connect( _settingsWidget->appletImage, SIGNAL( textChanged( const QString& ) ), this, SLOT( slotCheckAppletImage( const QString& ) ) );
	return;
} // kfoldingSettingsDialogue ctor


kfoldingSettingsDialogue::~kfoldingSettingsDialogue()
{
	return;
} // kfoldingSettingsDialogue dtor



void kfoldingSettingsDialogue::slotOk()
{
	slotApply();
	
	if ( !_inputError ) {
		accept();
	}
	return;
} // slotOk


void kfoldingSettingsDialogue::slotApply()
{
	if ( checkData() ) {
		// commit the changes and notify the app that the settings have been updated
		_configData->setExecutable( _settingsWidget->executable->url().stripWhiteSpace() );
		_configData->setWorkingDir( _settingsWidget->workingDir->url().stripWhiteSpace() );
		if ( !_configData->workingDir().endsWith( "/" ) )
			_configData->setWorkingDir( _configData->workingDir() + '/' );
		
		_configData->setStartOnLaunch( _settingsWidget->startOnLaunch->isChecked() );
		_configData->setRestoreState( _settingsWidget->rememberState->isChecked() );
		
		_configData->setUserName( _settingsWidget->userName->text().stripWhiteSpace() );
		_configData->setTeamNumber( _settingsWidget->teamNumber->value() );
		_configData->setClientType( _settingsWidget->clientType->currentItem() );
		_configData->setBigPackets( _settingsWidget->bigPackets->isChecked() );
		_configData->setForceASM( _settingsWidget->forceasm->isChecked() );
		_configData->setUseAdvMethods( _settingsWidget->useadvmethods->isChecked() );
		_configData->setMachineId( _settingsWidget->machineId->value() );
		
		_configData->setAppletImage( _settingsWidget->appletImage->url().stripWhiteSpace() );
		
		_configData->save();
		emit settingsChanged();
		_inputError = false;
	} else {
		_inputError = true;
	} // else
	return;
} // slotApply


void kfoldingSettingsDialogue::slotCancel()
{
	KDialogBase::slotCancel();
	return;
} // slotCancel


void kfoldingSettingsDialogue::slotWorkingDirChanged( const QString& newDir )
{
	int pos;
	QString userName, teamNumber, clientType, machineId;
	bool bigPacketsOn = false;
	
	QString workingDir = newDir;
	if ( !workingDir.endsWith( "/" ) )
		workingDir = workingDir.append( '/' );

	QFileInfo fileInfo( workingDir );
	if ( fileInfo.exists() ) {
		QString configFile = workingDir + QString::fromLatin1( "client.cfg" );
	
		fileInfo.setFile( configFile );
		if ( fileInfo.exists() ) {	
			QFile src( configFile );
			
			if ( src.open( IO_ReadOnly ) ) {
				kdDebug() << "client.cfg found. parsing" << endl;
				QRegExp rxUserName( "^username=(.*)$" );
				QRegExp rxTeamNumber( "^team=(.*)$" );
				QRegExp rxClientType( "^type=(.*)$" );
				QRegExp rxMachineId( "^machineid=(.*)$" );
				QRegExp rxBigPackets( "^bigpackets=(.*)$" );
		
				QTextStream stream( &src );
				QString line;
				while( !stream.atEnd() ) {
					line = stream.readLine().stripWhiteSpace();
					
					pos = rxUserName.search( line );
					if ( pos > -1 ) {
						userName = rxUserName.cap( 1 );
					}
					
					pos = rxTeamNumber.search( line );
					if ( pos > -1 ) {
						teamNumber = rxTeamNumber.cap( 1 );
					}
					
					pos = rxClientType.search( line );
					if ( pos > -1 ) {
						clientType = rxClientType.cap( 1 );
					}
					
					pos = rxMachineId.search( line );
					if ( pos > -1 ) {
						machineId = rxMachineId.cap( 1 );
					}
					
					pos = rxBigPackets.search( line );
					if ( pos > -1 ) {
						kdDebug() << "Found bigpackets: " << rxBigPackets.cap( 1 ) << endl;
						if ( rxBigPackets.cap( 1 ).stripWhiteSpace() == QString::fromLatin1( "yes" ) ) {
							bigPacketsOn = true;
						} else {
							bigPacketsOn = false;
						} // else
					} // if
				} // while
				src.close();
				
				int rc = KMessageBox::questionYesNo( this, 
					i18n( "An existing client.cfg file has been detected in the selected working directory.\n \
					Would you like to use the values contained in this file?\n \
					Note: The current settings will be over-written." ) );
				if ( rc == KMessageBox::Yes ) {
					_settingsWidget->userName->setText( userName );
					_settingsWidget->teamNumber->setValue( teamNumber.toInt() );
					_settingsWidget->clientType->setCurrentItem( clientType.toInt() );
					_settingsWidget->machineId->setValue( machineId.toInt() );
					_settingsWidget->bigPackets->setChecked( bigPacketsOn );
				}
			} // if
		} // if
	} // if
	return;
} // slotWorkingDirChanged



bool kfoldingSettingsDialogue::slotCheckAppletImage( const QString& newImage )
{
	QImage image( newImage );
	if ( image.isNull() ) {
		int rc = KMessageBox::questionYesNo( this,
			i18n( "The file you have chosen for the applet image is not a valid image.\n \
			Would you like to restore the default?" ) );
		if ( rc == KMessageBox::Yes ) {
			_settingsWidget->appletImage->setURL( locate( "data", "kfolding/pics/foldinglogo.png" ) );
		} else {
			return false;
		} // else
	} // if
	return showImagePreview();
} // slotCheckAppletImage



void kfoldingSettingsDialogue::setupSettingsWidget()
{
	QFrame *page = plainPage();
	QGridLayout *layout = new QGridLayout( page, 1, 1 );
	
	_settingsWidget = new kfoldingSettingsWidget( page, QString::fromLatin1( "settings" ) );
	_settingsWidget->executable->setMode( KFile::File | KFile::LocalOnly | KFile::ExistingOnly );
	_settingsWidget->workingDir->setMode( KFile::Directory | KFile::LocalOnly );
	_settingsWidget->appletImage->setMode( KFile::File | KFile::LocalOnly | KFile::ExistingOnly );
	
	layout->addWidget( _settingsWidget, 0, 0 );
	
	if ( _configData->executable().isEmpty() ) {
		_settingsWidget->executable->setURL( findFoldingExe() );
	} else {
		_settingsWidget->executable->setURL( _configData->executable() );
	}
	_settingsWidget->startOnLaunch->setChecked( _configData->startOnLaunch() );
	_settingsWidget->rememberState->setChecked( _configData->restoreState() );
	_settingsWidget->appletImage->setURL( _configData->appletImage() );
	
	_settingsWidget->workingDir->setURL( _configData->workingDir() );
	_settingsWidget->userName->setText( _configData->userName() );
	_settingsWidget->teamNumber->setValue( _configData->teamNumber() );
	_settingsWidget->clientType->setCurrentItem( _configData->clientType() );
	_settingsWidget->forceasm->setChecked( _configData->forceASM() );
	_settingsWidget->bigPackets->setChecked( _configData->bigPackets() );
	_settingsWidget->useadvmethods->setChecked( _configData->useAdvMethods() );
	_settingsWidget->machineId->setValue( _configData->machineId() );

	showImagePreview();
	return;
} // setupSettingsWidget



bool kfoldingSettingsDialogue::checkData()
{
	if ( _settingsWidget->executable->url().stripWhiteSpace().isEmpty() ) {
		KMessageBox::error( this, i18n( "You must specify the location of the Folding@home executable" ) );
		_settingsWidget->executable->setFocus();
		_inputError = true;
		return false;
	}
	
	if ( _settingsWidget->workingDir->url().stripWhiteSpace().isEmpty() ) {
		KMessageBox::error( this, i18n( "You must specify a working directory" ) );
		_settingsWidget->workingDir->setFocus();
		_inputError = true;
		return false;
	}

	QFileInfo fileInfo( _settingsWidget->executable->url().stripWhiteSpace() );
	if ( !fileInfo.isFile() || !fileInfo.isExecutable() ) {
		KMessageBox::error( this, i18n( "The selected executable does not appear to be the Folding@home executable." ) );
		_settingsWidget->executable->setFocus();
		return false;
	}
	
	fileInfo.setFile( _settingsWidget->workingDir->url().stripWhiteSpace() );
	if ( !fileInfo.isDir() || !fileInfo.isWritable() ) {
		KMessageBox::error( this, i18n( "You do not have permission to directory %1.  Please select a different working directory." ).arg( _settingsWidget->workingDir->url().stripWhiteSpace() ) );
		_settingsWidget->workingDir->setFocus();
		return false;
	}
	
	if ( !slotCheckAppletImage( _settingsWidget->appletImage->url().stripWhiteSpace() ) ) {
		return false;
	}
	
	return true;
} // checkData



bool kfoldingSettingsDialogue::showImagePreview()
{
	QImage image( _settingsWidget->appletImage->url().stripWhiteSpace() );
	if (  !image.isNull() ) {
		int height = PREVIEW_HEIGHT;
		float width = ( ( image.width() * PREVIEW_HEIGHT ) / image.height() );
		
		image = image.smoothScale( (int) width, height, QImage::ScaleFree );
		_settingsWidget->imagePreview->setPixmap( QPixmap( image ) );
	} else {
		return false;
	} // else
	return true;
} // showImagePreview



QString kfoldingSettingsDialogue::findFoldingExe()
{
	QStringList targets;
	
	targets << "FAH504-Linux";	// Prediction
	targets << "FAH503-Linux";	// Prediction
	targets << "FAH502-Linux";
	targets << "FAH501-Linux";
	targets << "FAH500-Linux";
	
	QString targetPath = QString::null;
	for ( QStringList::iterator it = targets.begin(); it != targets.end(); ++it ) {
		 targetPath  = KStandardDirs::findExe( *it );
		 if ( targetPath !=  QString::null )
		 	break;
	} // for
	return targetPath;
} // findFoldingExe

#include "kfoldingsettingsdialog.moc"
