/*********************************************************************
 *
 *	Copyright (C) 1999 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Utilities
 * MODULE:      Resource Pool
 * FILE:        ResourcePool.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      10/22/99        Initial version
 *
 * DESCRIPTION:
 *      Defines the resource pool class.
 *
 * $Id: ResourcePool.java,v 1.3 2000/09/10 05:09:43 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.util;

import java.lang.ref.SoftReference;

/**
 * Class ResourcePool manages a pool of object resources, based
 * on any single class. Objects can be pulled from the pool and
 * later returned when the caller is finished with the resource.
 * <p>
 * This resource pool implementation makes use of SoftReference
 * objects in order to allow the unused resource objects to be
 * garbage collected in low-memory situations.
 *
 * @author  Nathan Fiedler
 * @version 1.0  10/22/99
 * @see ResourceInstantiator
 */
public class ResourcePool implements ResourceInstantiator {
    /** List of all available resources. Keys are the resource's
     * identity hash code, values are the resource's soft
     * reference object. */
    protected IntHashtable availableResources;
    /** List of all borrowed resources. Keys are the resource's
     * identity hash code, values are the resource's soft
     * reference object. */
    protected IntHashtable borrowedResources;
    /** Reference to a resource instantiator. If this is null,
     * then the resource class is used to create new resources. */
    protected ResourceInstantiator instantiator;
    /** Reference to the class of resource managed by this pool. If
     * this is null, the instantiator is used to create new resources. */
    protected Class resourceClass;
    /** Maximum number of resources allowed to be borrowed at any one
     * time. Zero indicates no upper limit. */
    protected int maximumAllowed;

    /**
     * Constructs a new ResourcePool with empty lists.
     */
    protected ResourcePool() {
        availableResources = new IntHashtable();
        borrowedResources = new IntHashtable();
    } // ResourcePool

    /**
     * Creates a ResourcePool object to hold resource objects
     * of the given class. The class must define a no-argument
     * constructor for creating new resource objects.
     *
     * @param  type  Class of resources to be pooled.
     */
    public ResourcePool(Class type) {
        this();
        resourceClass = type;
    } // ResourcePool

    /**
     * Creates a ResourcePool object to hold resource objects
     * of the given class. The class must define a no-argument
     * constructor for creating new resource objects.
     *
     * @param  type     Class of resources to be pooled.
     * @param  maximum  Sets maximum allowable borrowed resources.
     */
    public ResourcePool(Class type, int maximum) {
        this();
        resourceClass = type;
        maximumAllowed = maximum;
    } // ResourcePool

    /**
     * Creates a ResourcePool object to hold resource objects
     * prepared by the given resource pool instantiator.
     *
     * @param  ri  A resource pool instantiator.
     */
    public ResourcePool(ResourceInstantiator ri) {
        this();
        instantiator = ri;
    } // ResourcePool

    /**
     * Creates a ResourcePool object to hold resource objects
     * prepared by the given resource pool instantiator.
     *
     * @param  ri       A resource pool instantiator.
     * @param  maximum  Sets maximum allowable borrowed resources.
     */
    public ResourcePool(ResourceInstantiator ri, int maximum) {
        this();
        instantiator = ri;
        maximumAllowed = maximum;
    } // ResourcePool

    /**
     * Return a reference to a resource object. If a resource is
     * not available, a new one will be instantiated. It depends
     * on the constructor that was invoked to create this pool
     * whether the new resource will be created using the class's
     * default constructor, or using a resource pool instantiator.
     * The returned object reference is guaranteed to be non-null.
     *
     * @return  Resource object.
     * @exception  InstantiationException
     *             Thrown if the resource instantiator fails.
     */
    public Object getResource() throws InstantiationException {
        if (instantiator == null) {
            // Just call the other getResource method using ourselves
            // as the resource instantiator.
            return getResource(this);
        } else {
            return getResource(instantiator);
        }
    } // getResource

    /**
     * Return a reference to a resource object. If a resource is
     * not available, a new one will be instantiated. The returned
     * object reference is guaranteed to be non-null.
     *
     * @param  ri  A resource pool instantiator to create a new
     *             resource object, if no others are available.
     *             This instantiator will override any others.
     * @return  Resource object.
     * @exception  InstantiationException
     *             Thrown if the resource instantiator fails.
     */
    public synchronized Object getResource(ResourceInstantiator ri)
        throws InstantiationException {

        while ((maximumAllowed > 0) &&
               (borrowedResources.size() >= maximumAllowed)) {
            // We've borrowed the maximum allowable resources.
            // Wait until someone else returns a resource.
            try {
                wait();
            } catch (InterruptedException ie) {
            }
        }

        Object resource = null;
        SoftReference sref = null;
        int hash = 0;
        while ((availableResources.size() > 0) && (resource == null)) {
            // Pick an object from the available list.
            hash = availableResources.getFirstKey();
            sref = (SoftReference) availableResources.remove(hash);
            resource = sref.get();
            // Resource reference may be null if element was
            // garbage collected. We'll loop again in that case.
        }

        // If no available objects are left, create a new one.
        if (resource == null) {
            resource = ri.instantiateResource();
            if (resource == null) {
                // Eek, failed to create new resource.
                throw new InstantiationException
                    ("resource instantation failed");
            }
            hash = System.identityHashCode(resource);
            sref = new SoftReference(resource);
        }

        // Track the borrowed resource for debugging and to save
        // from re-creating the soft reference later on.
        borrowedResources.put(hash, sref);
        return resource;
    } // getResource

    /**
     * Instantiates and initializes a resource object. Only the
     * ResourcePool class should call this method. Other classes
     * should call the appropriate <code>getResource</code>
     * method.
     *
     * @return  Newly created and prepared resource object,
     *          or null if error.
     * @see #getResource
     */
    public Object instantiateResource() {
        if (resourceClass == null) {
            return null;
        }
        // Create new instance of object using class's default constructor.
        // These exceptions should not occur under normal circumstances.
        try {
            return resourceClass.newInstance();
        } catch (ExceptionInInitializerError eiie) {
            eiie.printStackTrace();
        } catch (IllegalAccessException iae) {
            iae.printStackTrace();
        } catch (InstantiationException ie) {
            ie.printStackTrace();
        } catch (SecurityException se) {
            se.printStackTrace();
        }
        return null;
    } // instantiateResource

    /**
     * Discards the given resource such that it will never be used again.
     * Useful if the resource has become invalid and should be removed
     * from the pool's tracking mechanism (i.e. "borrowed list").
     *
     * @param  resource  Resource object to remove from pool.
     */
    public synchronized void removeResource(Object resource) {
        if (resource == null) {
            throw new IllegalArgumentException("resource cannot be null");
        }

        // Get object's identity hash code value.
        int hash = System.identityHashCode(resource);
        // Yank the resource from the borrowed list and discard it.
        borrowedResources.remove(hash);

        // Notify others that we may borrow again.
        notify();
    } // removeResource

    /**
     * Returns the given resource object back to the availability
     * pool, for other users to make use of. The resource object
     * should be suitably reset to be ready for another user to
     * use the object.
     *
     * @param  resource  Resource object to return to pool.
     */
    public synchronized void returnResource(Object resource) {
        if (resource == null) {
            throw new IllegalArgumentException("resource cannot be null");
        }

        // Get object's identity hash code value.
        int hash = System.identityHashCode(resource);
        // Pull reference from borrowed list and put in available list.
        SoftReference sref = (SoftReference) borrowedResources.remove(hash);
        availableResources.put(hash, sref);

        // Notify others that a resource is available.
        notify();
    } // returnResource

    /**
     * Returns a string representation of this resource pool.
     *
     * @return  String.
     */
    public String toString() {
        StringBuffer buff = new StringBuffer("ResourcePool=[");
        buff.append("class = " + resourceClass);
        buff.append(", instantiator = " + instantiator);
        buff.append(", available = " + availableResources.size());
        buff.append(", borrowed = " + borrowedResources.size());
        buff.append("]");
        return buff.toString();
    } // toString
} // ResourcePool
