/*  tilp - link program for TI calculators
 *  Copyright (C) 1999-2001  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
  This file contains utility functions about files, attributes,
  sorting routines for selection, conversion routines between dirlist
  and glists.
 */

#ifdef __WIN32__
# include <windows.h>
#endif
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#include "str.h"
#include "platform.h"

#include "struct.h"
#include "support.h"
#include "paths.h"

/*******************************/
/* File manipulation functions */
/*******************************/

/*
  Copy a file from src to dst 
 */
int copy_file(char *src, char *dst)
{
  FILE *in, *out;
  int c;

   if((in=fopen(src, "rb")) == NULL)
    {
      return 1;
    }
   if((out=fopen(dst, "wb")) == NULL)
     {
      return 2;
    }
   while(!feof(in))
     {
	   c=fgetc(in);
	   if(feof(in)) break;
       fputc(c, out);
     }
   fclose(in);
   fclose(out);

  return 0;
}

/*
  Move the file
 */
int move_file(char *src, char *dst)
{
  int ret;

  ret=copy_file(src, dst);
  if(ret) return ret;
  unlink(src);

  return 0;
}

/* Remove '\r' characters for GtkText */
void process_buffer(gchar *buf)
{
  gint i;
  
  for(i=0; i<strlen(buf); i++)
    if(buf[i]=='\r') buf[i]=' ';
}

/*************************************/
/* Extracting informations functions */
/*************************************/

/*
  Returns the user's home directory
*/
int get_home_path(char **path)
{
#if defined(__UNIX__)
  uid_t uid;
  struct passwd *p;
  
  uid = getuid();
  //DISPLAY("UID: %i\n", uid);
  
  if((p = getpwuid(uid)) == NULL)
    {
      *path=NULL;
      return 0;
    }
  else
    {
      //DISPLAY("User login name: %s\n", p->pw_name);
      //DISPLAY("User's home directory: %s\n", p->pw_dir);
      *path = g_strdup(p->pw_dir);
      return 1;
    }
#endif
  return 0;
}

/* Return the filename or its extension if it has one */
char *file_extension(char *filename)
{
  int i;
  char *p;
  
  for(i=strlen(filename); i > 0; i--)
    {
      if(filename[i] == '.') break;
    }
  p=filename+i+1;
  
  return p;
}

#define INIT_LOCALE( domain )	G_STMT_START{	\
	gtk_set_locale ();			\
	setlocale (LC_NUMERIC, "C");		\
	bindtextdomain (domain, LOCALEDIR);	\
	textdomain (domain);			\
				}G_STMT_END

int initialize_paths(void)
{
  char *home_dir;
#ifdef __WIN32__
  HMODULE hModule;
  DWORD dWord;
  char sBuffer[4*MAXCHARS];
  char *dirname;
#endif
/* 
     Init the path for the Windows version by getting the executable
     location.
  */
#if defined(__WIN32__)
  /* Retrieve the directory of the executable */
  hModule = GetModuleHandle("tilp.exe");
  dWord = GetModuleFileName(hModule, sBuffer, 4*MAXCHARS);
  dirname = g_dirname(sBuffer);
  DISPLAY("Current path: <%s>\n", dirname);
  inst_paths.base_dir = g_strdup(dirname);
  g_free(dirname);
#else
  inst_paths.base_dir = g_strdup("");
#endif
  //DISPLAY("inst_path.base_dir = <%s>\n", inst_paths.base_dir);

  /*
    Initialize share path
  */
  inst_paths.share_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				   G_DIR_SEPARATOR_S, NULL);
  //DISPLAY("inst_paths.share_dir = <%s>\n", inst_paths.share_dir);

  /*
    Initialize pixmaps path
  */
#ifndef __WIN32__
  inst_paths.pixmap_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				    "/pixmaps/", NULL);
#else
  inst_paths.pixmap_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				    "\\pixmaps\\", NULL);
#endif
  //DISPLAY("inst_paths.pixmap_dir = <%s>\n", inst_paths.pixmap_dir);
  add_pixmap_directory(inst_paths.pixmap_dir);

  /* 
     Init internationalization for Linux or Windows
     Some things about i18 paths:
     bintextdomain(PACKAGE, "/usr/share/locale"); ->
     '/usr/share/locale/fr/LC_MESSAGES/tilp.mo'
  */
#if defined(HAVE_LIBINTL_H) || defined(ENABLE_NLS)
#ifdef __UNIX__
  inst_paths.locale_dir = g_strjoin(inst_paths.base_dir, PACKAGE_LOCALE_DIR,
				    G_DIR_SEPARATOR_S, NULL);
#elif defined(__WIN32__)
  inst_paths.locale_dir = g_strjoin(inst_paths.base_dir, PACKAGE_LOCALE_DIR,
				    "\\locale\\", NULL);
#endif
  //DISPLAY("inst_paths.locale_dir = <%s>\n", inst_paths.locale_dir);
#endif
  
  /*
    Initialize help path
  */
#ifndef __WIN32__
  inst_paths.help_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				  "/help/", NULL);
#else
  inst_paths.help_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				  "\\help\\", NULL);
#endif
  //DISPLAY("inst_paths.help_dir = <%s>\n", inst_paths.help_dir);
  
  /*
    Initialize manpage path
  */
#ifndef __WIN32__
  inst_paths.manpage_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				     "/", NULL);
#else
  inst_paths.manpage_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				     "\\", NULL);
#endif
  //DISPLAY("inst_paths.manpage_dir = <%s>\n", inst_paths.manpage_dir);
  
  /*
    Initialize ROMs path
  */
#ifndef __WIN32__
  inst_paths.rom_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				    "/ROMs/", NULL);
#else
  inst_paths.rom_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				    "\\ROMs\\", NULL);
#endif
  //DISPLAY("inst_paths.rom_dir = <%s>\n", inst_paths.rom_dir);

  /*
    Initialize skin path
  */
#ifndef __WIN32__
  inst_paths.skin_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				    "/skins/", NULL);
#else
  inst_paths.skin_dir = g_strjoin(inst_paths.base_dir, SHARE_DIR,
				    "\\skins\\", NULL);
#endif
  //DISPLAY("inst_paths.skin_dir = <%s>\n", inst_paths.skin_dir);

  // on UNIX systems, HOME directory by default
#ifdef __UNIX__ 
  if(get_home_path(&home_dir))
    {
      chdir(home_dir);
      g_free(home_dir);
    }
  else
    {
      DISPLAY("Can not get HOME directory.\n");
      exit(-1);
    }
#endif

  return 0;
}
