/* GSnes9x -- confiles.c: New format for the configuration files.
 * Copyright (C) 1999 Canek Pelez Valds <canek@abulafia.fciencias.unam.mx>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsnes9x.h"

/* These are not constants because the autoinitialization. */

gchar *bool_keywords[] = {
	/* Mixed. */
	"TRANSPARENCY",	"SIXTEEN", "HIRES", "NOMODESWITCH",
	"FULLSCREEN", "SCALE", "DISPLAYFRAMERATE", "NOSOUND", "SOUNDSKIP",
	"STEREO", "SOUNDQUALITY", "ENVX", "SOUNDSYNC", "ALTSAMPLEDECODE",
	"INTERPOLATEDSOUND","BUFFERSIZE", "NOSAMPLECACHING", "NOECHO", "NOMASTERVOLUME",
	"FRAMESKIP", "FRAMETIME", "CYCLES", "NOJOY", "SWAP",
	"JOYMAP0", "JOYMAP1", "JOYMAP2", "JOYMAP3",

	/* ROM. */
	"GAMEGENIE","ACTIONREPLAY", "GOLDFINGER", "INTERLEAVED",
	"INTERLEAVED2", "HIROM", "LOROM", "HEADER", "NOHEADER", "PAL",
	"NTSC", "LAYERING", "LOADSNAPSHOT", "NOHDMA", "NOSPEEDHACKS",
	"NOWINDOWS",

	/* Global. */
	"WITH_ESD", "NETPLAY",

	NULL
};

gchar *string_keywords[] = {
	/* ROM. */
	"ROM_FILE", "NAME", "COMPANY", "IMAGE_FILE", "SNAP_FILE",
	/* Global. */
	"SNES9X_COMMAND", "ROMS_DIR", "SERVER", "NETPORT",

	/* Conf. files only. */
	"JOYMAP0MAP", "JOYMAP1MAP", "JOYMAP2MAP", "JOYMAP3MAP",
	"GGCODE", "ARCODE", "GFCODE",
	
	NULL
};

gchar *number_keywords[] = {
	"SOUNDSKIPNUMBER", "SOUNDQUALITYNUMBER", "BUFFERSIZENUMBER",
	"FRAMESKIPNUMBER", "FRAMETIMENUMBER", "CYCLESNUMBER", "FILTERNUMBER",

	NULL
};

GlobalPrefs*
read_conf_file ()
{
	GlobalPrefs *prefs = global_prefs_new ();
	FILE        *file  = NULL;
	gchar       *conf  = gnome_util_home_file (CONF_FILE);
	gchar       *lval, *rval, **sides;
	gchar       *str   = g_new (gchar, CONF_STR_LEN);
	gint         size  = CONF_STR_LEN;
	gint         cont;
	
	F_NAME();
	
	file = fopen (conf, "r");
	
	if (!file) {
		create_conf_file ();
		file = fopen (conf, "r");
	}

	getline (&str, &size, file);
	str[strlen (str)-1] = '\0';

	while (!feof (file)) {
		lval  = rval = NULL;
		sides = NULL;
		cont = 0;
		if (!is_comment_line (str)) {
			sides = g_strsplit (str, "=", 2);
			lval = g_strdup (g_strstrip (sides[0]));
			rval = g_strdup (g_strstrip (sides[1]));
			g_strfreev (sides);
			if (!lval || !rval) {
				fclose (file);
				g_warning (_("Bad line formart:\n\n\t%s"),
					   str);
				g_warning (_("Configuration file corrupted. "
					     "Recreating it with default "
					     "values."));
				create_conf_file ();
				return read_conf_file ();
			}

			switch (keyword_type (lval)) {
			case BOOLEAN_KEYWORD:
				if (!g_strcasecmp (rval, "true"))
					assign_global_boolean_value (
						prefs,
						keyword_key (lval,
							     BOOLEAN_KEYWORD),
						TRUE);
				else
					assign_global_boolean_value (
						prefs,
						keyword_key (lval,
							     BOOLEAN_KEYWORD),
						FALSE);
				break;
			case NUMBER_KEYWORD:
				assign_global_number_value (
					prefs,
					keyword_key (lval, NUMBER_KEYWORD),
					atof (rval));
				break;
			case STRING_KEYWORD:
				assign_global_string_value (
					prefs,
					keyword_key (lval, STRING_KEYWORD),
					rval);
				break;
			case UNKNOW_KEYWORD:
				fclose (file);
				g_warning (_("Unknow keyword %s."), lval);
				g_warning (_("Configuration file corrupted. "
					     "Recreating it with default "
					     "values."));
				create_conf_file ();
				return read_conf_file ();
			}
		}

		getline (&str, &size, file);
		str[strlen (str)-1] = '\0';

		if (lval)
			g_free (lval);
		if (rval)
			g_free (rval);
	}

	return prefs;
}

ROMInfo*
read_rom_info_file (const gchar *rom_name)
{
	ROMInfo *rom_info     = rom_info_new ();
	FILE        *file     = NULL;
	gchar       *lval, *rval, **sides;
	gchar       *str      = g_new (gchar, CONF_STR_LEN);
	gchar       *rom_file;
	gint         size     = CONF_STR_LEN;
	gint         cont;
	
	F_NAME();

	rom_file = g_strconcat (CONF_DIR, "/", rom_name, NULL);
	
	file = fopen (gnome_util_home_file (rom_file), "r");
	
	if (!file) {
		g_warning (_("No info file for ROM: %s. ROM %s skipped."),
			   rom_name, rom_name);
		g_free (rom_file);
		g_free (rom_info);
		return NULL;
	}
	
	getline (&str, &size, file);
	str[strlen (str)-1] = '\0';
	
	while (!feof (file)) {
		lval  = rval = NULL;
		sides = NULL;
		cont = 0;
		if (!is_comment_line (str)) {
			sides = g_strsplit (str, "=", 2);
			lval  = g_strdup (g_strstrip (sides[0]));
			rval  = g_strdup (g_strstrip (sides[1]));
			g_strfreev (sides);
			if (!lval || !rval) {
				fclose (file);
				g_free (rom_info);
				g_warning (_("Bad line formart:\n\n\t%s\n"
					     "ROM %s skipped."), str,
					   rom_name);
				return NULL;
			}

			switch (keyword_type (lval)) {
			case BOOLEAN_KEYWORD:
				if (!g_strcasecmp (rval, "true"))
					assign_rom_boolean_value (
						rom_info,
						keyword_key (lval,
							     BOOLEAN_KEYWORD),
						TRUE);
				else
					assign_rom_boolean_value (
						rom_info,
						keyword_key (lval,
							     BOOLEAN_KEYWORD),
						FALSE);
				break;
			case NUMBER_KEYWORD:
				assign_rom_number_value (
					rom_info,
					keyword_key (lval, NUMBER_KEYWORD),
					atof (rval));
				break;
			case STRING_KEYWORD:
				assign_rom_string_value (
					rom_info,
					keyword_key (lval, STRING_KEYWORD),
					rval);
				break;
			case UNKNOW_KEYWORD:
				fclose (file);
				g_free (rom_file);
				g_warning (_("Unknow keyword %s. "
					     "ROM %s skipped."),
					   lval, rom_name);
				return NULL;
			}
		}

		getline (&str, &size, file);
		str[strlen (str)-1] = '\0';
		
		if (lval)
			g_free (lval);
		if (rval)
			g_free (rval);
	}
	
	g_free (rom_file);
	fclose (file);

	return rom_info;
}

void
assign_global_boolean_value (GlobalPrefs *prefs, gint key, gboolean val)
{
	F_NAME();
	
	if (key < MIX_BOOL_PREFS) {
		prefs->bool_prefs[key] = val;
		return;
	}
		
	if (key >= MIX_BOOL_PREFS && key < MIX_BOOL_PREFS+ROM_BOOL_PREFS) {
		g_warning (_("Try to assign a exclusively ROM option "
			     "in the global options."));
		return;
	}

	if (key >= MIX_BOOL_PREFS+ROM_BOOL_PREFS) {
		prefs->bool_prefs[key-ROM_BOOL_PREFS] = val;
		return;
	}
}

void
assign_global_string_value (GlobalPrefs *prefs, gint key, gchar *val)
{  
	gint i;
	
	F_NAME();
	
	if (key < MIX_STR_PREFS) {
		/* Empty */;
		return;
	}

	if ((key >= MIX_STR_PREFS && key < MIX_STR_PREFS+ROM_STR_PREFS) ||
	    (key >= MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS+NUM_OF_JOYS &&
	     key <  MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS+NUM_OF_JOYS+
	     CODES)) {
		g_warning (_("Try to assign a exclusively ROM "
			     "option in the global options."));
		return;
	}

	if (key >= MIX_STR_PREFS+ROM_STR_PREFS &&
	    key <  MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS) {
		key = key - (MIX_STR_PREFS+ROM_STR_PREFS);
		if (prefs->string_prefs[key])
			g_free (prefs->string_prefs[key]);
		prefs->string_prefs[key] = g_strdup (val);
		return;
	}

	if (key >= MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS &&
	    key <  MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS+NUM_OF_JOYS) {
		key = key  - (MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS);
		for (i = 0; i < NUM_OF_BUTTONS; i++)
			if (strlen (val) > 7)
				prefs->buttons[key][i] = (gint)(val[i]-'0');
		return;
	}
}

void
assign_global_number_value (GlobalPrefs *prefs, gint key, gfloat val)
{
	F_NAME();

	if (key < MIX_NUM_PREFS) {
		if (key == FRAMETIME_NUMBER)
			prefs->number_prefs[key] = val;
		else
			prefs->number_prefs[key] = (gint) val;
		return;
	}

	if (key >= MIX_NUM_PREFS &&
	    key < MIX_NUM_PREFS+ROM_NUM_PREFS) {
		/* Empty. */
		return;
	}

	if (key >= MIX_NUM_PREFS+ROM_NUM_PREFS &&
	    key <  MIX_NUM_PREFS+ROM_NUM_PREFS+GLOB_NUM_PREFS) {
		/* Empty. */
		return;
	}	
}

void
assign_rom_boolean_value (ROMInfo *rom_info, gint key, gboolean val)
{
	F_NAME();

	if (key < MIX_BOOL_PREFS+ROM_BOOL_PREFS) {
		rom_info->bool_prefs[key] = val;
		return;
	}
	
	if (key >= MIX_BOOL_PREFS+ROM_BOOL_PREFS &&
	    key <  MIX_BOOL_PREFS+ROM_BOOL_PREFS+GLOB_BOOL_PREFS) {
		g_warning (_("Try to assign a exclusively global "
			     "option in a ROM option."));
		return;
	}
}

void
assign_rom_string_value (ROMInfo *rom_info, gint key, gchar *val)
{
	gchar **codes = NULL;
	gint    i, free;
	
	F_NAME();

	if (key < MIX_STR_PREFS+ROM_STR_PREFS) {
		rom_info->string_prefs[key] = g_strdup (val);
		return;
	}

	if (key >= MIX_STR_PREFS+ROM_STR_PREFS &&
	    key <  MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS) {
		g_warning (_("Try to assign a exclusively global "
			     "option in a ROM option."));
		return;
	}

	if (key >= MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS &&
	    key <  MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS+NUM_OF_JOYS) {
		key = key - (MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS);
		for (i = 0; i < NUM_OF_BUTTONS; i++)
			if (strlen (val) > 7)
				rom_info->buttons[key][i] = (gint)(val[i]-'0');
		return;
	}

	if (key >= MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS+NUM_OF_JOYS &&
	    key <  MIX_STR_PREFS+ROM_STR_PREFS+GLOB_STR_PREFS+NUM_OF_JOYS+
	    CODES) {
		key = key - (MIX_STR_PREFS+ROM_STR_PREFS+
			     GLOB_STR_PREFS+NUM_OF_JOYS);
		switch (key) {
		case GG_CODE:
			codes = (gchar**)rom_info->gamegenie_code;
			break;
		case AR_CODE:
			codes = (gchar**)rom_info->actionreplay_code;
			break;
		case GF_CODE:
			codes = (gchar**)rom_info->goldfinger_code;
			break;
		}

		free = -1;

		for (i = 0; i < MAX_CODES; i++) {
			if (!codes[i]) {
				free = i;
				i = MAX_CODES;
			}
		}

		if (free == -1) {
			g_free (codes[0]);
			free = 0;
		}

		codes[free] = g_strdup (val);
		return;
	}
}

void
assign_rom_number_value (ROMInfo *rom_info, gint key, gfloat val)
{
	F_NAME();

	if (key < MIX_NUM_PREFS) {
		if (key == FRAMETIME_NUMBER)
			rom_info->number_prefs[key] = val;
		else
			rom_info->number_prefs[key] = (gint) val;
		return;
	}

	if (key >= MIX_NUM_PREFS &&
	    key < MIX_NUM_PREFS+ROM_NUM_PREFS) {
		/* Empty. */
		return;
	}

	if (key >= MIX_NUM_PREFS+ROM_NUM_PREFS &&
	    key <  MIX_NUM_PREFS+ROM_NUM_PREFS+GLOB_NUM_PREFS) {
		/* Empty. */
		return;
	}	

}

gint
keyword_key (const gchar *keyword, GSnes9xKeywordTypes type)
{
	gint cont = 0;

	F_NAME();
	
	switch (type) {
	case BOOLEAN_KEYWORD:
		while (bool_keywords[cont]) {
			if (!strcmp (keyword, bool_keywords[cont]))
				return cont;
			cont++;
		}
		break;
	case NUMBER_KEYWORD:
		while (number_keywords[cont]) {
			if (!strcmp (keyword, number_keywords[cont]))
				return cont;
			cont++;
		}
		break;
	case STRING_KEYWORD:
		while (string_keywords[cont]) {
			if (!strcmp (keyword, string_keywords[cont]))
				return cont;
			cont++;
		}
		break;
	case UNKNOW_KEYWORD:
		break;
	}

	/* Unreachable */
	return -1;
}

GSnes9xKeywordTypes
keyword_type (const gchar *lval)
{
	gint cont = 0;

	F_NAME();
	
	while (bool_keywords[cont]) {
		if (!g_strcasecmp (lval, bool_keywords[cont]))
			return BOOLEAN_KEYWORD;
		cont++;
	}
	
	cont = 0;

	while (number_keywords[cont]) {
		if (!g_strcasecmp (lval, number_keywords[cont]))
			return NUMBER_KEYWORD;
		cont++;
	}

	cont = 0;

	while (string_keywords[cont]) {
		if (!g_strcasecmp (lval, string_keywords[cont]))
			return STRING_KEYWORD;
		cont++;
	}

	return UNKNOW_KEYWORD;
}

gboolean
is_comment_line (const gchar* str)
{
	gint i = 0;

	F_NAME();
	
	while (str[i] == ' ')
		i++;

	if (str[i] == '\n' || str[i] == '#')
		return TRUE;

	return FALSE;
}

void
save_global_prefs (GlobalPrefs *gp)
{
	FILE *file;
	gint  i, j;
	gint  stringops[] = {
		-2, /* Name */
		-2, /* Company */
		-2, /* ROM file */
		-2, /* Image file */
		-2, /* Snapshot file */
		-1,  /* Snes9X command */
		-1, /* Rom directory */
		NETPLAY,
		NETPLAY
	};
	
	gint numberops[] = {
		SOUNDSKIP,
		SOUNDQUALITY,
		BUFFERSIZE,
		FRAMESKIP,
		FRAMETIME,
		CYCLES
	};

	F_NAME();
	
	file = fopen (gnome_util_home_file (CONF_FILE), "w");

	fprintf (file, "# GSnes9x global configuration file.\n");

	for (i = 0; i < ROM_STR_PREFS+GLOB_STR_PREFS; i++) {
		switch (stringops[i]) {
		case -2:
			break;
		case -1:
			fprintf (file, "%s=%s\n", string_keywords[i],
				 gp->string_prefs[i-ROM_STR_PREFS]);
			break;
		default:
			if (gp->string_prefs[i-ROM_STR_PREFS])
				fprintf (file, "%s=%s\n", string_keywords[i],
					 gp->string_prefs[i-ROM_STR_PREFS]);
			break;
		}
	}
	
	for (i = 0; i < MIX_BOOL_PREFS; i++)
		if (gp->bool_prefs[i])
			fprintf (file, "%s=TRUE\n", bool_keywords[i]);

	for (i = MIX_BOOL_PREFS; i < MIX_BOOL_PREFS+GLOB_BOOL_PREFS; i++)
		if (gp->bool_prefs[i])
			fprintf (file, "%s=TRUE\n",
				 bool_keywords[i+ROM_BOOL_PREFS]);
	
	for (i = 0; i < MIX_NUM_PREFS; i++) {
		if (i == FILTER_NUMBER) {
			fprintf (file, "%s=%d\n", number_keywords[i],
				 (gint)gp->number_prefs[i]);	
		}
		else {
			if (gp->bool_prefs[(numberops[i])]) {
				if (i != FRAMETIME_NUMBER) {
					fprintf (file, "%s=%d\n",
					 	number_keywords[i],
				 		(gint)gp->number_prefs[i]);
				}
		       		else {
					fprintf (file, "%s=%f\n",
				 		number_keywords[i],
				 		(gfloat)gp->number_prefs[i]);
				}
			}
		}
	}		

	
	
	for (i = 0; i < NUM_OF_JOYS; i++) {
		if (gp->bool_prefs[JOYMAP0+i]) {
			fprintf (file, "JOYMAP%dMAP=", i);
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				fprintf (file, "%d", gp->buttons[i][j]);
			fprintf (file, "\n");
		}
	}

	fclose (file);
}

void
save_rom (ROMInfo *rom_info)
{
	gint     i, j;
	FILE    *file;
	gchar   *conf_file;

	gint  stringops[] = {
		-1,
		-1,
		-1,
		-1,
		LOADSNAPSHOT,
		-2,
		-2,
		-2
	};
	
	gint numberops[] = {
		SOUNDSKIP,
		SOUNDQUALITY,
		BUFFERSIZE,
		FRAMESKIP,
		FRAMETIME,
		CYCLES
	};
	
	F_NAME();

	conf_file = g_strconcat (CONF_DIR, "/", rom_info->string_prefs[NAME],
				 NULL);

	file = fopen (gnome_util_home_file (conf_file), "w");

	fprintf (file, "# GSnes9x configurarion file for \"%s\".\n",
		 rom_info->string_prefs[NAME]);
	
	for (i = 0; i < ROM_STR_PREFS; i++) {
		switch (stringops[i]) {
		case -2:
			break;
		case -1:
			fprintf (file, "%s=%s\n", string_keywords[i],
				 rom_info->
				 string_prefs[i]);
			break;
		default:
			if (rom_info->string_prefs[i])
				fprintf (file, "%s=%s\n", string_keywords[i],
					 rom_info->
					 string_prefs[i]);
			break;
		}
	}

	for (i = 0; i < MIX_BOOL_PREFS+ROM_BOOL_PREFS; i++)
		if (rom_info->bool_prefs[i])
			fprintf (file, "%s=TRUE\n", bool_keywords[i]);
	
	for (i = 0; i < MIX_NUM_PREFS; i++) {
		if (i == FILTER_NUMBER) {
			fprintf (file, "%s=%d\n", number_keywords[i],
				 (gint)rom_info->number_prefs[i]);	
		}
		else {
			if (rom_info->bool_prefs[(numberops[i])]) {
				if (i != FRAMETIME_NUMBER) {
					fprintf (file, "%s=%d\n",
					 	number_keywords[i],
				 		(gint)rom_info->number_prefs[i]);
				}
		       		else {
					fprintf (file, "%s=%f\n",
				 		number_keywords[i],
				 		(gfloat)rom_info->number_prefs[i]);
				}
			}
		}
	}		
	
	for (i = 0; i < NUM_OF_JOYS; i++) {
		if (rom_info->bool_prefs[JOYMAP0+i]) {
			fprintf (file, "JOYMAP%dMAP=", i);
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				fprintf (file, "%d", rom_info->buttons[i][j]);
			fprintf (file, "\n");
		}
	}

	for (i = 0; i < MAX_CODES; i++) {
		if (rom_info->gamegenie_code[i])
			fprintf (file, "GGCODE=%s\n",
				 rom_info->gamegenie_code[i]);
	}

	for (i = 0; i < MAX_CODES; i++) {
		if (rom_info->actionreplay_code[i])
			fprintf (file, "ARCODE=%s\n",
				 rom_info->actionreplay_code[i]);
	}

	for (i = 0; i < MAX_CODES; i++) {
		if (rom_info->goldfinger_code[i])
			fprintf (file, "GFCODE=%s\n",
				 rom_info->goldfinger_code[i]);
	}
	
	fclose (file);

	file = fopen (gnome_util_home_file (ROMS_FILE), "a");
	fprintf (file, "%s\n", rom_info->string_prefs[NAME]);
	fclose (file);
}

void
create_conf_file ()
{
	FILE  *file;
	gchar *conf_file = gnome_util_home_file (CONF_FILE);	

	F_NAME();
	
	file = fopen (gnome_util_home_file (CONF_DIR), "r");

	if (!file)
		mkdir (gnome_util_home_file (CONF_DIR),
		       (S_IRUSR | S_IWUSR | S_IXUSR));
	else
		fclose (file);

	file = fopen (conf_file, "w");

	if (!file)
		g_error (_("I can't create the configuration file."));

	fprintf (file, "# GSnes9x conf file v2.5\n");
	fprintf (file, "STEREO=TRUE\n");
	fprintf (file, "SNES9X_COMMAND=snes9x\n");
#ifndef NO_ESD_SUPPORT
	fprintf (file, "WITH_ESD=TRUE\n");
#endif

	fclose (file);
}

/* confiles.c ends here. */
