/*
 * $Id: gnuvd.c,v 1.12 2002/01/09 23:20:59 binnema Exp $
 */


/*
 ** Copyright (C) 2001 Dirk-Jan C. Binnema <dirk-jan@djcbsoftware.nl>
 **  
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **  
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **  
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **  
 */

#include <stdio.h>
#include <libgnuvd.h>


static int  show_results     (Buffer *results);
static char get_special_char (char *str);

int
main (int argc, char* argv[])
{
	int result;
	VDQuery *vd_query;
	VDException *vd_ex;
	
	if (argc != 2) {
		fprintf (stderr, "expected: gnuvd <word>\n");
		return 1;
	}

	vd_ex = vd_exception_new ();
  	if (!vd_ex) {
		fprintf (stderr, "internal error vd_ex == NULL\n");
		return 1;
	}

	
	vd_query = vd_query_new (argv[1], vd_ex);
	if (!vd_query) {

		if (vd_ex->_msg)
			fprintf (stderr, "error: %s\n", vd_ex->_msg);
		else
			fprintf (stderr, "unknown error\n");
		
		vd_exception_destroy (vd_ex);
		return 1;
	}
			
	result = vd_query_perform (vd_query, vd_ex);
	if (result != VD_ERR_OK && result != VD_ERR_NOT_FOUND) {
		
		fprintf (stderr, "error %d: check your internet settings\n",
			 vd_ex->_err, vd_ex->_msg);
		vd_query_destroy (vd_query);
		vd_exception_destroy (vd_ex);
		return 1;
	
	} else  if (result == VD_ERR_NOT_FOUND) {
	
		fprintf (stderr, "word not found\n");
		return 1;
	}

	show_results (vd_query->_result_buffer);
	vd_query_destroy (vd_query);
	vd_exception_destroy (vd_ex);
	
	return 0;
}


static int
show_results (Buffer* buffer)
{
	BufferIterator i;
	int j;
	char *data;
	char k;

	const int s_kars_max = 10;
	char s_kars[s_kars_max+2];
	
	/* strip superfluous space before  */
	for (i = buffer_begin (buffer); i != buffer_end(buffer); ++i) {
		if (!isspace (buffer_at (buffer, i)))
		    break;
	}
	
	buffer_erase (buffer, buffer_begin(buffer), i);
	

	/* strip superfluous space after */
	for (i = buffer_end (buffer) - 1; i >= buffer_begin(buffer) + 3; --i) 
		if ((buffer_at (buffer, i-3) == 'D') &&
		    (buffer_at (buffer, i-2) == 'L') &&
		    (buffer_at (buffer, i-1) == '>'))
			break;
	
	buffer_erase (buffer, i, buffer_end(buffer) - i - 1);

	data = (char*) buffer_data (buffer);

	for (j = buffer_begin (buffer); j < buffer_end(buffer); ++j) {

		/* DL->\n, DD ->\n */
		if (buffer_end(buffer) - j > 5) {
			if ((strncmp (&data[j], "</DL>", 5) == 0)
			    || (strncmp (&data[j],"</DD>", 5) == 0)) {
				printf ("\n");
				j += 4;
				continue;
			}
		}

		/* remove " <DL>" */
		if (j < buffer_end(buffer) - 4 
		    && (strncmp (&data[j], " <DL>", 5) == 0))
		    j += 5;


		/* remove tags */
		if (data[j] == '<') {
			while (j < buffer_end(buffer) && data[j] != '>')
				++j;
			continue;
		}
		
		/* replace special chars */
		if (data[j] == '&') {
			int c = 0;
			while (j < buffer_end(buffer) && data[j] != ';' && c < s_kars_max)
				s_kars [c++] = data[j++];
			if (data[j] == ';') {
				s_kars [c++] = ';';
				s_kars [c]   = '\0';

				k = get_special_char (s_kars);
				if (k)
					printf ("%c", k);
			}
			continue;
		}
		
		printf ("%c", data[j]);
	}
}


static char
get_special_char (char *str)
{
	int i;
	
	struct map_entry {
		char *entity;
		char ascii;
	};
	
	static struct map_entry symbol_map[]  = {
		/* special chars */
		{ "&#36;",     '$'},
		{ "&#37;",     '%'},
		{ "&#38;",     '&'},
		{ "&#39;",     '\''},
		{ "&#64;",     '@'},
		{ "&#180;",    0},
		{ "&#183;",    0},
		{ "&#126;",    '~'},
		
		/* A/a */
		{ "&agrave;",  ''},
		{ "&acute;",   ''},
		{ "&acirc;",   ''},
		{ "&auml;",    ''},
		{ "&aring;",   ''},
		{ "&aelig;",   ''}, 
		
		{ "&Agrave;",  ''},
		{ "&Acute;",   ''},
		{ "&Acirc;",   ''},
		{ "&Auml;",    ''},
		{ "&Aring;",   ''},
		{ "&Aelig;",   ''}, 

		/* C/c */
		{ "&ccedil;",   ''},
		{ "&Ccdeil;",   ''},
		
		/* E/e */
		{ "&egrave;",  ''},
		{ "&ecute;",   ''},
		{ "&ecirc;",   ''},
		{ "&euml;",    ''},
		{ "&eacute;",  ''}, /* weird -> 'sao tom_' */ 
			
		{ "&Egrave;",  ''},
		{ "&Ecute;",   ''},
		{ "&Ecirc;",   ''},
		{ "&Euml;",    ''},
	
		/* I/i */
		{ "&igrave;",  ''},
		{ "&icute;",   ''},
		{ "&icirc;",   ''},
		{ "&iuml;",    ''},
			
		{ "&Igrave;",  ''},
		{ "&Icute;",   ''},
		{ "&Icirc;",   ''},
		{ "&Iuml;",    ''},

		/* n */
		{ "&ntilde",   ''},
		{ "&Ntilde",   ''},
		
		/* O/o */
		{ "&ograve;",  ''},
		{ "&ocute;",   ''},
		{ "&ocirc;",   ''},
		{ "&ouml;",    ''},
		{ "&oslash;",  ''},
		{ "&otilde;",  ''},
		
		{ "&Ograve;",  ''},
		{ "&Ocute;",   ''},
		{ "&Ocirc;",   ''},
		{ "&Ouml;",    ''},
		{ "&Oslash;",  ''},
		{ "&Otilde;",  ''},
		
		/* U/u */
		{ "&ugrave;",  ''},
		{ "&ucute;",   ''},
		{ "&ucirc;",   ''},
		{ "&uuml;",    ''},
			
		{ "&Ugrave;",  ''},
		{ "&Ucute;",   ''},
		{ "&Ucirc;",   ''},
		{ "&Uuml;",    ''},
		
		/* ij */
		{ "&yuml;",    ''}};
	

	for (i = 0; i != sizeof (symbol_map) / sizeof (struct map_entry); ++i)
		if (strcmp (str, symbol_map[i].entity) == 0)
			return symbol_map[i].ascii;

	return '?';
}
