/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.property.test;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.avalon.framework.context.Resolvable;
import org.apache.avalon.excalibur.property.PropertyException;
import org.apache.avalon.excalibur.property.PropertyUtil;
import junit.framework.TestCase;

/**
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 */
public final class PropertyUtilTestCase
    extends TestCase
{
    public PropertyUtilTestCase( final String name )
    {
        super( name );
    }

    private static final class ResolveTest
        implements Resolvable
    {
        int  m_count;
        int  m_current;

        public ResolveTest( final int count )
        {
            m_count = count;
        }

        public Object resolve( final Context context )
        {
            m_current++;

            if( m_current >= m_count ) return new Integer( m_count );
            else return this;
        }
    }

    private final static Object     OBJ1         = new Object();
    private final static Object     OBJ2         = new Object();

    private DefaultContext          m_context;


    public void setUp()
    {
        m_context = new DefaultContext();
        m_context.put( "obj1", OBJ1 );
        m_context.put( "obj2", OBJ2 );
        m_context.put( "res1", new ResolveTest( 1 ) );
        m_context.put( "res2", new ResolveTest( 2 ) );
        m_context.put( "res3", new ResolveTest( 3 ) );
        m_context.put( "res4", new ResolveTest( 4 ) );
        m_context.put( "key1", "obj1" );
        m_context.put( "key2", "obj2" );
        m_context.put( "key3", "2" );
        m_context.put( "key4", "1" );
        m_context.put( "key5", "obj" );
        m_context.put( "key6", "key" );
    }

    public void testNoResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.resolveProperty( "blah", m_context, false );

        assertEquals( result, "blah" );
    }

    public void testObjResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.resolveProperty( "${obj1}", m_context, false );

        assertEquals( result, OBJ1 );
    }

    public void testObjResolveToText()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.resolveProperty( "${obj1} ", m_context, false );

        assertEquals( result, OBJ1 + " " );
    }

    public void testDualObjResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.resolveProperty( " ${obj1} ${obj2} ", m_context, false );

        assertEquals( result, " " + OBJ1 + " " + OBJ2 + " " );
    }

    public void testRecurseObjResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.resolveProperty( "${res1}", m_context, false );

        assertEquals( result, new Integer( 1 ) );
    }

    public void testRecurseObjResolve2()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.resolveProperty( "${res2}", m_context, false );

        assertEquals( result, new Integer( 2 ) );
    }

    public void testNullObjResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.resolveProperty( "${blahaaa}", m_context, true );

        assertEquals( result, "" );
    }

    public void testNullObjResolveForException()
        throws PropertyException
    {
        try
        {
            final Object result =
                PropertyUtil.resolveProperty( "${blahaaa}", m_context, false );
        }
        catch( final PropertyException pe )
        {
            return;
        }
        fail( "NUll resolve occured without exception" );
    }

//////////////////////

    public void testRecursiveNoResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "blah", m_context, false );

        assertEquals( result, "blah" );
    }

    public void testRecursiveObjResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${obj1}", m_context, false );

        assertEquals( result, OBJ1 );
    }

    public void testRecursiveObjResolveToText()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${obj1} ", m_context, false );

        assertEquals( result, OBJ1 + " " );
    }

    public void testRecursiveDualObjResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( " ${obj1} ${obj2} ", m_context, false );

        assertEquals( result, " " + OBJ1 + " " + OBJ2 + " " );
    }

    public void testRecursiveRecurseObjResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${res1}", m_context, false );

        assertEquals( result, new Integer( 1 ) );
    }

    public void testRecursiveRecurseObjResolve2()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${res2}", m_context, false );

        assertEquals( result, new Integer( 2 ) );
    }

    public void testRecursiveNullObjResolve()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${blahaaa}", m_context, true );

        assertEquals( result, "" );
    }

    public void testRecursiveNullObjResolveForException()
        throws PropertyException
    {
        try
        {
            final Object result =
                PropertyUtil.recursiveResolveProperty( "${blahaaa}", m_context, false );
        }
        catch( final PropertyException pe )
        {
            return;
        }
        fail( "NUll resolve occured without exception" );
    }
    
    public void testKey1()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${${key1}}", m_context, false );
        assertEquals( result, OBJ1 );
    }

    public void testKey2AsString()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${${key2}} ", m_context, false );
        assertEquals( result, OBJ2 + " " );
    }

    public void testKey54Peers()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${${key5}${key3}} ", m_context, false );
        assertEquals( result, OBJ2 + " " );
    }

    public void testKey64Recursive()
        throws PropertyException
    {
        final Object result =
            PropertyUtil.recursiveResolveProperty( "${${${key6}${key3}}} ", m_context, false );
        assertEquals( result, OBJ2 + " " );
    }
/*
  "key1", "obj1"
  "key2", "obj2"
  "key3", "2"
  "key4", "1"
  "key5", "obj"
  "key6" "key"
 */
}
