// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "decoder_ogg.h"
#include "constants.h"
#include "buffer.h"
#include "output.h"
#include "recycler.h"

#include <qobject.h>
#include <qiodevice.h>


// static functions for OggVorbis

static size_t oggread (void *buf, size_t size, size_t nmemb, void *src) {
    if (! src) return 0;

    DecoderOgg *dogg = (DecoderOgg *) src;
    int len = dogg->input()->readBlock((char *) buf, (size * nmemb));
    return len / size;
}


static int oggseek(void *src, int64_t offset, int whence) {
    DecoderOgg *dogg = (DecoderOgg *) src;

    if (! dogg->input()->isDirectAccess())
	return -1;

    long start = 0;
    switch (whence) {
    case SEEK_END:
        start = dogg->input()->size();
        break;

    case SEEK_CUR:
        start = dogg->input()->at();
        break;

    case SEEK_SET:
    default:
        start = 0;
    }

    if (dogg->input()->at(start + offset))
        return 0;
    return -1;
}


static int oggclose(void *src)
{
    DecoderOgg *dogg = (DecoderOgg *) src;
    dogg->input()->close();
    return 0;
}


static long oggtell(void *src)
{
    DecoderOgg *dogg = (DecoderOgg *) src;
    long t = dogg->input()->at();
    return t;
}


// Decoder class

DecoderOgg::DecoderOgg(DecoderFactory *d, QIODevice *i, Output *o)
    : Decoder(d, i, o)
{
    inited = FALSE;
    user_stop = FALSE;
    stat = 0;
    output_buf = 0;
    output_bytes = 0;
    output_at = 0;
    bks = 0;
    done = FALSE;
    finish = FALSE;
    len = 0;
    freq = 0;
    bitrate = 0;
    seekTime = -1.0;
    totalTime = 0.0;
    chan = 0;
    output_size = 0;
}


DecoderOgg::~DecoderOgg()
{
    deinit();

    if (output_buf)
	delete [] output_buf;
    output_buf = 0;
}


void DecoderOgg::stop()
{
    user_stop = TRUE;
}


void DecoderOgg::flush(bool final)
{
    ulong min = final ? 0 : bks;

    while ((! done && ! finish) && output_bytes > min) {
	output()->recycler()->mutex()->lock();

	while ((! done && ! finish) && output()->recycler()->full()) {
	    mutex()->unlock();

	    output()->recycler()->cond()->wait(output()->recycler()->mutex());

	    mutex()->lock();
	    done = user_stop;
	}

	if (user_stop || finish) {
	    inited = FALSE;
	    done = TRUE;
	} else {
	    ulong sz = output_bytes < bks ? output_bytes : bks;
	    Buffer *b = output()->recycler()->get();

	    memcpy(b->data, output_buf, sz);
	    if (sz != bks) memset(b->data + sz, 0, bks - sz);

	    b->nbytes = bks;
	    b->rate = bitrate;
	    output_size += b->nbytes;
	    output()->recycler()->add();

	    output_bytes -= sz;
	    memmove(output_buf, output_buf + sz, output_bytes);
	    output_at = output_bytes;
	}

	if (output()->recycler()->full()) {
	    output()->recycler()->cond()->wakeOne();
	}

	output()->recycler()->mutex()->unlock();
    }
}


bool DecoderOgg::initialize()
{
    bks = blockSize();

    inited = user_stop = done = finish = FALSE;
    len = freq = bitrate = 0;
    stat = chan = 0;
    output_size = 0;
    seekTime = -1.0;
    totalTime = 0.0;

    if (! input()) {
	error("DecoderOgg: cannot initialize.  No input.");

	return FALSE;
    }

    if (! output_buf)
	output_buf = new char[globalBufferSize];
    output_at = 0;
    output_bytes = 0;

    if (! input()->isOpen()) {
	if (! input()->open(IO_ReadOnly)) {
	    error("DecoderOgg: Failed to open input. Error " +
		  QString::number(input()->status()) + ".");
	    return FALSE;
	}
    }

    ov_callbacks oggcb =
	{
	    oggread,
	    oggseek,
	    oggclose,
	    oggtell
	};
    if (ov_open_callbacks(this, &oggfile, NULL, 0, oggcb) < 0) {
	error("DecoderOgg: Cannot open stream.");

	return FALSE;
    }

    freq = 0;
    bitrate = ov_bitrate(&oggfile, -1) / 1000;
    chan = 0;

    totalTime = long(ov_time_total(&oggfile, 0));
    totalTime = totalTime < 0 ? 0 : totalTime;

    vorbis_info *ogginfo = ov_info(&oggfile, -1);
    if (ogginfo) {
	freq = ogginfo->rate;
	chan = ogginfo->channels;
    }

    if (output())
	output()->configure(freq, chan, 16, bitrate);

    inited = TRUE;
    return TRUE;
}


double DecoderOgg::lengthInSeconds()
{
    if (! inited)
	return 0;

    return totalTime;
}


void DecoderOgg::seek(double pos)
{
    seekTime = pos;
}


void DecoderOgg::deinit()
{
    ov_clear(&oggfile);

    inited = user_stop = done = finish = FALSE;
    len = freq = bitrate = 0;
    stat = chan = 0;
    output_size = 0;
    setInput(0);
    setOutput(0);
}

void DecoderOgg::run()
{
    mutex()->lock();

    if (! inited) {
	mutex()->unlock();

	return;
    }

    stat = DecoderEvent::Decoding;

    mutex()->unlock();

    {
	DecoderEvent e((DecoderEvent::Type) stat);
	dispatch(e);
    }

    int section = 0;

    while (! done && ! finish) {
	mutex()->lock();
	// decode

	if (seekTime >= 0.0) {
	    ov_time_seek(&oggfile, double(seekTime));
	    seekTime = -1.0;

	    output_size = long(ov_time_tell(&oggfile)) * long(freq * chan * 2);
	}

	len = ov_read(&oggfile, (char *) (output_buf + output_at), bks, 0, 2, 1,
		      &section);

	if (len > 0) {
	    bitrate = ov_bitrate_instant(&oggfile) / 1000;

	    output_at += len;
	    output_bytes += len;

	    if (output())
		flush();
	} else if (len == 0) {
	    flush(TRUE);

	    if (output()) {
		output()->recycler()->mutex()->lock();
		// end of stream
		while (! output()->recycler()->empty() && ! user_stop) {
		    output()->recycler()->cond()->wakeOne();
		    mutex()->unlock();
		    output()->recycler()->cond()->wait(output()->recycler()->mutex());
		    mutex()->lock();
		}
		output()->recycler()->mutex()->unlock();
	    }

	    done = TRUE;
	    if (! user_stop) {
		finish = TRUE;
	    }
	} else {
	    // error in read
	    error("DecoderOgg: Error while decoding stream, File appears to be "
		  "corrupted");

	    finish = TRUE;
	}

	mutex()->unlock();
    }

    mutex()->lock();

    if (finish)
	stat = DecoderEvent::Finished;
    else if (user_stop)
	stat = DecoderEvent::Stopped;

    mutex()->unlock();

    {
	DecoderEvent e((DecoderEvent::Type) stat);
	dispatch(e);
    }

    deinit();
}


// DecoderOggFactory

bool DecoderOggFactory::supports(const QString &source) const
{
    return (source.right(extension().length()).lower() == extension());
}


const QString &DecoderOggFactory::extension() const
{
    static QString ext(".ogg");
    return ext;
}


const QString &DecoderOggFactory::description() const
{
    static QString desc("Ogg Vorbis Audio");
    return desc;
}


Decoder *DecoderOggFactory::create(QIODevice *input,
				   Output *output,
				   bool deletable)
{
    if (deletable)
	return new DecoderOgg(this, input, output);

    static DecoderOgg *decoder = 0;
    if (! decoder) {
	decoder = new DecoderOgg(this, input, output);
    } else {
	decoder->setInput(input);
	decoder->setOutput(output);
    }

    return decoder;
}
