/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_plugin_example.mod example plugin

    \section examplesection1 SYNOPSIS
    \b  lcmaps_plugin_example.mod [options ...]
  
    This example plugin shows how the plugin interface can be implemented.
    Currently no options available.
*/

/*!
    \file   lcmaps_plugin_example.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains a reference implementation of the LCMAPS plugin interface.
*/
/*!
    \defgroup LcmapsPluginInterface The interface to be implemented by LCMAPS plugins

    This is the interface that LCMAPS the plugin have to implement.
*/
/*@{*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcmaps_config.h"
#ifdef LCMAPS_GSI_MODE
#   include <gssapi.h>
#endif

#include <stdio.h>

#include "lcmaps_modules.h"
#include "lcmaps_arguments.h"
#include "lcmaps_plugin_prototypes.h"

/******************************************************************************
                                Definitions
******************************************************************************/

/******************************************************************************
                          Module specific prototypes
******************************************************************************/

/******************************************************************************
                       Define module specific variables
******************************************************************************/

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
/*!
    \fn plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
        )
    \brief Plugin asks for required arguments

    \param argc  the number of arguments requested by this plugin
    \param argv  the (empty) list of arguments requested by this plugin
    \retval LCMAPS_MOD_SUCCESS success
    \retval LCMAPS_MOD_FAIL    failure (will result in a lcmaps failure)
*/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    static lcmaps_argument_t argList[] = {
        { "job_request"  , "lcmaps_request_t" ,  1,   NULL},
#ifdef LCMAPS_GSI_MODE
        { "user_cred"    , "gss_cred_id_t"    ,  0,   NULL},
#endif
        { "user_dn"      , "char *"           ,  0,   NULL},
        { "job_request"  , "char *"           ,  0,   NULL},
        { "mapcounter"   , "int"           ,  0,   NULL},
        { NULL           , NULL               , -1,   NULL}
    };

    lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_introspect(): introspecting\n");

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_introspect(): address first argument: %p\n",(void*)argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
/*!
    \fn plugin_initialize(
        int argc,
        char ** argv
        )
    \brief initialize the plugin.
    
    Everything that is needed to initialize the plugin should be put inside this function.
    Arguments as read from the LCMAPS database (argc, argv) are passed to the plugin.

    \param argc number of passed arguments.
    \param argv argument list. argv[0] contains the name of the plugin.
    \retval LCMAPS_MOD_SUCCESS successful initialization
    \retval LCMAPS_MOD_FAIL    failure in the plugin initialization
    \retval LCMAPS_MOD_NOFILE  private plugin database could not be found (same effect as
                             LCMAPS_MOD_FAIL)
*/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    int i;

    lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_initialize(): passed arguments:\n");
    for (i=0; i < argc; i++)
    {
        lcmaps_log_debug(2,"lcmaps_plugin_example-plugin_initialize(): arg %d is %s\n",
             i,argv[i]);
    }

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
/*!
    \fn plugin_run(
        int argc,
        lcmaps_argument_t * argv
        )
    \brief Gather credentials for user making use of the ordered arguments

    Ask for credentials by passing the arguments (like JDL, globus DN, VOMS roles etc.)
    that were ordered earlier by the plugin_introspect() function

    \param argc number of arguments
    \param argv list of arguments
    \retval LCMAPS_MOD_SUCCESS authorization succeeded
    \retval LCMAPS_MOD_FAIL    authorization failed
*/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    lcmaps_request_t * prequest=NULL;
#ifdef LCMAPS_GSI_MODE
    gss_cred_id_t * pcred=NULL;
    gss_cred_id_t cred;
#endif

    char ** pstring=NULL;
    int *   pmapcounter=NULL;
    void *  value=NULL;

    lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run():\n");

    /*
     * Try to get the ordered values:
     */
    if ( ( value = lcmaps_getArgValue("mapcounter", "int", argc, argv) ) )    {
	pmapcounter = (int *) value;
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): mapcounter: %d\n",*pmapcounter);
    }
    else
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): could not get value of mapcounter !\n");

    if ( ( value = lcmaps_getArgValue("job_request", "char *", argc, argv) ) )  {
	pstring = (char **) value;
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): job_request: %s\n",*pstring);
    }
    else
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): could not get value of job_request !\n");

    if ( ( value = lcmaps_getArgValue("user_dn", "char *", argc, argv) ) )  {
	pstring = (char **) value;
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): user_dn: %s\n",*pstring);
    }
    else
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): could not get value of user_dn !\n");

    if ( ( value = lcmaps_getArgValue("job_request", "lcmaps_request_t", argc, argv) ) )    {
	prequest = (lcmaps_request_t *) value;
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): job_request: %s\n",*prequest);
    }
    else
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): could not get value of job_request !\n");

#ifdef LCMAPS_GSI_MODE
    if ( ( value = lcmaps_getArgValue("user_cred", "gss_cred_id_t", argc, argv) ) )
    {
	pcred = (gss_cred_id_t *) value;
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): address user_cred: %p\n",(void*)pcred);
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): value user_cred: %p\n",(void*)*pcred);
        cred=*pcred;
        if (cred) {
            lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): inside value user_cred: %d\n",*(int *)(cred));
        }
    }
    else
        lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): could not get address of user_cred !\n");
#endif

    lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_run(): address first argument: %p\n",(void*)argv);

    /* succes */
    return LCMAPS_MOD_SUCCESS;

/* fail_example:
    return LCMAPS_MOD_FAIL;*/
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
/*!
    \fn plugin_terminate()
    \brief Whatever is needed to terminate the plugin module goes in here.
    \retval LCMAPS_MOD_SUCCESS success
    \retval LCMAPS_MOD_FAIL    failure (will result in an authorization failure)
*/
int plugin_terminate(void)
{
    lcmaps_log_debug(1,"lcmaps_plugin_example-plugin_terminate(): terminating\n");

    return LCMAPS_MOD_SUCCESS;
}
/*@}*/

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/examples/lcmaps_plugin_example.c,v $
    $Date: 2014-07-07 21:28:25 +0200 (Mon, 07 Jul 2014) $
    $Revision: 17839 $
    $Author: msalle $
******************************************************************************/
