/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_voms_poolgroup.mod voms poolgroup plugin

    \section vomspoolgroupsyn SYNOPSIS

    \b lcmaps_voms_poolgroup.mod
        -GROUPMAPFILE|-groupmapfile|-GROUPMAP|-groupmap \<groupmapfile\>
        -GROUPMAPDIR|-groupmapdir \<groupmapdir\>
        [-mapall] [-mapmin \<group count\>]

    \section vomspoolgroupdesc DESCRIPTION

    The poolgroup acquisition plugin is a 'VOMS-aware' plugin.
    It uses the VOMS information (acquired by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod")
    to gather primary and secondary GIDs.
    This is accomplished by matching VO-GROUP-ROLE(-CAPABILITY) combinations in the so-called
    \e groupmapfile (gridmapfile style) and by finding the corresponding 'poolgroup' (similar to
    the 'poolaccount' procedure, see \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod")
    Wildcards can be used in the groupmapfile to match VO-GROUP-ROLE combinations.

    EXAMPLE 'groupmapfile':

    \c "/VO=atlas/GROUP=mcprod"   \c mcprod

    \c "/VO=atlas/GROUP=mcprod"   \c .atlas

    \c "/VO=atlas/GROUP=dev"      \c .atlas

    \c "/VO=atlas/GROUP=*"        \c .atlas

    The VO-GROUP-ROLE combination \c "/VO=atlas/GROUP=mcprod" starts with an alfanumeric character
    (not ".") and indicates a localgroup entry in the groupmapfile (will be resolved by the
    \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod").
    The VO-GROUP-ROLE combination \c "/VO=atlas/GROUP=*"
    indicates that all users from the Atlas VO with every other group than 'mcprod'
    will be mapped to the '.atlas' pool of (system) groups.
    Just like the \e poolaccount plugin this plugin will link an entry
    (in this case a VO-GROUP-ROLE combination)
    to a locally known group (a.k.a. poolgroup) in the \e groupmapdir directory.
    The difference with the \e poolaccount plugin is that there is not a Distinghuished Name
    but a VO-GROUP-ROLE combination and there is no poolaccount but poolgroup defined in de
    groupmapfile (similar to the gridmapfile).
    Instead of the \e gridmapdir the \e groupmapdir directory is used
    for the registration of thew mapping between poolgroups and the VO-GROUP-ROLE combination.

    As you can see the in the example the 'mcprod' GROUP can be found by using the localgroup
    plugin and the poolgroup plugin.
    With the poolgroup plugin there can be made a mapping between "/VO=atlas/GROUP=mcprod" and the group
    'atlas001' (based on the .atlas pool).
    The entry \c "/VO=atlas/GROUP=dev" will also result in a group from this '.atlas' pool,
    but a different one, e.g. 'atlas002'.
    Finally, we have random other groups not predefined in the groupmapfile, for example
    \c "/VO=atlas/GROUP=foo", which matches \c "/VO=atlas/GROUP=*" in the groupmapfile.
    This VO-GROUP combination will be mapped to a poolgroup (probably) called 'atlas003'.

    The poolgroup plugin will try to match each VO-GROUP-ROLE combination that was
    found by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod".
    The first VO-GROUP-ROLE combination will become the primary group, the others secondary groups.
    As the primary GID may be used for auditing and accounting purposes it is important that
    the user uses the correct ordering of VO-GROUP-ROLE combinations in his grid credential (X509
    certificate).

    \section vomspoolgroupoptions OPTIONS
    \subsection vomspoolgroupoptie1 -GROUPMAPFILE \<groupmapfile\>
        See \ref vomspoolgroupoptie4 "-groupmap"

    \subsection vomspoolgroupoptie2 -groupmapfile \<groupmapfile\>
        See \ref vomspoolgroupoptie4 "-groupmap"

    \subsection vomspoolgroupoptie3 -GROUPMAP \<groupmapfile\>
        See \ref vomspoolgroupoptie4 "-groupmap"

    \subsection vomspoolgroupoptie4 -groupmap \<groupmapfile\>
        If this option is set, it will override the default path to the groupmapfile.
        It is advised to use an absolute path to the groupmapfile to avoid usage of the wrong file(path).

    \subsection vomspoolgroupoptie5 -GROUPMAPDIR \<groupmapdir\>
        See \ref vomspoolgroupoptie6 "-groupmapdir"

    \subsection vomspoolgroupoptie6 -groupmapdir \<groupmapdir\>
        Here you can override the default directory path to the 'groupmapdir'.
        This directory is just like the \e gridmapdir and holds all the poolgroup mappings
        that has/will be made by linking filenames to a i-node indicating a mapping
        between a VO-GROUP-ROLE combination and a (system) group or GID.

    \subsection vomspoolgroupoptie7 -mapall
        If this parameter is set, the plugin only succeeds if it manages to map all voms data entries
        to (system) groups and find their GID.
        There is no communication between different plugins (like the voms_localgroup plugin)
        about the failures.
        A log entry will state the VO-GROUP-ROLE combination that made the plugin fail.

    \subsection vomspoolgroupoptie8 -OVERRIDE_INCONSISTENCY
        See \ref vomspoolgroupoptie9 "-override_inconsistency"

    \subsection vomspoolgroupoptie9 -override_inconsistency
        Moving a VO group from one pool to another
        should only be done by changing the groupmapfile indicating the new pool for this VO group.
        If a VO group has already been mapped previously to a poolaccount, there is a link present
        between this poolgroup and its VO-GROUP-ROLE combination.
        By default the voms_poolgroup plugin will \e fail if the pool designated by the gridmapfile
        doesn't match the previously mapped poolgroup leasename.
        If the site doesn't want a failure on this inconsistency it can turn on this parameter.
        When the inconsistency is detected the plugin will automatically unlink the previous mapping
        and will proceed by making a \e new lease from the new pool.

    \subsection vomspoolgroupoptie10 -mapmin \<group count\>
        This option will set a minimum amount of groups that have to be resolved for later mapping.
        If the minimum is not set then the minimum amount is set to '0' by default.
        If the plugin is not able to the required number of poolgroups it will fail.
        Note: if the minimum is set to zero or the minimum is not set
        the plugin will return a success if no other errors occur, even if no poolgroups were found.

    \subsection vomspoolaccountoptie12 -strict_poolprefix_match [yes|no]. Default is 'yes'.
        If this is set to 'yes', a line in the groupmapfile like
        <FQAN> .poolgr
        will result in groups matching the regexp 'poolgr[0-9]+'.
        Otherwise it will be allowed to match 'poolgr.*' (legacy behaviour).
    \section vomspoolgroupReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure

    \section vomspoolgroupErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)

    \section vomspoolgroupSeeAlso SEE ALSO
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod",
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
*/

/*!
    \file   lcmaps_voms_poolgroup.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the voms_poolgroup plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <ctype.h>
#include <sys/stat.h>

#include "lcmaps_voms_config.h"
#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
#include "lcmaps_gridlist.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static char *groupmapfile            = NULL;
static char *groupmapdir             = NULL;
static int   mapall                  = 0;
static int   override_inconsistency  = 0;
static int   mapmin                  = 0;
static int   strict_poolprefix_match = 1; /* By default strict matching */
static int   map_to_secondary_groups = 0;

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char *  logstr = "lcmaps_plugin_voms_poolgroup-plugin_initialize()";
    int i;
    size_t j;
    struct stat s;

    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * the first will be the thing to edit/select (groupmap(file))
     */

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        if ( ((strcmp(argv[i], "-groupmap") == 0) ||
              (strcmp(argv[i], "-GROUPMAP") == 0) ||
              (strcmp(argv[i], "-groupmapfile") == 0) ||
              (strcmp(argv[i], "-GROUPMAPFILE") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                /* check if the setting exists */
                if (stat (argv[i + 1], &s) < 0)
                {
                    lcmaps_log(LOG_ERR, "%s: Error: groupmapfile not accessible at \"%s\"\n", logstr, argv[i + 1]);
                    return LCMAPS_MOD_FAIL;
                }
                groupmapfile = strdup(argv[i + 1]);
            }
            i++;
        }
        else if ( ((strcmp(argv[i], "-groupmapdir") == 0) ||
              (strcmp(argv[i], "-GROUPMAPDIR") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                /* check if the setting exists */
                if (stat (argv[i + 1], &s) < 0)
                {
                    lcmaps_log(LOG_ERR, "%s: Error: groupmapdir not accessible at \"%s\"\n", logstr, argv[i + 1]);
                    return LCMAPS_MOD_FAIL;
                }
                groupmapdir = strdup(argv[i + 1]);
            }
            i++;
        }
        else if (strcmp(argv[i], "--map-to-secondary-groups") == 0)
        {
             map_to_secondary_groups = 1;
        }
        else if (strcmp(argv[i], "-mapall") == 0)
        {
             mapall = 1;
        }
        else if ( (strcmp(argv[i], "-override_inconsistency") == 0) ||
                  (strcmp(argv[i], "-OVERRIDE_INCONSISTENCY") == 0))
        {
            override_inconsistency = 1;
        }
        else if ((strcmp(argv[i], "-mapmin") == 0)
                 && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 /* check parameter integrety */
                 for (j = 0; j < (strlen(argv[i + 1])); j++)
                 {
                     if (isdigit((argv[i + 1])[j]) == 0)
                     {
                         lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (%s is not a number)\n", logstr, argv[i], argv[i + 1]);
                         return LCMAPS_MOD_FAIL;
                     }
                 }

                 mapmin = atoi(argv[i + 1]);
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-strict_poolprefix_match") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 if (strcmp(argv[i+1],"yes") == 0)
                 {
                     strict_poolprefix_match = 1;
                 }
                 else if (strcmp(argv[i+1],"no") == 0)
                 {
                     strict_poolprefix_match = 0;
                 }
                 else
                 {
                     lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n", logstr, argv[i]);
                     return LCMAPS_MOD_FAIL;
                 }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                return LCMAPS_MOD_FAIL;
            }
            i++;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr,
                       argv[i]);
            return LCMAPS_MOD_FAIL;
        }
    }

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char *                   logstr = "lcmaps_plugin_voms_poolgroup-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        {"user_dn"              ,   "char *"    ,   0   ,   NULL},
        {"fqan_list"            ,   "char **"   ,   0   ,   NULL},
        {"nfqan"                ,   "int"       ,   0   ,   NULL},
        {"requested_pgid_list"  ,   "gid_t *"   ,   0   ,   NULL},
        {"requested_npgid"      ,   "int"       ,   0   ,   NULL},
        {"requested_sgid_list"  ,   "gid_t *"   ,   0   ,   NULL},
        {"requested_nsgid"      ,   "int"       ,   0   ,   NULL},
        {NULL                   ,   NULL        ,   -1  ,   NULL}
    };

    lcmaps_log_debug(5,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr, argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().

    Policy: This method will not check if the gids found are also part of the
            list gids requested for the user. This is deferred to the stage
            after which all plugins have run.

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char *                  logstr = "lcmaps_plugin_voms_poolgroup-plugin_run()";
    char *                  groupname           = NULL;
    struct group *          group_info          = NULL;
    int                     i                   = 0;
    unsigned short          matching_type       = ((unsigned short)0x0000);
    int                     group_counter       = 0;
    int                     rc                  = 0;
    lcmaps_vo_mapping_t *   lcmaps_vo_mapping   = NULL;
    char **                 fqan_list           = NULL;
    int                     nfqan               = -1;
    int                     requested_npgid     = 0;
    gid_t *                 requested_pgid_list = NULL;
    int                     requested_nsgid     = 0;
    gid_t *                 requested_sgid_list = NULL;
    void *                  value               = NULL;

    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_voms_poolgroup-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_voms_poolgroup-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_voms_poolgroup-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_voms_poolgroup;
    }
    lcmaps_log_debug(5,"%s:\n", logstr);


    /*
     * Try to fetch the list of groups the invocator of LCMAPS wants to be
     * verified. (only in PLUGIN_VERIFY mode).
     */
    if (lcmaps_mode == PLUGIN_VERIFY)
    {
        if ( (value = lcmaps_getArgValue("requested_npgid", "int", argc, argv)) != NULL )
        {
	    requested_npgid = *(int *) value;
            lcmaps_log_debug(1,"%s: the list of pgids should contain %d elements\n", logstr, requested_npgid);
            if ( ( value = lcmaps_getArgValue("requested_pgid_list", "gid_t *", argc, argv) ) != NULL  )   {
		requested_pgid_list = *(gid_t **) value;
                lcmaps_log_debug(1, "%s: found list of pgids\n", logstr);
	    }
            else
            {
                lcmaps_log_debug(1, "%s: could not retrieve list of pgids (failure)!\n", logstr);
                goto fail_voms_poolgroup;
            }
            for (i = 0; i < requested_npgid; i++)
            {
                lcmaps_log_debug(3, "%s: pgid[%d]: %d\n", logstr, i, (int)(requested_pgid_list[i]));
            }
        }
        if ( (value = lcmaps_getArgValue("requested_nsgid", "int", argc, argv)) != NULL )
        {
	    requested_nsgid = *(int *) value;
            lcmaps_log_debug(1,"%s: the list of sgids should contain %d elements\n", logstr, requested_nsgid);
            if ( ( value = lcmaps_getArgValue("requested_sgid_list", "gid_t *", argc, argv) ) != NULL )   {
		requested_sgid_list = *(gid_t **) value;
                lcmaps_log_debug(1, "%s: found list of sgids\n", logstr);
	    }
            else
            {
                lcmaps_log_debug(1, "%s: could not retrieve list of sgids (failure)!\n", logstr);
                goto fail_voms_poolgroup;
            }
            for (i = 0; i < requested_nsgid; i++)
            {
                lcmaps_log_debug(3, "%s: sgid[%d]: %d\n", logstr, i, (int)(requested_sgid_list[i]));
            }
        }
    }

    /*
     * Get the VO user information.
     * We can either order it by lcmaps_argument_t or use the getCredentialData() function.
     * The latter case requires the voms parsing plugin (lcmaps_voms.mod) to have run beforehand.
     * Unfortunately the formats of the VOMS strings (from getCredentialData()) and
     * FQANs (from lcmaps_argument_t) are not the same. We may have to introduce
     * two-way conversion functions.
     * The VOMS info has to matched against the info in the gridmapfile
     */
    lcmaps_log_debug(5,"%s: First try to get the FQAN list from input credential repository ...\n", logstr);
    if ( ( value = lcmaps_getArgValue("nfqan", "int", argc, argv) ) != NULL )
    {
	nfqan = *(int *) value;
	if (nfqan < 1)	{
	    lcmaps_log(LOG_ERR,"%s: no (valid) VOMS groups found --> no mapping\n", logstr);
	    goto fail_voms_poolgroup;
	}

        lcmaps_log_debug(5,"%s: the list of FQANs should contain %d elements\n", logstr, nfqan);
        if ( ( value = lcmaps_getArgValue("fqan_list", "char **", argc, argv) ) != NULL )   {
	    fqan_list = *(char ***) value;
            lcmaps_log_debug(5, "%s: found list of FQANs\n", logstr);
	}
        else
        {
            lcmaps_log(LOG_NOTICE, "%s: could not retrieve list of FQANs (failure)!\n", logstr);
            goto fail_voms_poolgroup;
        }
        for (i = 0; i < nfqan; i++)
        {
            lcmaps_log_debug(3, "%s: FQAN %d: %s\n", logstr, i, fqan_list[i]);
        }
    }
    else
    {
        lcmaps_log_debug(1,"%s: ... did not find input credentials in input credential repository ... trying the internal credential repository ...\n", logstr);
        fqan_list = getCredentialData(LCMAPS_VO_CRED_STRING, &nfqan);
    }

    if (nfqan == 0)
    {
        lcmaps_log(LOG_ERR,"%s: no VOMS group info --> no mapping (failure)\n", logstr);
        goto fail_voms_poolgroup;
    }
    else if (nfqan < 0)
    {
        lcmaps_log(LOG_ERR,"%s: negative number of VOMS groups found ! (failure)\n", logstr);
        goto fail_voms_poolgroup;
    }

    /*
     * Check the groupmapfile
     */

    if ((groupmapfile != NULL) && (strlen(groupmapfile) > 0))
        lcmaps_log_debug(1,"%s: groupmapfile is: %s\n", logstr, groupmapfile);
    else
    {
        lcmaps_log(LOG_ERR,"%s: error finding the groupmapfile: %s. (use the option \"-groupmapfile <groupmapfile>\"\n", logstr, groupmapfile);
        goto fail_voms_poolgroup;
    }

    /*
     * Check groupmapdir
     */
    if (groupmapdir == NULL) /* try if GROUPMAPDIR is already set */
    {
        char * tmpptr=NULL;
        if ((tmpptr = getenv("GROUPMAPDIR")) == NULL)
        {
            lcmaps_log(LOG_ERR,"%s: GROUPMAPDIR unknown! Specify as option or set GROUPMAPDIR\n", logstr);
            goto fail_voms_poolgroup;
        }
        else
        {
            groupmapdir = strdup(tmpptr);
        }
    }
    if (strlen(groupmapdir) == 0)
    {
        lcmaps_log(LOG_ERR,"%s: cannot set MAPDIR (strlen(groupmapdir) == 0)\n", logstr);
        goto fail_voms_poolgroup;
    }
    lcmaps_log_debug(1,"%s: setting MAPDIR to %s\n", logstr, groupmapdir);
    if (setenv("MAPDIR", groupmapdir, 1))
    {
        lcmaps_log(LOG_ERR,"%s: cannot set MAPDIR\n", logstr);
        goto fail_voms_poolgroup;
    }

    /*
     * Try to find the unix groups from the VO info in the groupmapfile
     * The first group (if found) should become the primary group
     */

    matching_type = MATCH_INCLUDE|MATCH_WILD_CHARS;

    /* if override_consistency is set add this to the matchin_type so it will take effect */
    if (override_inconsistency)
        matching_type = matching_type|OVERRIDE_INCONSISTANCY;

    /* if strict_poolprefix_match is set add this to the matchin_type so it will take effect */
    if (strict_poolprefix_match)
        matching_type = matching_type|MATCH_STRICT_PREFIX_NUM;

    /* Do not create new leases in verification mode */
    if (lcmaps_mode == PLUGIN_VERIFY)
        matching_type = matching_type|ONLY_USE_EXISTING_LEASE;

    for (i = 0; i < nfqan; i++)
    {
        /* clean groupname before each call to lcmaps_gridlist */
        if (groupname) free(groupname);
        groupname = NULL;
        if ( (rc = lcmaps_gridlist(fqan_list[i], &groupname, groupmapfile, matching_type, ".", NULL)) == 0)
        {
            lcmaps_log_debug(4,"%s: found groupname: %s\n", logstr, groupname);
            group_counter++;

            if (groupname && (strlen(groupname) > 0))
            {
                if ( ( group_info = getgrnam(groupname) ) )
                {
                    /* When map_to_secondary_groups is true, all results will be stored as secondary Unix group IDs */
                    if ((i == 0) && (!map_to_secondary_groups))
                    {
                        /* First VO group */
                        addCredentialData(PRI_GID, (void *) &(group_info->gr_gid));
                    }
                    else
                    {
                        /* Other VO groups */
                        addCredentialData(SEC_GID, (void *) &(group_info->gr_gid));
                    }
                    /*
                     * The coupling between VO information and the GID is maintained
                     * in the lcmaps_vo_mapping structure, which is added to the credential data
                     */
                    lcmaps_vo_mapping=lcmaps_createVoMapping(
                        fqan_list[i],
                        groupname,
                        group_info->gr_gid
                    );
                    if (! lcmaps_vo_mapping)
                    {
                        lcmaps_log(LOG_ERR,"%s: could not create VoMapping structure (failure)\n", logstr);
                        goto fail_voms_poolgroup;
                    }
                        /* lcmaps_printVoMapping(2, lcmaps_vo_mapping); */
                    /* Add credential */
                    addCredentialData(LCMAPS_VO_CRED_MAPPING, (void *) lcmaps_vo_mapping);
                    if ( lcmaps_deleteVoMapping(&lcmaps_vo_mapping) )
                    {
                        lcmaps_log(LOG_ERR,"%s: error while deleting VoMapping structure (failure)\n", logstr);
                        goto fail_voms_poolgroup;
                    }
                }
                else
                {
                    lcmaps_log(LOG_ERR,"%s: no group id found in /etc/group (or equivalent, e.g. LDAP) for groupname = \"%s\"\n", logstr, groupname);
                    goto fail_voms_poolgroup;
                }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: error getting value of groupname (failure)!\n", logstr);
                goto fail_voms_poolgroup;
            }
        }
        else if (rc == LCMAPS_MOD_NOFILE)
        {
            lcmaps_log(LOG_ERR, "%s: Could not find the groupmapfile %s\n", logstr, groupmapfile);
            goto fail_voms_poolgroup;
        }
        else
        {
            lcmaps_log_debug(1,"%s: could not get value of groupname !\n", logstr);
            if (mapall)
            {
                lcmaps_log(LOG_NOTICE,"%s: no mapping for VO group %s\n", logstr, fqan_list[i]);
                goto fail_voms_poolgroup;
            }
        }
    }

    if (group_counter < mapmin)
    {
        lcmaps_log(LOG_ERR,"%s: Not enough groups found. The minimum is set to %d. The plugin found %d\n", logstr, mapmin, group_counter);
        goto fail_voms_poolgroup;
    }

    /* success */
/* success_voms_poolgroup:*/
    if (groupname) free(groupname);
    lcmaps_log(LOG_INFO,"%s: voms_poolgroup plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_voms_poolgroup:
    if (groupname) free(groupname);
    lcmaps_log(LOG_INFO,"%s: voms_poolgroup plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate(void)
{
    char * logstr = "lcmaps_plugin_voms_poolgroup-plugin_terminate()";

    lcmaps_log_debug(5,"%s: terminating\n", logstr);

    if (groupmapfile) free(groupmapfile);
    if (groupmapdir) free(groupmapdir);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-voms/src/voms/lcmaps_voms_poolgroup.c,v $
    $Date: 2010-02-19 06:01:37 $
    $Revision: 1.9 $
    $Author: okoeroo $
******************************************************************************/
