/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtktable.h>
#include <gtk/gtktogglebutton.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "oscilloscope_gui_gtk.h"
#include "potentiometer_gui_gtk.h"
#include "potentiometer_gui.h"
#include "glue-gui-gtk.h"
#include "glue-gui-recorder.h"
#include "glue-gui-gtk-monitor.h"
#include "glue-gui-gtk-potentiometer.h"
#include "glue.h"

#define SCREEN_WIDTH 	400
#define SCREEN_HEIGHT	800


struct cpssp{
	GtkWidget *gui;
	void *recorder;
	void *sec_div_poti;
	void *volts_div_poti;
	void *trigger_poti;
	void *vert_pos_poti;

	struct sig_integer *port_adj_c;
	struct sig_integer *port_adj_d;
	struct sig_integer *port_adj_e;
	struct sig_integer *port_adj_f;
};

static void
oscilloscope_gui_gtk_pixel_set(
	void *_cpssp,
	unsigned int x,
	unsigned int y,
	uint8_t r,
	uint8_t g,
	uint8_t b
)
{
	struct cpssp *cpssp = (struct cpssp*) _cpssp;
	gui_gtk_monitor_pixel_set(GUI_GTK_MONITOR(cpssp->gui), x, y, r, g, b);
	gui_recorder_pixel_set(cpssp->recorder, x, y, r, g, b);
}


static void
oscilloscope_gui_gtk_size_set(void *_cpssp, unsigned int width, unsigned int height)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	gui_recorder_size_set(cpssp->recorder, width, height);
	gui_gtk_monitor_size_set(GUI_GTK_MONITOR(cpssp->gui), width, height);
}

static void
oscilloscope_gui_gtk_sync(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	gui_gtk_monitor_sync(GUI_GTK_MONITOR(cpssp->gui));
}


/*
 * GUI Callbacks
 */
static void
oscilloscope_gui_gtk_recording_on_event(GtkWidget *w, gpointer _cpssp)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	gui_recorder_rec_set(cpssp->recorder, 1);
}

static void
oscilloscope_gui_gtk_recording_off_event(GtkWidget *w, gpointer _cpssp)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	gui_recorder_rec_set(cpssp->recorder, 0);
}

static void
trigger_adjustment_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkPotentiometer *poti;
	struct cpssp *cpssp;
	int val;

	poti = (GuiGtkPotentiometer*)w;
	cpssp = (struct cpssp*)_cpssp;
	val = (int) poti->adj->value;
	sig_integer_set(cpssp->port_adj_c, cpssp, val);
}

static void
volts_div_adjustment_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkPotentiometer *poti;
	struct cpssp *cpssp;
	int val;

	poti = (GuiGtkPotentiometer*)w;
	cpssp = (struct cpssp*)_cpssp;
	val = (int) poti->adj->value;
	sig_integer_set(cpssp->port_adj_d, cpssp, val);
}

static void
sec_div_adjustment_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkPotentiometer *poti;
	struct cpssp *cpssp;
	int val;

	poti = (GuiGtkPotentiometer*)w;
	cpssp = (struct cpssp*)_cpssp;
	val = (int) poti->adj->value;
	sig_integer_set(cpssp->port_adj_e, cpssp, val);
}

static void
vert_pos_adjustment_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkPotentiometer *poti;
	struct cpssp *cpssp;
	int val;

	poti = (GuiGtkPotentiometer*)w;
	cpssp = (struct cpssp*)_cpssp;
	val = (int) poti->adj->value;
	sig_integer_set(cpssp->port_adj_f, cpssp, val);
}

void *
oscilloscope_gui_gtk_create(
	unsigned int page,
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_a,
	struct sig_std_logic *port_b,
	struct sig_opt_rgb *port_monitor,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_vcc,
	struct sig_integer *port_adj_c,
	struct sig_integer *port_adj_d,
	struct sig_integer *port_adj_e,
	struct sig_integer *port_adj_f

)
{
	static const struct sig_opt_rgb_funcs opt_video_funcs = {
		.pixel_set = oscilloscope_gui_gtk_pixel_set,
		.size_set = oscilloscope_gui_gtk_size_set,
		.sync = oscilloscope_gui_gtk_sync,
	};
	struct cpssp *cpssp;
	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->port_adj_c = port_adj_c;
	cpssp->port_adj_d = port_adj_d;
	cpssp->port_adj_e = port_adj_e;
	cpssp->port_adj_f = port_adj_f;

	cpssp->gui = gui_gtk_monitor_new("VGA", 1600, 1200);
	cpssp->recorder = gui_recorder_create(1600, 1200);
	gtk_widget_show(cpssp->gui);


	cpssp->trigger_poti = gui_gtk_potentiometer_new("trigger");
	gui_gtk_comp_add(page, "trigger_poti", name, cpssp->trigger_poti, TRUE, TRUE, NULL);
	g_signal_connect(G_OBJECT(cpssp->trigger_poti), "value_changed_gui",
			G_CALLBACK(trigger_adjustment_value_changed),cpssp);


	cpssp->volts_div_poti = gui_gtk_potentiometer_new("volts_div");
	gui_gtk_comp_add(page, "volts_div_poti", name, cpssp->volts_div_poti, TRUE, TRUE, NULL);
	g_signal_connect(G_OBJECT(cpssp->volts_div_poti), "value_changed_gui",
			G_CALLBACK(volts_div_adjustment_value_changed),cpssp);


	cpssp->vert_pos_poti = gui_gtk_potentiometer_new("vert_pos");
	gui_gtk_comp_add(page, "vert_pos_poti", name, cpssp->vert_pos_poti, TRUE, TRUE, NULL);
	g_signal_connect(G_OBJECT(cpssp->vert_pos_poti), "value_changed_gui",
			G_CALLBACK(vert_pos_adjustment_value_changed),cpssp);


	cpssp->sec_div_poti = gui_gtk_potentiometer_new("sec_div");
	gui_gtk_comp_add(page, "sec_div_poti", name, cpssp->sec_div_poti, TRUE, TRUE, NULL);
	g_signal_connect(G_OBJECT(cpssp->sec_div_poti), "value_changed_gui",
			G_CALLBACK(sec_div_adjustment_value_changed),cpssp);

	gui_gtk_monitor_sync(GUI_GTK_MONITOR(cpssp->gui));
	gui_gtk_comp_add(page, "oscilloscope", name, cpssp->gui, TRUE, TRUE, NULL);
	gui_gtk_monitor_grab_focus(GUI_GTK_MONITOR(cpssp->gui));


	g_signal_connect(G_OBJECT(cpssp->gui), "recording-on",
			G_CALLBACK(oscilloscope_gui_gtk_recording_on_event),
			cpssp);
	g_signal_connect(G_OBJECT(cpssp->gui), "recording-off",
			G_CALLBACK(oscilloscope_gui_gtk_recording_off_event),
			cpssp);
	/* In */
	sig_opt_rgb_connect(port_monitor, cpssp, &opt_video_funcs);
	return cpssp;
}

void
oscilloscope_gui_gtk_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
oscilloscope_gui_gtk_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	gui_gtk_monitor_suspend(cpssp->gui, fComp);
}

void
oscilloscope_gui_gtk_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	gui_gtk_monitor_resume(cpssp->gui, fComp);
}

#if 0
/* older, slower cairo version */

typedef struct _OscilloscopeGuiGtk OscilloscopeGuiGtk;
typedef struct _OscilloscopeGuiGtkClass OscilloscopeGuiGtkClass;


struct _OscilloscopeGuiGtk{
	GtkTable table;
	GtkWidget *draw_area;
	volatile int running;
	volatile int zoom_factor;
};

struct _OscilloscopeGuiGtkClass{
	GtkTableClass parent_class;
	void (* oscilloscope) (OscilloscopeGuiGtk *cw);
};

/* older cairo version */
static GType oscilloscope_gui_gtk_get_type (void);
static GtkWidget* oscilloscope_gui_gtk_new (void);


static void oscilloscope_gui_gtk_class_init	(OscilloscopeGuiGtkClass *klass);
static void oscilloscope_gui_gtk_init		(OscilloscopeGuiGtk *cw);

struct cpssp{
	GtkWidget *widget;
	GtkWidget *area;
};

static void
drawOscillator(GtkWidget *widget)
{
	int row, column;
	cairo_t *cr;

	cr = gdk_cairo_create(widget->window);

	//draw black background
	cairo_set_source_rgb(cr, 0, 0, 0);
	cairo_rectangle(cr, 0, 0, SCREENWIDTH, SCREENHEIGHT);
	cairo_fill(cr);

	//draw grid lines
	cairo_set_source_rgb(cr, 0.2, 0.2, 0.2);
	for(row = 0; row < SCREENHEIGHT; row++) {
		for(column = 0; column < SCREENWIDTH; column++) {
			if (column % GRIDSTEP == 0 || row % GRIDSTEP == 0) {
				cairo_rectangle(cr, column, row, 1, 1);
				cairo_fill(cr);
			}
		}
	}
	gui_gtk_flush();
}

static void
oscilloscope_gui_gtk_draw_pixel(void *_cpssp, int x, int y, double r, double g, double b) {
	cairo_t *cr;
	struct cpssp *cpssp;
	OscilloscopeGuiGtk *osc;

	cpssp = (struct cpssp*) _cpssp;
	osc = (OscilloscopeGuiGtk*) cpssp->widget;
	cr = gdk_cairo_create(osc->draw_area->window);
	fprintf(stderr, "draw pixel\n");
	if (r == 1 && g == 1 && b == 1) {
		cairo_set_source_rgb (cr, 0.2, 0.2, 0.2);
	} else {
		cairo_set_source_rgb(cr, r, g, b);
	}
	cairo_move_to(cr, x, y);
	cairo_rectangle(cr, x, y, 1, 1);
	cairo_fill(cr);
	cairo_destroy(cr);
	#if 0
	if (x % GRIDSTEP == 0
	 || x == SCREENWIDTH - 1) {
		gui_gtk_flush();
	}
	#endif

	if (x == (SCREENWIDTH - 1)) {
		gui_gtk_flush();
	}
}

static void
reset_callback(void *_cpssp, unsigned int val)
{
	cairo_t *cr;
	struct cpssp *cpssp;
	OscilloscopeGuiGtk *osc;

	cpssp = (struct cpssp*) _cpssp;
	if (cpssp == NULL) {
		printf("cpssp null\n");
		return;
	}
	osc = (OscilloscopeGuiGtk*) cpssp->widget;
	if (osc == NULL) {
		printf("osc null\n");
		return;
	}
	if (osc->draw_area == NULL || osc->draw_area->window == NULL) {
		printf("osc->draw_area null\n");
		return;
	}
	cr = gdk_cairo_create(osc->draw_area->window);
	if (cr == NULL) {
		printf("osc->draw_area->window null\n");
		return;
	}
}

static void
timer_callback(void *_cpssp, unsigned int val)
{
	//TODO
}


static int
oscilloscope_gui_gtk_expose(OscilloscopeGuiGtk *w, GdkEventExpose *event, gpointer data)
{

	if (0 < event->count) {
		return FALSE;
	}

	gdk_window_clear_area(w->draw_area->window,
			0, 0,
			w->draw_area->allocation.width,
			w->draw_area->allocation.height);

	drawOscillator(w->draw_area);
	return 0;
}

static void
oscilloscope_gui_gtk_class_init(OscilloscopeGuiGtkClass *klass)
{
}

static void
oscilloscope_gui_gtk_init(OscilloscopeGuiGtk *cw)
{
	//prepare table
	gtk_table_resize(GTK_TABLE(cw), ROWS,COLUMNS);
	gtk_table_set_homogeneous(GTK_TABLE(cw), 0 );

	//create widgets
	cw->draw_area = gtk_drawing_area_new();


	//add widgets to table
	gtk_table_attach(GTK_TABLE(cw), cw->draw_area, 0, 1, 0, ROWS, GTK_SHRINK, GTK_SHRINK, 5, 5);
	gtk_widget_set_size_request(cw->draw_area, SCREENWIDTH, SCREENHEIGHT);

	//show widgets
	gtk_widget_show(cw->draw_area);
	g_signal_connect(G_OBJECT(cw), "expose_event", G_CALLBACK(oscilloscope_gui_gtk_expose), NULL);
}

static GType
oscilloscope_gui_gtk_get_type(void)
{
	static GType osc_type = 0;
	if (!osc_type) {
		const GTypeInfo osc_info =
		{
			sizeof(OscilloscopeGuiGtkClass),
			NULL,
			NULL,
			(GClassInitFunc) oscilloscope_gui_gtk_class_init,
			NULL,
			NULL,
			sizeof(OscilloscopeGuiGtk),
			0,
			(GInstanceInitFunc) oscilloscope_gui_gtk_init,
		};
		osc_type = g_type_register_static(GTK_TYPE_TABLE, "OscilloscopeGuiGtk", &osc_info, 0);
	}
	return osc_type;
}

static GtkWidget *
oscilloscope_gui_gtk_new(void)
{
	GType t = oscilloscope_gui_gtk_get_type();
	return GTK_WIDGET (g_object_new (t, NULL));
}

static void
oscilloscope_gui_gtk_pixel_set(
	void *_cpssp,
	unsigned int x,
	unsigned int y,
	uint8_t r,
	uint8_t g,
	uint8_t b)
{
	oscilloscope_gui_gtk_draw_pixel(_cpssp, x, y, r, g, b);
}


void *
oscilloscope_gui_gtk_create(
	unsigned int page,
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_a,
	struct sig_std_logic *port_b,
	struct sig_std_logic *port_c,
	struct sig_opt_rgb *port_monitor,
	struct sig_std_logic *port_d,
	struct sig_std_logic *port_e,
	struct sig_std_logic *port_f
)
{
	static const struct sig_opt_rgb_funcs opt_monitor_in_funcs = {
		.pixel_set = oscilloscope_gui_gtk_pixel_set,
	};
	static const struct sig_std_logic_funcs port_c_func = {
		.std_logic_set = timer_callback,
	};
	static const struct sig_std_logic_funcs port_func = {
		.std_logic_set = reset_callback,
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->widget = oscilloscope_gui_gtk_new();
	cpssp->area = gtk_drawing_area_new();
	gtk_widget_show(cpssp->widget);

	gui_gtk_comp_add(page, "Oscilloscope", name, cpssp->widget, FALSE, FALSE, NULL);

	sig_std_logic_connect_in(port_c, cpssp, &port_c_func);
	sig_std_logic_connect_in(port_b, cpssp, &port_func);
	sig_opt_rgb_connect(port_monitor, cpssp, &opt_monitor_in_funcs);
	return cpssp->widget;
}


void
oscilloscope_gui_gtk_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
oscilloscope_gui_gtk_suspend(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
oscilloscope_gui_gtk_resume(void *_cpssp, FILE *fp)
{
	/* FIXME */
}
#endif
