/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * DMA Multiplexer
 */

#define DEBUG	0

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */
#ifdef STATE

struct {
	unsigned int request_in_state[64];
	unsigned int trigger_in_state[2];
	unsigned int req[4];

	/* DMAMUX Channel Configuration Register (DMAMUXx_CHCFGn) */
	/*
	 * WARNING: Bit numbers in manual wrong!!!
	 */
	uint8_t enbl[4];
	uint8_t trig[2];
	uint8_t source[4];
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void                                       
NAME_(request_inN_set)(struct cpssp *cpssp, unsigned int n, unsigned int val);
/*forward*/ static void                                       
NAME_(trigger_inN_set)(struct cpssp *cpssp, unsigned int n, unsigned int val);
/*forward*/ static void                                       
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void                                       
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void                                       
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static void
NAME_(request_update)(struct cpssp *cpssp)
{
	unsigned int src;
	unsigned int ch;

	for (ch = 0; ch < 4; ch++) {
		src = cpssp->NAME.source[ch];

		if (ch < 2
		 && cpssp->NAME.trig[ch]) {
			cpssp->NAME.req[ch] |= cpssp->NAME.request_in_state[src]
					& cpssp->NAME.trigger_in_state[ch];
			cpssp->NAME.req[ch] &= cpssp->NAME.request_in_state[src];
		} else {
			cpssp->NAME.req[ch] = cpssp->NAME.request_in_state[src];
		}

		NAME_(request_outN_set)(cpssp, ch,
				cpssp->NAME.enbl[ch]
				& cpssp->NAME.req[ch]);
	}
}

static void
NAME_(request_inN_set)(struct cpssp *cpssp, unsigned int src, unsigned int val)
{
#if DEBUG
	fprintf(stderr, "%s: src=%u, val=%u\n", __FUNCTION__, src, val);
#endif
	assert(0 < src && src < 60);
			/* 0 is the always "disabled" channel. */
			/* 60-63 are the always "enabled" channels. */

	cpssp->NAME.request_in_state[src] = val;
	NAME_(request_update)(cpssp);
}

static void
NAME_(trigger_inN_set)(struct cpssp *cpssp, unsigned int ch, unsigned int val)
{
#if DEBUG
	fprintf(stderr, "%s: ch=%u, val=%u\n", __FUNCTION__, ch, val);
#endif
	assert(/* 0 <= ch && */ ch < 2);

	cpssp->NAME.trigger_in_state[ch] = val;
	NAME_(request_update)(cpssp);
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		/* DMAMUX Channel Configuration Register (DMAMUXx_CHCFGn) */
		/*
		 * WARNING: Bit numbers in manual wrong!!!
		 */
		if ((bs >> 0) & 1) {
			*valp |= cpssp->NAME.enbl[0] << (7+0);
			*valp |= cpssp->NAME.trig[0] << (6+0);
			*valp |= cpssp->NAME.source[0] << (0+0);
		}
		if ((bs >> 1) & 1) {
			*valp |= cpssp->NAME.enbl[1] << (7+8);
			*valp |= cpssp->NAME.trig[1] << (6+8);
			*valp |= cpssp->NAME.source[1] << (0+8);
		}
		if ((bs >> 2) & 1) {
			*valp |= cpssp->NAME.enbl[2] << (7+16);
			/* 1: Reserved */
			*valp |= cpssp->NAME.source[2] << (0+16);
		}
		if ((bs >> 3) & 1) {
			*valp |= cpssp->NAME.enbl[3] << (7+24);
			/* 1: Reserved */
			*valp |= cpssp->NAME.source[3] << (0+24);
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		*valp = 0;
		assert(0); /* FIXME */
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x1000 - 1;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		/* DMAMUX Channel Configuration Register (DMAMUXx_CHCFGn) */
		/*
		 * WARNING: Bit numbers in manual wrong!!!
		 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.enbl[0] = (val >> (7+0)) & 1;
			cpssp->NAME.trig[0] = (val >> (6+0)) & 1;
			cpssp->NAME.source[0] = (val >> (0+0)) & 0b111111;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.enbl[1] = (val >> (7+8)) & 1;
			cpssp->NAME.trig[1] = (val >> (6+8)) & 1;
			cpssp->NAME.source[1] = (val >> (0+8)) & 0b111111;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.enbl[2] = (val >> (7+16)) & 1;
			/* 1: Reserved */
			cpssp->NAME.source[2] = (val >> (0+16)) & 0b111111;
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.enbl[3] = (val >> (7+24)) & 1;
			/* 1: Reserved */
			cpssp->NAME.source[3] = (val >> (0+24)) & 0b111111;
		}
		NAME_(request_update)(cpssp);
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int ch;

	for (ch = 0; ch < 4; ch++) {
		cpssp->NAME.req[ch] = 0;
	}

	/* DMAMUX Channel Configuration Register (DMAMUXx_CHCFGn) */
	for (ch = 0; ch < 4; ch++) {
		cpssp->NAME.enbl[ch] = 0;
		if (ch < 2) cpssp->NAME.trig[ch] = 0;
		cpssp->NAME.source[ch] = 0b000000;
	}
	NAME_(request_update)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	unsigned int src;

	for (src = 0; src < 64; src++) {
		cpssp->NAME.request_in_state[src] = 0;
	}
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
