/*
 * Copyright (C) 2005-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef INCLUDE
#if DEVICE_ID == 0x7111
#define PATA_PORTS	2
#define SATA_PORTS	0
#elif DEVICE_ID == 0x27c0
#define PATA_PORTS	1
#define SATA_PORTS	4
#elif DEVICE_ID == 0x3a00
#define PATA_PORTS	0
#define SATA_PORTS	6
#else
#error Unsupported IDE controller!
#endif
#endif

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	/* Ports */
	volatile unsigned int dma_val[2];
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	volatile unsigned int irq_val[2];
#endif

	/* Config Space */
	uint8_t io_enable;
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	uint8_t mem_enable;
#endif
	uint8_t master_enable;
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	uint8_t native_enable;
#endif
#if DEVICE_ID == 0x7111
	uint8_t master_latency_timer;
#endif
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	uint16_t pcmd_bar;
	uint16_t pcnl_bar;
	uint16_t scmd_bar;
	uint16_t scnl_bar;
#endif
	uint32_t bar;
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	uint32_t sidpba_abar;
#endif
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	uint16_t svid;
	uint16_t sid;
#endif
	uint8_t interrupt_line;
	uint16_t idetim0;
	uint8_t ide0;
	uint16_t idetim1;
	uint8_t ide1;
	uint8_t sidetim;
	uint8_t udmactl;

	/* Address Map Register */
	/* ICH7: 12.1.33 */
#if PATA_PORTS && SATA_PORTS
	uint8_t map_value;
#endif
#if SATA_PORTS
	uint8_t sata_mode_select;
#endif

	/* Port Control and Status Register */
	/* 14.1.31 */
#if SATA_PORTS
	uint8_t port_enabled[SATA_PORTS];
	uint8_t port_present[SATA_PORTS];
#endif

#if SATA_PORTS
	/* SIDPBA */
	uint8_t register_index;
	uint8_t port_index;
#endif

#if SATA_PORTS
	/*
	 * ABAR
	 */
	uint8_t devsel[2];

	uint8_t ahci_irq_enable;
	uint8_t ahci_enable;
	uint8_t ahci_irq_pending[6]; /* FIXME */
#if DEVICE_ID == 0x3a00
	uint8_t ahci_coalescing_irq_pending;
#endif
	uint8_t ahci_ports_implemented_locked;
	uint8_t ahci_ports_implemented[6]; /* FIXME */

	/* Port Command List Base Address Register */
	/* 14.4.3.1-14.4.3.2 */
	uint32_t ahci_port_command_list_base[6];
	/* Port FIS Base Address Register */
	/* 14.4.3.3-14.4.3.4 */
	uint32_t ahci_port_fis_base[6];
	/* Port Interrupt Status Register */
	/* 14.4.3.5 */
	uint32_t ahci_port_irq_status[6];
	/* Port Interrupt Enable Register */
	/* 14.4.3.6 */
	uint32_t ahci_port_irq_enable[6];
	/* Port Command Register */
	/* 14.4.3.7 */
	uint8_t ahci_port_start[6];
	uint8_t ahci_port_fis_receive_enable[6];
	/* Port Task File Data Register */
	/* 14.4.3.8 */
	uint8_t ahci_port_status[6];
	uint8_t ahci_port_error[6];
	/* Port Signature Register */
	/* 14.4.3.9 */
	uint32_t ahci_port_signature[6];
	/* Port Serial ATA Status Register */
	/* 14.4.3.10 */
	uint8_t ahci_port_device_detection[6];
	uint8_t ahci_port_current_interface_speed[6];
	uint8_t ahci_port_interface_power_management[6];
	/* Port Serial ATA Control Register */
	/* 14.4.3.11 */
	uint8_t ahci_port_device_detection_initialization[6];
	uint8_t ahci_port_speed_allowed[6];
	uint8_t ahci_port_interface_power_management_transitions_allowed[6];
	/* Port Serial ATA Error Register */
	/* 14.4.3.12 */
	/* FIXME */
	/* Port Serial ATA Active Register */
	/* 14.4.3.13 */
	/* FIXME */
	/* Port Command Issue Register */
	/* 14.4.3.14 */
	/* FIXME */
#endif

	/* Bus Master */
	uint8_t STATE_bmi_start[2];
	uint8_t STATE_bmi_write[2];
	volatile uint8_t bmi_active[2];
	uint8_t bmi_int[2];
	uint8_t bmi_prdis[2];
	volatile uint32_t bmi_addr[2];
#if SATA_PORTS
	uint32_t ahci_index;
#endif

	uint32_t STATE_addr[2];
	uint16_t STATE_count[2];
	uint8_t STATE_eot[2];

#if SATA_PORTS
	/* SATA Ports */
	struct {
		uint8_t error;
		uint8_t features;
		uint8_t nsector;
		uint16_t cyl;
		uint8_t status;
		uint8_t select;
		uint8_t command;
		uint8_t control;

		enum {
			SATA_IDLE,
			SATA_DMA_IN,
			SATA_DMA_OUT,
			SATA_PIO_IN,
			SATA_PIO_OUT,
		} state;

		uint16_t buf[4096];
		unsigned int head;
		unsigned int tail;
		unsigned int count;
	} sata_port[SATA_PORTS];
#endif

	/* Busy IDE */
	unsigned long long time;
	int busy;
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

/* Reset Interface ----------------------------------------------------- */

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int nr;

	/*
	 * Config Space
	 */
	/* PCI Command Register */
	/* 14.1.3 */
	cpssp->NAME.io_enable = 0b0;
#if SATA_PORTS
	cpssp->NAME.mem_enable = 0b0;
#endif
	cpssp->NAME.master_enable = 0b0;

	/* Programming Interface Register */
	/* 4.1.6 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	cpssp->NAME.native_enable = 0b0;
#endif

#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	/* Primary Command Block Base Address Register */
	/* 14.1.11 */
	cpssp->NAME.pcmd_bar = 0x0000;

	/* Primary Control Block Base Address Register */
	/* 14.1.12 */
	cpssp->NAME.pcnl_bar = 0x0000;

	/* Secondary Command Block Base Address Register */
	/* 14.1.13 */
	cpssp->NAME.scmd_bar = 0x0000;

	/* Secondary Control Block Base Address Register */
	/* 14.1.14 */
	cpssp->NAME.scnl_bar = 0x0000;
#endif

	/* Legacy Bus Master Base Address Register */
	/* 14.1.15 */
	cpssp->NAME.bar = 0x00000000;

#if SATA_PORTS
	/* Serial ATA Index Data Pair Base Address Register */
	/* AHCI Base Address Register */
	/* 14.1.16 */
	cpssp->NAME.sidpba_abar = 0x00000000;
#endif

#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	/* Subsystem Vendor Identification Register */
	/* 14.1.17 */
	cpssp->NAME.svid = 0x0000;

	/* Subsystem Identification Register */
	/* 14.1.18 */
	cpssp->NAME.sid = 0x0000;
#endif

	/* Interrupt Line Register */
	/* 14.1.20 */
	cpssp->NAME.interrupt_line = 0x00;

	/* IDE Timing Register */
	/* 14.1.22 */
	cpssp->NAME.idetim0 = 0x8000;	/* FIXME */
	cpssp->NAME.ide0 = 1;	/* FIXME */
	cpssp->NAME.idetim1 = 0x8000;	/* FIXME */
	cpssp->NAME.ide1 = 1;	/* FIXME */

	/* Slave IDE Timing Register */
	/* ? */
	cpssp->NAME.sidetim = 0x00;	/* FIXME */

	/* Ultra DMA Control Register */
	/* ? */
	cpssp->NAME.udmactl = 0x0;

	/* Address Map Register */
	/* ICH7: 12.1.33 */
#if PATA_PORTS && SATA_PORTS
	cpssp->NAME.map_value = 0b00; /* Non-combined mode */
#endif
#if SATA_PORTS
	cpssp->NAME.sata_mode_select = 0b00; /* IDE mode */
#endif

#if SATA_PORTS
	/* Port Control and Status Register */
	/* 14.1.31 */
	for (nr = 0; nr < SATA_PORTS; nr++) {
		cpssp->NAME.port_enabled[nr] = 1; /* FIXME */
		cpssp->NAME.port_present[nr] = (nr == 0) || (nr == 2); /* FIXME */
	}
#endif

#if SATA_PORTS
	/*
	 * ABAR
	 */
	cpssp->NAME.devsel[0] = 0;
	cpssp->NAME.devsel[1] = 0;

	cpssp->NAME.ahci_irq_enable = 0b0;
	cpssp->NAME.ahci_enable = 0b0;
	for (nr = 0; nr < 6; nr++) { /* FIXME */
		cpssp->NAME.ahci_irq_pending[nr] = 0b0;
	}
#if DEVICE_ID == 0x3a00
	cpssp->NAME.ahci_coalescing_irq_pending = 0b0;
#endif
	cpssp->NAME.ahci_ports_implemented_locked = 0b0;
	for (nr = 0; nr < 6; nr++) { /* FIXME */
		cpssp->NAME.ahci_ports_implemented[nr] = 0b1; /* FIXME */
	}
	for (nr = 0; nr < 6; nr++) {
		/* Port Command List Base Address Register */
		/* 14.4.3.1-14.4.3.2 */
		cpssp->NAME.ahci_port_command_list_base[nr] = 0x00000000;
		/* Port FIS Base Address Register */
		/* 14.4.3.3-14.4.3.4 */
		cpssp->NAME.ahci_port_fis_base[nr] = 0x00000000;
		/* Port Interrupt Status Register */
		/* 14.4.3.5 */
		cpssp->NAME.ahci_port_irq_status[nr] = 0x00000000;
		/* Port Interrupt Enable Register */
		/* 14.4.3.6 */
		cpssp->NAME.ahci_port_irq_enable[nr] = 0x00000000;
		/* Port Command Register */
		/* 14.4.3.7 */
		cpssp->NAME.ahci_port_start[nr] = 0b0;
		cpssp->NAME.ahci_port_fis_receive_enable[nr] = 0b0;
		/* Port Task File Data Register */
		/* 14.4.3.8 */
		cpssp->NAME.ahci_port_status[nr] = 0x00;
		cpssp->NAME.ahci_port_error[nr] = 0x00;
		/* Port Signature Register */
		/* 14.4.3.9 */
		cpssp->NAME.ahci_port_signature[nr] = 0xffffffff;
		/* Port Serial ATA Status Register */
		/* 14.4.3.10 */
		cpssp->NAME.ahci_port_device_detection[nr] = 0x0;
		cpssp->NAME.ahci_port_current_interface_speed[nr] = 0x0;
		cpssp->NAME.ahci_port_interface_power_management[nr] = 0x0;
		/* Port Serial ATA Control Register */
		/* 14.4.3.11 */
		cpssp->NAME.ahci_port_device_detection_initialization[nr] = 0x4;
		cpssp->NAME.ahci_port_speed_allowed[nr] = 0x0;
		cpssp->NAME.ahci_port_interface_power_management_transitions_allowed[nr] = 0x0;
		/* FIXME */
		/* Port Serial ATA Error Register */
		/* 14.4.3.12 */
		/* FIXME */
		/* Port Serial ATA Active Register */
		/* 14.4.3.13 */
		/* FIXME */
		/* Port Command Issue Register */
		/* 14.4.3.14 */
		/* FIXME */
	}
#endif

	/*
	 * Bus Master
	 */
	for (nr = 0; nr < 2; nr++) {
		/* Bus Master IDE Command Register */
		/* 14.2.1 */
		STATE_SET(bmi_start[nr], 0b0);
		STATE_SET(bmi_write[nr], 0b0);

		/* Bus Master IDE Status Register */
		/* 14.2.2 */
		cpssp->NAME.bmi_active[nr] = 0b0;
		cpssp->NAME.bmi_int[nr] = 0b0;
		cpssp->NAME.bmi_prdis[nr] = 0b0;
		
		/* Bus Master IDE Descriptor Table Pointer Register */
		/* 14.2.3 */
		cpssp->NAME.bmi_addr[nr] = 0x00000000;
	}
#if SATA_PORTS
	cpssp->NAME.ahci_index = 0x00000000;
#endif
}

/* Config Space Interface ---------------------------------------------- */

static int
NAME_(c0r)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	*valp = 0;
	switch (addr) {
	case 0x00:
		/* Vendor Identification Register */
		/* 14.1.1 */
		*valp |= 0x8086 << 0;

		/* Device Identification Register */
		/* 14.1.2 */
#if DEVICE_ID == 0x27c0
		switch (cpssp->NAME.sata_mode_select) {
		case 0b00: /* IDE */
			*valp |= 0x27c0 << 16;
			break;
		case 0b01: /* AHCI */
			*valp |= 0x27c1 << 16;
			break;
		case 0b10: /* RAID */
			*valp |= 0x27c3 << 16;
			break;
		case 0b11: /* Reserved */
			*valp |= 0x0000 << 16;
			break;
		}
#elif DEVICE_ID == 0x3a00
		switch (cpssp->NAME.sata_mode_select) {
		case 0b00: /* IDE */
			*valp |= 0x3a00 << 16;
			break;
		case 0b01: /* AHCI */
			*valp |= 0x3a02 << 16;
			break;
		case 0b10: /* RAID */
			*valp |= 0x3a05 << 16;
			break;
		case 0b11: /* Reserved */
			*valp |= 0x0000 << 16;
			break;
		}
#else
		*valp |= DEVICE_ID << 16;
#endif
		break;

	case 0x04:
		/* PCI Command Register */
		/* 14.1.3 */
		*valp |= cpssp->NAME.io_enable << 0;
#if SATA_PORTS
		*valp |= cpssp->NAME.mem_enable << 1;
#else
		*valp |= 0b0 << 1; /* Hardwired to '0'. */
#endif
		*valp |= cpssp->NAME.master_enable << 2;
		/* FIXME */

		/* PCI Status Register */
		/* 14.1.4 */
		*valp |= 0b000 << 16; /* Reserved */
		*valp |= 0 << 19; /* Interrupt Status - FIXME */
		*valp |= 0 << 20; /* Capabilities List - FIXME */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		*valp |= 1 << 21; /* 66MHz Capable */
#else
		*valp |= 0 << 21; /* 66MHz Capable */
#endif
		*valp |= 0 << 22; /* Reserved */
		*valp |= 1 << 23; /* Fast Back to Back Capable */
		*valp |= 0 << 24; /* Data Parity Error Detected */
		*valp |= 0b01 << 25; /* DEVSEL# Timing Status */
		*valp |= 0 << 27; /* Signaled Target Abort */
		*valp |= 0 << 28; /* Reserved */
		*valp |= 0 << 29; /* Received Master Abort */
		*valp |= 0 << 30; /* Signaled System Error */
		*valp |= 0 << 31; /* Detected Parity Error */
		break;

	case 0x08:
		/* Revision Identification Register */
		/* 4.1.5 */
		*valp |= REVISION_ID << 0;

		/* Programming Interface Register */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		switch (cpssp->NAME.sata_mode_select) {
		case 0b00: /* IDE */
			/* 4.1.6.1 */
			*valp |= cpssp->NAME.native_enable << (0 + 8); /* Primary Mode Native Enable */
			*valp |= 0b0 << (1 + 8); /* Primary Mode Native Capable */
			*valp |= cpssp->NAME.native_enable << (2 + 8); /* Secondary Mode Native Enable */
			*valp |= 0b0 << (3 + 8); /* Secondary Mode Native Capable */
			*valp |= 0b000 << (4 + 8); /* Reserved */
			*valp |= 0b1 << (7 + 8); /* Bus Master Operations */
			break;
		case 0b01: /* AHCI */
			/* 14.1.6.3 */
			*valp |= 0x01 << (0 + 8);
			break;
		case 0b10: /* RAID */
			/* 14.1.6.2 */
			*valp |= 0x00 << (0 + 8);
			break;
		case 0b11: /* Reserved */
			break;
		}
#else
		*valp |= 0b0 << (0 + 8); /* Primary Mode Native Enable */
		*valp |= 0b0 << (1 + 8); /* Primary Mode Native Capable */
		*valp |= 0b0 << (2 + 8); /* Secondary Mode Native Enable */
		*valp |= 0b0 << (3 + 8); /* Secondary Mode Native Capable */
		*valp |= 0b000 << (4 + 8); /* Reserved */
		*valp |= 0b1 << (7 + 8); /* Bus Master Operations */
#endif

		/* Sub Class Code Register */
		/* 14.1.7 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		switch (cpssp->NAME.sata_mode_select) {
		case 0b00: /* IDE */
			*valp |= 0x01 << (0 + 16); /* IDE */
			break;
		case 0b01: /* AHCI */
			*valp |= 0x06 << (0 + 16); /* AHCI */
			break;
		case 0b10: /* RAID */
			*valp |= 0x04 << (0 + 16); /* RAID */
			break;
		default: /* Reserved */
			*valp |= 0x00 << (0 + 16); /* FIXME */
			break;
		}
#else
		*valp |= 0x01 << (0 + 16); /* IDE */
#endif

		/* Base Class Code Register */
		/* 14.1.8 */
		*valp |= 0x01 << (0 + 24); /* Mass Storage Device */
		break;

	case 0x0c:
		/* Reserved */
		*valp |= 0x00 << 0;

		/* Primary Master Latency Timer Register */
		/* 12.1.9/14.1.9 */
#if DEVICE_ID == 0x7111
		*valp |= cpssp->NAME.master_latency_timer << 8;
#else
		*valp |= 0x00 << 8; /* Hardwired to 0x00. */
#endif

		/* Header Type Register */
		/* -/14.1.10 */
		*valp |= 0b0000000 << (0 + 16); /* Header Layout */
		*valp |= MULTIFUNC << (7 + 16); /* Multi-part? */

		/* Reserved */
		*valp |= 0x00 << (0 + 24);
		break;

	case 0x10:
		/* Primary Command Block Base Address */
		/* 12.1.10/14.1.11 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		*valp |= 0b1 << 0; /* Resource Type Indicator (I/O) */
		*valp |= cpssp->NAME.pcmd_bar << (1-1);
#else
		*valp |= 0x0000 << 0; /* Reserved */
#endif
		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x14:
		/* Primary Control Block Base Address */
		/* 12.1.11/14.1.12 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		*valp |= 0b1 << 0; /* Resource Type Indicator (I/O) */
		*valp |= cpssp->NAME.pcnl_bar << (1-1);
#else
		*valp |= 0x0000 << 0; /* Reserved */
#endif
		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x18:
		/* Secondary Command Block Base Address */
		/* 12.1.12/14.1.13 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		*valp |= 0b1 << 0; /* Resource Type Indicator (I/O) */
		*valp |= cpssp->NAME.scmd_bar << (1-1);
#else
		*valp |= 0x0000 << 0; /* Reserved */
#endif
		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x1c:
		/* Secondary Control Block Base Address */
		/* 12.1.13/14.1.14 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		*valp |= 0b1 << 0; /* Resource Type Indicator (I/O) */
		*valp |= cpssp->NAME.scnl_bar << (1-1);
#else
		*valp |= 0x0000 << 0; /* Reserved */
#endif
		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x20:
		/* Legacy Bus Master Base Address Register */
		/* 12.1.14/14.1.15 */
		*valp |= 0b1 << 0; /* Resource Type Indicator (I/O) */
		*valp |= 0b000 << 1; /* Reserved */
#if SATA_PORTS
		if (cpssp->NAME.sata_mode_select != 0b00) {
			/* Non-IDE */
			*valp |= 0b0 << 4; /* Reserved */
		} else
#endif
		{
			*valp |= cpssp->NAME.bar & (0b1 << 4);
		}
		*valp |= cpssp->NAME.bar & 0xffe0;
		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x24:
		/* Serial ATA Index Data Pair Base Address Register */
		/* AHCI Base Address Register */
		/* 14.1.16 */
#if SATA_PORTS
		if (cpssp->NAME.sata_mode_select == 0b00) {
			/* IDE */
			/* 14.1.16.2 */
			*valp |= 0b1 << 0; /* Resource Type Indicator (I/O) */
			*valp |= cpssp->NAME.sidpba_abar << (1-1);
			*valp |= 0x0000 << 16; /* Reserved */
		} else {
			/* AHCI/RAID */
			/* 14.1.16.1 */
			*valp |= 0b0 << 0; /* Resource Type Indicator (Mem) */
			*valp |= 0b00 << 1; /* 32-bit BAR */
			*valp |= 0b0 << 3; /* non-prefetchable */
			*valp |= 0b0000000 << 4; /* Reserved */
			*valp |= cpssp->NAME.sidpba_abar << (11-11);
		}
#else
		*valp |= 0x00000000 << 0; /* Reserved */
#endif
		break;

	case 0x28:
		/* Reserved */
		*valp |= 0x00000000 << 0;
		break;

	case 0x2c:
		/* Subsystem Vendor Identification Register */
		/* 14.1.17 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		*valp |= cpssp->NAME.svid << 0;
#else
		*valp |= 0x0000 << 0; /* Reserved */
#endif

		/* Subsystem Identification Register */
		/* 14.1.17 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		*valp |= cpssp->NAME.sid << (0 + 16);
#else
		*valp |= 0x0000 << 16; /* Reserved */
#endif
		break;

	case 0x30:
		*valp |= 0x00000000 << 0; /* Reserved */
		break;

	case 0x34:
		/* Capabilities Pointer Register */
		/* 14.1.19 */
#if SATA_PORTS
		if (cpssp->NAME.sata_mode_select == 0b00) {
			/* IDE */
			*valp |= 0x70 << 0;
		} else {
			/* Non-IDE */
			*valp |= 0x80 << 0;
		}
#else
		*valp |= 0x00 << 0; /* Reserved */
#endif

		/* Reserved */
		*valp |= 0x000000 << 8;
		break;

	case 0x38:
		/* Reserved */
		*valp |= 0x00000000 << 0;
		break;

	case 0x3c:
		/* Interrupt Line Register */
		/* 14.1.20 */
		*valp |= cpssp->NAME.interrupt_line << 0;

		/* Interrupt Pin Register */
		/* 14.1.21 */
		*valp |= INTERRUPT_PIN << 8;

		/* Reserved */
		*valp |= 0x0000 << 16;
		break;

	case 0x40:
		/* IDE Timing Register */
		/* 14.1.22 */
		*valp |= cpssp->NAME.idetim0 << 0;
		*valp |= cpssp->NAME.ide0 << 15;

		*valp |= cpssp->NAME.idetim1 << (0 + 16);
		*valp |= cpssp->NAME.ide1 << (15 + 16);
		break;

	case 0x44:
		/* Slave IDE Timing Register */
		/* ? */
		*valp |= cpssp->NAME.sidetim << 0;

		*valp |= 0x000000 << 8; /* Reserved */
		break;

	case 0x48:
		/* Ultra DMA Control Register */
		/* ? */
		*valp |= cpssp->NAME.udmactl << 0;
		*valp |= 0x0000000 << 4; /* Reserved */
		break;

	/* FIXME */

	case 0x90:
#if SATA_PORTS
		/* Address Map Register */
		/* 14.1.30 */
		/* FIXME */
#if PATA_PORTS && SATA_PORTS
		*valp |= cpssp->NAME.map_value << 0;
#else
		*valp |= 0b00 << 0; /* Non-combined mode only. */
#endif
		*valp |= 0b0000 << 2; /* Reserved */
		*valp |= cpssp->NAME.sata_mode_select << 6;

		*valp |= 0x00 << 8; /* Reserved */

		/* Port Control and Status Register */
		/* 14.1.31 */
#if 1 <= SATA_PORTS
		*valp |= cpssp->NAME.port_enabled[0] << (0 + 16);
#else
		*valp |= 0b0 << (0 + 16); /* Reserved */
#endif
#if 2 <= SATA_PORTS
		*valp |= cpssp->NAME.port_enabled[1] << (1 + 16);
#else
		*valp |= 0b0 << (1 + 16); /* Reserved */
#endif
#if 3 <= SATA_PORTS
		*valp |= cpssp->NAME.port_enabled[2] << (2 + 16);
#else
		*valp |= 0b0 << (2 + 16); /* Reserved */
#endif
#if 4 <= SATA_PORTS
		*valp |= cpssp->NAME.port_enabled[3] << (3 + 16);
#else
		*valp |= 0b0 << (3 + 16); /* Reserved */
#endif
#if 5 <= SATA_PORTS
		*valp |= cpssp->NAME.port_enabled[4] << (4 + 16);
#else
		*valp |= 0b0 << (4 + 16); /* Reserved */
#endif
#if 6 <= SATA_PORTS
		*valp |= cpssp->NAME.port_enabled[5] << (5 + 16);
#else
		*valp |= 0b0 << (5 + 16); /* Reserved */
#endif
		*valp |= 0b00 << (6 + 16); /* Reserved */
#if 1 <= SATA_PORTS
		*valp |= cpssp->NAME.port_present[0] << (8 + 16);
#else
		*valp |= 0b0 << (8 + 16);
#endif
#if 2 <= SATA_PORTS
		*valp |= cpssp->NAME.port_present[1] << (9 + 16);
#else
		*valp |= 0b0 << (9 + 16);
#endif
#if 3 <= SATA_PORTS
		*valp |= cpssp->NAME.port_present[2] << (10 + 16);
#else
		*valp |= 0b0 << (10 + 16);
#endif
#if 4 <= SATA_PORTS
		*valp |= cpssp->NAME.port_present[3] << (11 + 16);
#else
		*valp |= 0b0 << (11 + 16);
#endif
#if 5 <= SATA_PORTS
		*valp |= cpssp->NAME.port_present[4] << (12 + 16);
#else
		*valp |= 0b0 << (12 + 16);
#endif
#if 6 <= SATA_PORTS
		*valp |= cpssp->NAME.port_present[5] << (13 + 16);
#else
		*valp |= 0b0 << (13 + 16);
#endif
		*valp |= 0b0 << (14 + 16); /* Reserved */
		*valp |= 0b0 << (15 + 16); /* FIXME */
#else /* SATA_PORTS == 0 */
		*valp |= 0x00000000 << 0; /* Reserved */
#endif
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x.\n",
				__FUNCTION__, (unsigned int) addr);
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=%02x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}

	return 0;
}

static int
NAME_(c0w)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	if (DEBUG) {
		fprintf(stderr, "%s: addr=%02x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* Vendor Identification Register */
		/* 14.1.1 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		/* Device Identification Register */
		/* 14.1.2 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x04:
		/* PCI Command Register */
		/* 14.1.3 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.io_enable = (val >> 0) & 0b1;
#if SATA_PORTS
			cpssp->NAME.mem_enable = (val >> 1) & 0b1;
#else
			/* Bit 1: Hardwired to '0'. */
#endif
			cpssp->NAME.master_enable = (val >> 2) & 0b1;
			/* Bit 3-7: FIXME */
		}
		if ((bs >> 1) & 1) {
			/* FIXME */
		}

		/* PCI Status Register */
		/* 14.1.4 */
		if ((bs >> 2) & 1) {
			/* Bit 0-2: Reserved */
			/* Bit 3-7: FIXME */
		}
		if ((bs >> 3) & 1) {
			/* FIXME */
		}
		break;

	case 0x08:
		/* Revision Identification Register */
		/* 14.1.5 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}

		/* Programming Interface Register */
		/* 14.1.6 */
		if ((bs >> 1) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			cpssp->NAME.native_enable = (val >> (0 + 8)) & 1;
#else
			/* Bit 0: Read-only */
#endif
			/* Bit 1: Read-only */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			/* Bit 2: Must have same value than bit 0. */
#else
			/* Bit 2: Read-only */
#endif
			/* Bit 3: Read-only */
			/* Bit 4-6: Reserved */
			/* Bit 7: Read-only */
		}

		/* Sub Class Code Register */
		/* 14.1.7 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}

		/* Base Class Code Register */
		/* 14.1.8 */
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x0c:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}

		/* Primary Master Latency Timer Register */
		/* 14.1.9 */
		if ((bs >> 1) & 1) {
#if DEVICE_ID == 0x7111
			cpssp->NAME.master_latency_timer = (val >> 8) & 0xf0;
#else
			/* Hardwired to 0x00. */
#endif
		}

		/* Header Type Register */
		/* 14.1.10 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}

		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x10:
		/* Primary Command Block Base Register */
		/* 14.1.11 */
		if ((bs >> 0) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			/* Bit 0: Hardwired to '1'. */
			/* Bit 1-2: Hardwired to '0' */
			cpssp->NAME.pcmd_bar &= ~(0xf8 << 0);
			cpssp->NAME.pcmd_bar |= val & (0xf8 << 0);
#else
			/* Reserved */
#endif
		}
		if ((bs >> 1) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			cpssp->NAME.pcmd_bar &= ~(0xff << 8);
			cpssp->NAME.pcmd_bar |= val & (0xff << 8);
#else
			/* Reserved */
#endif
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x14:
		/* Primary Control Block Base Register */
		/* 14.1.12 */
		if ((bs >> 0) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			/* Bit 0: Hardwired to '1'. */
			/* Bit 1-2: Hardwired to '0' */
			cpssp->NAME.pcnl_bar &= ~(0xf8 << 0);
			cpssp->NAME.pcnl_bar |= val & (0xf8 << 0);
#else
			/* Reserved */
#endif
		}
		if ((bs >> 1) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			cpssp->NAME.pcnl_bar &= ~(0xff << 8);
			cpssp->NAME.pcnl_bar |= val & (0xff << 8);
#else
			/* Reserved */
#endif
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x18:
		/* Secondary Command Block Base Register */
		/* 14.1.13 */
		if ((bs >> 0) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			/* Bit 0: Hardwired to '1'. */
			/* Bit 1-2: Hardwired to '0' */
			cpssp->NAME.scmd_bar &= ~(0xf8 << 0);
			cpssp->NAME.scmd_bar |= val & (0xf8 << 0);
#else
			/* Reserved */
#endif
		}
		if ((bs >> 1) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			cpssp->NAME.scmd_bar &= ~(0xff << 8);
			cpssp->NAME.scmd_bar |= val & (0xff << 8);
#else
			/* Reserved */
#endif
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x1c:
		/* Secondary Control Block Base Register */
		/* 14.1.14 */
		if ((bs >> 0) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			/* Bit 0: Hardwired to '1'. */
			/* Bit 1-2: Hardwired to '0' */
			cpssp->NAME.scnl_bar &= ~(0xf8 << 0);
			cpssp->NAME.scnl_bar |= val & (0xf8 << 0);
#else
			/* Reserved */
#endif
		}
		if ((bs >> 1) & 1) {
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			cpssp->NAME.scnl_bar &= ~(0xff << 8);
			cpssp->NAME.scnl_bar |= val & (0xff << 8);
#else
			/* Reserved */
#endif
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x20:
		/* Legacy Bus Master Base Address Register */
		/* 14.1.15 */
		if ((bs >> 0) & 1) {
			/* Bit 0: Hardwired to '1'. */
			/* Bit 1-3: Hardwired to '0'. */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			if (cpssp->NAME.sata_mode_select != 0b00) {
				/* Non-IDE */
				/* Bit 4: Hardwired to '0'. */
			} else
#endif
			{
				cpssp->NAME.bar &= ~(0x08 << 0);
				cpssp->NAME.bar |= val & (0x08 << 0);
			}
			cpssp->NAME.bar &= ~(0xf0 << 0);
			cpssp->NAME.bar |= val & (0xf0 << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.bar &= ~(0xff << 8);
			cpssp->NAME.bar |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x24:
		/* Serial ATA Index Data Pair Base Address Register */
		/* AHCI Base Address Register */
		/* 14.1.16 */
#if SATA_PORTS
		if (cpssp->NAME.sata_mode_select == 0b00) {
			/* IDE */
			/* 14.1.16.2 */
			if ((bs >> 0) & 1) {
				/* Bit 0: Hardwired to '1'. */
				/* Bit 1-3: Hardwired to '0'. */
				cpssp->NAME.sidpba_abar &= ~(0xf0 << 0);
				cpssp->NAME.sidpba_abar |= val & (0xf0 << 0);
			}
			if ((bs >> 1) & 1) {
				cpssp->NAME.sidpba_abar &= ~(0xff << 8);
				cpssp->NAME.sidpba_abar |= val & (0xff << 8);
			}
			if ((bs >> 2) & 1) {
				/* Reserved */
			}
			if ((bs >> 3) & 1) {
				/* Reserved */
			}
		} else {
			/* AHCI/RAID */
			/* 14.1.16.1 */
			if ((bs >> 0) & 1) {
				/* Hardwired to '0'. */
			}
			if ((bs >> 1) & 1) {
				/* Bit 8-9: Hardwired to '0'. Correct? FIXME */
				cpssp->NAME.sidpba_abar &= ~(0xfc << 8);
				cpssp->NAME.sidpba_abar |= val & (0xfc << 8);
			}
			if ((bs >> 2) & 1) {
				cpssp->NAME.sidpba_abar &= ~(0xff << 16);
				cpssp->NAME.sidpba_abar |= val & (0xff << 16);
			}
			if ((bs >> 3) & 1) {
				cpssp->NAME.sidpba_abar &= ~(0xff << 24);
				cpssp->NAME.sidpba_abar |= val & (0xff << 24);
			}
		}
#else
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
#endif
		break;

	case 0x28:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x2c:
		/* Subsystem Vendor Identification Register */
		/* 14.1.17 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		if (cpssp->NAME.svid == 0x0000) {
			if ((bs >> 0) & 1) {
				cpssp->NAME.svid &= ~(0xff << 0);
				cpssp->NAME.svid |= val & (0xff << 0);
			}
			if ((bs >> 1) & 1) {
				cpssp->NAME.svid &= ~(0xff << 8);
				cpssp->NAME.svid |= val & (0xff << 8);
			}
		} else
#endif
		{
			/* Read-only/Read-once */
		}

		/* Subsystem Identification Register */
		/* 14.1.18 */
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
		if (cpssp->NAME.sid == 0x0000) {
			if ((bs >> 2) & 1) {
				cpssp->NAME.sid &= ~(0xff << 0);
				cpssp->NAME.sid |= (val >> 16) & (0xff << 0);
			}
			if ((bs >> 3) & 1) {
				cpssp->NAME.sid &= ~(0xff << 8);
				cpssp->NAME.sid |= (val >> 16) & (0xff << 8);
			}
		} else
#endif
		{
			/* Read-only/Read-once */
		}
		break;

	case 0x30:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x34:
		/* Capabilities Pointer Register */
		/* 14.1.19 */
		if ((bs >> 0) & 1) {
			/* Reserved/Read-only */
		}

		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x38:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x3c:
		/* Interrupt Line Register */
		/* 14.1.20 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.interrupt_line = (val >> 0) & 0xff;
		}

		/* Interrupt Pin Register */
		/* 14.1.21 */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x40:
		/* IDE Timing Register */
		/* 14.1.22 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.idetim0 &= ~(0xff << 0);
			cpssp->NAME.idetim0 |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.idetim0 &= ~(0x7f << 8);
			cpssp->NAME.idetim0 |= val & (0x7f << 8);
			cpssp->NAME.ide0 = (val >> 15) & 1;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.idetim1 &= ~(0xff << 0);
			cpssp->NAME.idetim1 |= (val >> 16) & (0xff << 0);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.idetim1 &= ~(0x7f << 8);
			cpssp->NAME.idetim1 |= (val >> 16) & (0x7f << 8);
			cpssp->NAME.ide1 = (val >> (15 + 16)) & 1;
		}
		break;

	case 0x44:
		/* Slave IDE Timing Register */
		/* ? */
		if ((bs >> 0) & 1) {
			cpssp->NAME.sidetim = (val >> 0) & 0xff;
		}

		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x48:
		/* Ultra DMA Control Register */
		/* ? */
		if ((bs >> 0) & 1) {
			cpssp->NAME.udmactl = (val >> 0) & 0x0f;
			/* Bit 4-7: Reserved */
		}

		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	/* FIXME */

	case 0x90:
#if SATA_PORTS
		/* Address Map Register */
		/* 14.1.30 */
		if ((bs >> 0) & 1) {
#if PATA_PORTS && SATA_PORTS
			cpssp->NAME.map_value = (val >> 0) & 0b11;
			fprintf(stderr, "WARNING: %s: addr=0x%03x.\n",
					__FUNCTION__, (unsigned int) addr);
#else
			/* Bit 0-1: Reserved */
#endif
			/* Bit 2-5: Reserved */
			cpssp->NAME.sata_mode_select = (val >> 6) & 0b11;
		}

		if ((bs >> 1) & 1) {
			/* Reserved */
		}

		/* Port Control and Status Register */
		/* 14.1.31 */
		if ((bs >> 2) & 1) {
#if 1 <= SATA_PORTS
			cpssp->NAME.port_enabled[0] = (val >> (0 + 16)) & 1;
#else
			/* Bit 0: Reserved */
#endif
#if 2 <= SATA_PORTS
			cpssp->NAME.port_enabled[1] = (val >> (1 + 16)) & 1;
#else
			/* Bit 1: Reserved */
#endif
#if 3 <= SATA_PORTS
			cpssp->NAME.port_enabled[2] = (val >> (2 + 16)) & 1;
#else
			/* Bit 2: Reserved */
#endif
#if 4 <= SATA_PORTS
			cpssp->NAME.port_enabled[3] = (val >> (3 + 16)) & 1;
#else
			/* Bit 3: Reserved */
#endif
#if 5 <= SATA_PORTS
			cpssp->NAME.port_enabled[4] = (val >> (4 + 16)) & 1;
#else
			/* Bit 4: Reserved */
#endif
#if 6 <= SATA_PORTS
			cpssp->NAME.port_enabled[5] = (val >> (5 + 16)) & 1;
#else
			/* Bit 5: Reserved */
#endif
			/* Bit 6-7: Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Bit 8-13: Read-only */
			/* Bit 14: Reserved */
			/* Bit 15: FIXME */
		}
#else /* ! SATA_PORTS */
		/* Reserved */
#endif
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x.\n",
				__FUNCTION__, (unsigned int) addr);
		break;
	}

	return 0;
}

#if SATA_PORTS
/* Emulator Interface -------------------------------------------------- */

/*forward*/ static void
NAME_(pata_dmarq_set)(struct cpssp *cpssp, unsigned int nr, unsigned int val);
/*forward*/ static void
NAME_(pata_irqrq_in_set)(struct cpssp *cpssp, unsigned int nr, unsigned int val);

static void
NAME_(emul_port_read)(
	struct cpssp *cpssp,
	unsigned int nr,
	unsigned int port,
	uint16_t *valp
)
{
	/* FIXME */
	NAME_(sata_inw)(cpssp, (nr << 1) | cpssp->NAME.devsel[nr], port, valp);
	if (port == 6) {
		*valp |= cpssp->NAME.devsel[nr] << 4;
	}
}

static void
NAME_(emul_port_write)(
	struct cpssp *cpssp,
	unsigned int nr,
	unsigned int port,
	uint16_t val
)
{
	/* FIXME */
	if (port == 6) {
		cpssp->NAME.devsel[nr] = (val >> 4) & 0b1;
		val &= ~(1 << 4);
	}
	if (port == 0
	 || port == 7) {
		NAME_(sata_outw)(cpssp, (nr << 1) | cpssp->NAME.devsel[nr],
				port, val);
	} else {
		NAME_(sata_outw)(cpssp, (nr << 1) | 0, port, val);
		NAME_(sata_outw)(cpssp, (nr << 1) | 1, port, val);
	}
}

static void
NAME_(emul_dma_read)(struct cpssp *cpssp, unsigned int nr, uint16_t *valp)
{
	/* FIXME */
	NAME_(sata_dmainw)(cpssp, (nr << 1) | cpssp->NAME.devsel[nr], valp);
}

static void
NAME_(emul_dma_read_end)(struct cpssp *cpssp, unsigned int nr)
{
	/* FIXME */
}

static void
NAME_(emul_dma_write)(struct cpssp *cpssp, unsigned int nr, uint16_t val)
{
	/* FIXME */
	NAME_(sata_dmaoutw)(cpssp, (nr << 1) | cpssp->NAME.devsel[nr], val);
}

static void
NAME_(emul_dma_write_end)(struct cpssp *cpssp, unsigned int nr)
{
	/* FIXME */
}

static void
NAME_(sata_dmarq_set)(struct cpssp *cpssp, unsigned int nr, unsigned int val)
{
	/* FIXME */
	NAME_(pata_dmarq_set)(cpssp, nr / 2, val);
}

static void
NAME_(sata_irqrq_in_set)(struct cpssp *cpssp, unsigned int nr, unsigned int val)
{
	/* FIXME */
	NAME_(pata_irqrq_in_set)(cpssp, nr / 2, val);
}

static void
NAME_(d2h_register)(
	struct cpssp *cpssp,
	unsigned int nr,
	int i,
	uint8_t status,
	uint8_t error,
	uint16_t sector_number,
	uint32_t cyl,
	uint8_t dev_head,
	uint16_t sector_count
)
{
}

static void
NAME_(d2h_set_device_bits)(
	struct cpssp *cpssp,
	unsigned int nr,
	uint8_t error,
	uint8_t status
)
{
}

static void
NAME_(d2h_dma_activate)(struct cpssp *cpssp, unsigned int nr)
{
}

static void
NAME_(d2h_pio_setup)(
	struct cpssp *cpssp,
	unsigned int nr,
	int d,
	int i,
	uint8_t status,
	uint8_t error,
	uint16_t sector_number,
	uint32_t cyl,
	uint8_t dev_head,
	uint16_t sector_count,
	uint8_t e_status,
	uint16_t transfer_count
)
{
}

static void
NAME_(d2h_data)(
	struct cpssp *cpssp,
	unsigned int nr,
	uint16_t *buf,
	unsigned int bufcnt
)
{
}
#endif /* SATA_PORTS */

#if SATA_PORTS
/* AHCI Interface ------------------------------------------------------ */

static void
NAME_(ahci_soft_reset)(struct cpssp *cpssp)
{
	if (DEBUG) {
		fprintf(stderr, "Resetting...\n");
	}

	/* FIXME */
}

static void
NAME_(ahci_start_update)(struct cpssp *cpssp, unsigned int port)
{
	if (DEBUG) {
		fprintf(stderr, "Starting port %u...\n", port);
	}

	/* FIXME */
}

static void
NAME_(ahci_port_reset)(struct cpssp *cpssp, unsigned int port)
{
	if (DEBUG) {
		fprintf(stderr, "Resetting port %u...\n", port);
	}

	/* FIXME */
	if (port == 0
	 || port == 2) {
		// xxx(cpssp, 0x90); /* Execute Device Diagnostics */

		cpssp->NAME.ahci_port_device_detection[port] = 0x3;
		cpssp->NAME.ahci_port_current_interface_speed[port] = 0x2;
		cpssp->NAME.ahci_port_interface_power_management[port] = 0x1;
	} else {
		cpssp->NAME.ahci_port_device_detection[port] = 0x0;
		cpssp->NAME.ahci_port_current_interface_speed[port] = 0x0;
		cpssp->NAME.ahci_port_interface_power_management[port] = 0x0;
	}
}

static int
NAME_(ahci_host_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	addr &= 0xff;
	*valp = 0;
	switch (addr) {
	case 0x00:
		/* Host Capabilities Register */
		/* 12.3.1.1/14.4.1.1 */
		*valp |= (SATA_PORTS - 1) << 0; /* Number of Ports */
		*valp |= 0b000 << 5; /* Reserved */
		*valp |= (32 - 1) << 8; /* Number of Command Slots */
		/* FIXME */
		*valp |= 0x2 << 20; /* 3.0 Gb/s */
		/* FIXME */
		break;

	case 0x04:
		/* Global Control Register */
		/* 12.3.1.2/14.4.1.2 */
		*valp |= 0b0 << 0; /* AHCI Reset - FIXME */
		*valp |= cpssp->NAME.ahci_irq_enable << 1;
#if DEVICE_ID == 0x27c0
		*valp |= 0b0 << 2; /* Reserved */
#elif DEVICE_ID == 0x3a00
		/* Bit 2: FIXME */
#endif
		*valp |= 0x0000000 << 3; /* Reserved */
		*valp |= cpssp->NAME.ahci_enable << 31;
		break;

	case 0x08:
		/* Interrupt Status Register */
		/* 12.3.1.3/14.4.1.3 */
#if 1 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_irq_pending[0] << 0;
#else
		*valp |= 0b0 << 0;
#endif
#if 2 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_irq_pending[1] << 1;
#else
		*valp |= 0b0 << 1;
#endif
#if 3 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_irq_pending[2] << 2;
#else
		*valp |= 0b0 << 2;
#endif
#if 4 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_irq_pending[3] << 3;
#else
		*valp |= 0b0 << 3;
#endif
#if 5 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_irq_pending[4] << 4;
#else
		*valp |= 0b0 << 4;
#endif
#if 6 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_irq_pending[5] << 5;
#else
		*valp |= 0b0 << 5;
#endif
#if DEVICE_ID == 0x27c0
		*valp |= 0b0 << 6; /* Reserved */
#elif DEVICE_ID == 0x3a00
		*valp |= cpssp->NAME.ahci_coalescing_irq_pending << 6;
#endif
		/* Bit 7-31: Reserved */
		*valp |= 0x000000 << 7; /* Reserved */
		break;

	case 0x0c:
		/* Ports Implemented Register */
		/* 12.3.1.4/14.4.1.4 */
#if 1 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_ports_implemented[0] << 0;
#else
		*valp |= 0b0 << 0;
#endif
#if 2 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_ports_implemented[1] << 1;
#else
		*valp |= 0b0 << 1;
#endif
#if 3 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_ports_implemented[2] << 2;
#else
		*valp |= 0b0 << 2;
#endif
#if 4 <= SATA_PORTS
		*valp |= cpssp->NAME.ahci_ports_implemented[3] << 3;
#else
		*valp |= 0b0 << 3;
#endif
		if (cpssp->NAME.sata_mode_select != 0b00) {
#if 5 <= SATA_PORTS
			*valp |= cpssp->NAME.ahci_ports_implemented[4] << 4;
#else
			*valp |= 0b0 << 4;
#endif
#if 5 <= SATA_PORTS
			*valp |= cpssp->NAME.ahci_ports_implemented[5] << 5;
#else
			*valp |= 0b0 << 5;
#endif
		} else {
			*valp |= 0b00 << 4; /* Reserved */
		}
		*valp |= 0x000000 << 6; /* Reserved */
		break;

	case 0x10:
		/* AHCI Version Register */
		/* 12.3.1.5/14.4.1.5 */
#if DEVICE_ID == 0x27c0
		*valp |= 0x00010100 << 0; /* Version 1.10 */
#elif DEVICE_ID == 0x3a00
		*valp |= 0x00010200 << 0; /* Version 1.20 */
#endif
		break;

	case 0x14:
	case 0x18:
	case 0x1c:
	case 0x20:
	case 0xa0:
		/* FIXME */
		break;

	default:
		/* Reserved */
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}

	return 0;
}

static int
NAME_(ahci_host_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	addr &= 0xff;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* Host Capabilities Register */
		/* 12.3.1.1/14.4.1.1 */
		/* FIXME */
		break;

	case 0x04:
		/* Global Control Register */
		/* 12.3.1.2/14.4.1.2 */
		if ((bs >> 0) & 1) {
			if ((val >> 0) & 0b1) {
				NAME_(ahci_soft_reset)(cpssp);
			}
			cpssp->NAME.ahci_irq_enable = (val >> 1) & 0b1;
#if DEVICE_ID == 0x27c0
			/* Bit 2: Reserved */
#elif DEVICE_ID == 0x3a00
			/* Bit 2: FIXME */
#endif
			/* Bit 3-7: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Bit 24-30: Reserved */
			cpssp->NAME.ahci_enable = (val >> 31) & 0b1;
		}
		break;

	case 0x08:
		/* Interrupt Status Register */
		/* 12.3.1.3/14.4.1.3 */
		if ((bs >> 0) & 1) {
#if 1 <= SATA_PORTS
			cpssp->NAME.ahci_irq_pending[0] &= ~((val >> 0) & 0b1);
#else
			/* Bit 0: Reserved */
#endif
#if 2 <= SATA_PORTS
			cpssp->NAME.ahci_irq_pending[1] &= ~((val >> 1) & 0b1);
#else
			/* Bit 1: Reserved */
#endif
#if 3 <= SATA_PORTS
			cpssp->NAME.ahci_irq_pending[2] &= ~((val >> 2) & 0b1);
#else
			/* Bit 2: Reserved */
#endif
#if 4 <= SATA_PORTS
			cpssp->NAME.ahci_irq_pending[3] &= ~((val >> 3) & 0b1);
#else
			/* Bit 3: Reserved */
#endif
#if 5 <= SATA_PORTS
			cpssp->NAME.ahci_irq_pending[4] &= ~((val >> 4) & 0b1);
#else
			/* Bit 4: Reserved */
#endif
#if 6 <= SATA_PORTS
			cpssp->NAME.ahci_irq_pending[5] &= ~((val >> 5) & 0b1);
#else
			/* Bit 5: Reserved */
#endif
#if DEVICE_ID == 0x27c0
			/* Bit 6: Reserved */
#elif DEVICE_ID == 0x3a00
			cpssp->NAME.ahci_coalescing_irq_pending &= ~((val >> 6) & 0b1);
#endif
			/* Bit 7: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x0c:
		/* Ports Implemented Register */
		/* 12.3.1.4/14.4.1.4 */
		if ((bs >> 0) & 1) {
			if (! cpssp->NAME.ahci_ports_implemented_locked) {
#if 1 <= SATA_PORTS
				cpssp->NAME.ahci_ports_implemented[0] = (val >> 0) & 0b1;
#else
				/* Bit 0: Reserved */
#endif
#if 2 <= SATA_PORTS
				cpssp->NAME.ahci_ports_implemented[1] = (val >> 1) & 0b1;
#else
				/* Bit 1: Reserved */
#endif
#if 3 <= SATA_PORTS
				cpssp->NAME.ahci_ports_implemented[2] = (val >> 2) & 0b1;
#else
				/* Bit 2: Reserved */
#endif
#if 4 <= SATA_PORTS
				cpssp->NAME.ahci_ports_implemented[3] = (val >> 3) & 0b1;
#else
				/* Bit 3: Reserved */
#endif
#if 5 <= SATA_PORTS
				cpssp->NAME.ahci_ports_implemented[4] = (val >> 4) & 0b1;
#else
				/* Bit 4: Reserved */
#endif
#if 6 <= SATA_PORTS
				cpssp->NAME.ahci_ports_implemented[5] = (val >> 5) & 0b1;
#else
				/* Bit 5: Reserved */
#endif
				/* Bit 6-7: Reserved */
			}
			cpssp->NAME.ahci_ports_implemented_locked = 1;
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x10:
		/* AHCI Version Register */
		/* 12.3.1.5/14.4.1.5 */
		/* Read-only */
		break;

	case 0x14:
	case 0x18:
	case 0x1c:
	case 0x20:
	case 0xa0:
		/* FIXME */
		break;

	default:
		/* Reserved */
		break;
	}
	return 0;
}

static int
NAME_(ahci_port_r)(
	struct cpssp *cpssp,
	unsigned int port,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp)
{
	addr &= 0x7f;
	*valp = 0;
	switch (addr) {
	case 0x00:
		/* Command List Base Address Register */
		/* 14.4.3.1 */
		*valp |= cpssp->NAME.ahci_port_command_list_base[port] << 0;
		break;

	case 0x04:
		/* Command List Base Address Upper 32-Bits Register */
		/* 14.4.3.2 */
		*valp |= 0x00000000 << 0; /* Reserved */
		break;

	case 0x08:
		/* FIS Base Address Register */
		/* 14.4.3.3 */
		*valp |= cpssp->NAME.ahci_port_fis_base[port] << 0;
		break;

	case 0x0c:
		/* FIS Base Address Upper 32-Bits Register */
		/* 14.4.3.4 */
		*valp |= 0x00000000 << 0; /* Reserved */
		break;

	case 0x10:
		/* Interrupt Status Register */
		/* 14.4.3.5 */
		/* FIXME */
		break;

	case 0x14:
		/* Interrupt Enable Register */
		/* 14.4.3.6 */
		*valp |= cpssp->NAME.ahci_port_irq_enable[port] << 0;
		/* FIXME */
		break;

	case 0x18:
		/* Command Register */
		/* 14.4.3.7 */
		/* FIXME */
		*valp |= cpssp->NAME.ahci_port_start[port] << 0;
		/* FIXME */
		*valp |= cpssp->NAME.ahci_port_fis_receive_enable[port] << 4;
		/* FIXME */
		break;

	case 0x1c:
		/* Reserved */
		break;

	case 0x20:
		/* Task File Data Register */
		/* 14.4.3.8 */
		/* FIXME */
		break;

	case 0x24:
		/* Signature Register */
		/* 14.4.3.9 */
		/* FIXME */
		break;

	case 0x28:
		/* Serial ATA Status Register */
		/* 14.4.3.10 */
		*valp |= cpssp->NAME.ahci_port_device_detection[port] << 0;
		*valp |= cpssp->NAME.ahci_port_current_interface_speed[port] << 4;
		*valp |= cpssp->NAME.ahci_port_interface_power_management[port] << 8;
		/* Bit 12-31: Reserved */
		break;

	case 0x2c:
		/* Serial ATA Control Register */
		/* 14.4.3.11 */
		*valp |= cpssp->NAME.ahci_port_device_detection_initialization[port] << 0;
		*valp |= cpssp->NAME.ahci_port_speed_allowed[port] << 4;
		*valp |= cpssp->NAME.ahci_port_interface_power_management_transitions_allowed[port] << 8;
		/* Bit 12-15: FIXME */
		/* Bit 16-19: FIXME */
		/* Bit 20-31: Reserved */
		break;

	case 0x30:
		/* Serial ATA Error Register */
		/* 14.4.3.12 */
		/* FIXME */
		break;

	case 0x34:
		/* Serial ATA Active Register */
		/* 14.4.3.13 */
		/* FIXME */
		break;

	case 0x38:
		/* Command Issue Register */
		/* 14.4.3.14 */
		/* FIXME */
		break;

	default:
		/* Reserved */
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: port=%u, addr=0x%x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, port, addr, bs, *valp);
	}

	return 0;
}

static int
NAME_(ahci_port_w)(
	struct cpssp *cpssp,
	unsigned int port,
	uint32_t addr,
	unsigned int bs,
	uint32_t val)
{
	addr &= 0x7f;

	if (DEBUG) {
		fprintf(stderr, "%s: port=%u, addr=0x%x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, port, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* Command List Base Address Register */
		/* 14.4.3.1 */
		if ((bs >> 0) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 1) & 1) {
			/* Bit 8-9: Hardwired to '0'. */
			cpssp->NAME.ahci_port_command_list_base[port] &= ~(0xfc << 8);
			cpssp->NAME.ahci_port_command_list_base[port] |= val & (0xfc << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.ahci_port_command_list_base[port] &= ~(0xff << 16);
			cpssp->NAME.ahci_port_command_list_base[port] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.ahci_port_command_list_base[port] &= ~(0xff << 24);
			cpssp->NAME.ahci_port_command_list_base[port] |= val & (0xff << 24);
		}
		break;

	case 0x04:
		/* Command List Base Address Upper 32-Bits Register */
		/* 14.4.3.2 */
		/* Hardwired to '0'. */
		break;

	case 0x08:
		/* FIS Base Address Register */
		/* 14.4.3.3 */
		if ((bs >> 0) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.ahci_port_fis_base[port] &= ~(0xff << 8);
			cpssp->NAME.ahci_port_fis_base[port] |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.ahci_port_fis_base[port] &= ~(0xff << 16);
			cpssp->NAME.ahci_port_fis_base[port] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.ahci_port_fis_base[port] &= ~(0xff << 24);
			cpssp->NAME.ahci_port_fis_base[port] |= val & (0xff << 24);
		}
		break;

	case 0x0c:
		/* FIS Base Address Upper 32-Bits Register */
		/* 14.4.3.4 */
		/* Hardwired to '0'. */
		break;

	case 0x10:
		/* Interrupt Status Register */
		/* 14.4.3.5 */
		/* FIXME */
		break;
	case 0x14:
		/* Interrupt Enable Register */
		/* 14.4.3.6 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.ahci_port_irq_enable[port] &= ~(0xff << 0);
			cpssp->NAME.ahci_port_irq_enable[port] |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.ahci_port_irq_enable[port] &= ~(0xff << 8);
			cpssp->NAME.ahci_port_irq_enable[port] |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.ahci_port_irq_enable[port] &= ~(0xff << 16);
			cpssp->NAME.ahci_port_irq_enable[port] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.ahci_port_irq_enable[port] &= ~(0xff << 24);
			cpssp->NAME.ahci_port_irq_enable[port] |= val & (0xff << 24);
		}
		break;
	case 0x18: {
		/* Command Register */
		/* 14.4.3.7 */
		uint8_t start;

		start = cpssp->NAME.ahci_port_start[port];
		if ((bs >> 0) & 1) {
			cpssp->NAME.ahci_port_start[port] = (val >> 0) & 0b1;
			/* FIXME */
			cpssp->NAME.ahci_port_fis_receive_enable[port] = (val >> 4) & 0b1;
			/* FIXME */
		}
		if ((bs >> 1) & 1) {
			/* FIXME */
		}
		if ((bs >> 2) & 1) {
			/* FIXME */
		}
		if ((bs >> 3) & 1) {
			/* FIXME */
		}
		if (cpssp->NAME.ahci_port_start[port] != start
		 && cpssp->NAME.ahci_port_start[port] == 1) {
			NAME_(ahci_start_update)(cpssp, port);
		}
		break;
	    }
	case 0x1c:
		/* Reserved */
		break;

	case 0x20:
		/* Task File Data Register */
		/* 14.4.3.8 */
		/* FIXME */
		break;

	case 0x24:
		/* Signature Register */
		/* 14.4.3.9 */
		/* FIXME */
		break;

	case 0x28:
		/* Serial ATA Status Register */
		/* 14.4.3.10 */
		/* Bit 0-11: Read-only */
		/* Bit 12-31: Reserved */
		break;

	case 0x2c: {
		/* Serial ATA Control Register */
		/* 14.4.3.11 */
		uint8_t det;

		det = cpssp->NAME.ahci_port_device_detection_initialization[port];
		if ((bs >> 0) & 1) {
			cpssp->NAME.ahci_port_device_detection_initialization[port] = (val >> 0) & 0xf;
			cpssp->NAME.ahci_port_speed_allowed[port] = (val >> 4) & 0xf;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.ahci_port_interface_power_management_transitions_allowed[port] = (val >> 8) & 0xf;
			/* Bit 12-15: FIXME */
		}
		if ((bs >> 2) & 1) {
			/* Bit 16-19: FIXME */
			/* Bit 20-23: Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		if (cpssp->NAME.ahci_port_device_detection_initialization[port] != det
		 && cpssp->NAME.ahci_port_device_detection_initialization[port] == 0x1) {
			NAME_(ahci_port_reset)(cpssp, port);
		}
		break;
	    }
	case 0x30:
		/* Serial ATA Error Register */
		/* 14.4.3.12 */
		/* FIXME */
		break;

	case 0x34:
		/* Serial ATA Active Register */
		/* 14.4.3.13 */
		/* FIXME */
		break;

	case 0x38:
		/* Command Issue Register */
		/* 14.4.3.14 */
		/* FIXME */
		break;

	default:
		/* Reserved */
		break;
	}

	return 0;
}

#if 1 <= SATA_PORTS
static int
NAME_(ahci_p0_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	return NAME_(ahci_port_r)(_cpssp, 0, addr, bs, valp);
}

static int
NAME_(ahci_p0_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	return NAME_(ahci_port_w)(_cpssp, 0, addr, bs, val);
}
#endif

#if 2 <= SATA_PORTS
static int
NAME_(ahci_p1_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	return NAME_(ahci_port_r)(_cpssp, 1, addr, bs, valp);
}

static int
NAME_(ahci_p1_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	return NAME_(ahci_port_w)(_cpssp, 1, addr, bs, val);
}
#endif

#if 3 <= SATA_PORTS
static int
NAME_(ahci_p2_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	return NAME_(ahci_port_r)(_cpssp, 2, addr, bs, valp);
}

static int
NAME_(ahci_p2_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	return NAME_(ahci_port_w)(_cpssp, 2, addr, bs, val);
}
#endif

#if 4 <= SATA_PORTS
static int
NAME_(ahci_p3_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	return NAME_(ahci_port_r)(_cpssp, 3, addr, bs, valp);
}

static int
NAME_(ahci_p3_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	return NAME_(ahci_port_w)(_cpssp, 3, addr, bs, val);
}
#endif

#if 5 <= SATA_PORTS
static int
NAME_(ahci_p4_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	return NAME_(ahci_port_r)(_cpssp, 4, addr, bs, valp);
}

static int
NAME_(ahci_p4_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	return NAME_(ahci_port_w)(_cpssp, 4, addr, bs, val);
}
#endif

#if 6 <= SATA_PORTS
static int
NAME_(ahci_p5_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	return NAME_(ahci_port_r)(_cpssp, 5, addr, bs, valp);
}

static int
NAME_(ahci_p5_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	return NAME_(ahci_port_w)(_cpssp, 5, addr, bs, val);
}
#endif

static enum {
	NAME_(AHCI_HOST),
#if 1 <= SATA_PORTS
	NAME_(AHCI_PORT0),
#endif
#if 2 <= SATA_PORTS
	NAME_(AHCI_PORT1),
#endif
#if 3 <= SATA_PORTS
	NAME_(AHCI_PORT2),
#endif
#if 4 <= SATA_PORTS
	NAME_(AHCI_PORT3),
#endif
#if 5 <= SATA_PORTS
	NAME_(AHCI_PORT4),
#endif
#if 6 <= SATA_PORTS
	NAME_(AHCI_PORT5),
#endif
	NAME_(AHCI_NONE),
} NAME_(ahci_region)(void *_cpssp, uint32_t addr)
{
	addr &= 1024 - 1;
	switch (addr >> 7) {
	case 0:
	case 1:
		return NAME_(AHCI_HOST);
#if 1 <= SATA_PORTS
	case 2:
		return NAME_(AHCI_PORT0);
#endif
#if 2 <= SATA_PORTS
	case 3:
		return NAME_(AHCI_PORT1);
#endif
#if 3 <= SATA_PORTS
	case 4:
		return NAME_(AHCI_PORT2);
#endif
#if 4 <= SATA_PORTS
	case 5:
		return NAME_(AHCI_PORT3);
#endif
#if 5 <= SATA_PORTS
	case 6:
		return NAME_(AHCI_PORT4);
#endif
#if 6 <= SATA_PORTS
	case 7:
		return NAME_(AHCI_PORT5);
#endif
	default:
		return NAME_(AHCI_NONE);
	}
}

static int
NAME_(ahci_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	switch (NAME_(ahci_region)(cpssp, addr)) {
	case NAME_(AHCI_HOST):
		ret = NAME_(ahci_host_r)(cpssp, addr, bs, valp);
		break;
#if 1 <= SATA_PORTS
	case NAME_(AHCI_PORT0):
		ret = NAME_(ahci_p0_r)(cpssp, addr, bs, valp);
		break;
#endif
#if 2 <= SATA_PORTS
	case NAME_(AHCI_PORT1):
		ret = NAME_(ahci_p1_r)(cpssp, addr, bs, valp);
		break;
#endif
#if 3 <= SATA_PORTS
	case NAME_(AHCI_PORT2):
		ret = NAME_(ahci_p2_r)(cpssp, addr, bs, valp);
		break;
#endif
#if 4 <= SATA_PORTS
	case NAME_(AHCI_PORT3):
		ret = NAME_(ahci_p3_r)(cpssp, addr, bs, valp);
		break;
#endif
#if 5 <= SATA_PORTS
	case NAME_(AHCI_PORT4):
		ret = NAME_(ahci_p4_r)(cpssp, addr, bs, valp);
		break;
#endif
#if 6 <= SATA_PORTS
	case NAME_(AHCI_PORT5):
		ret = NAME_(ahci_p5_r)(cpssp, addr, bs, valp);
		break;
#endif
	case NAME_(AHCI_NONE):
	default:
		ret = 1;
		break;
	}

	return ret;
}

static int
NAME_(ahci_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	switch (NAME_(ahci_region)(cpssp, addr)) {
	case NAME_(AHCI_HOST):
		ret = NAME_(ahci_host_w)(cpssp, addr, bs, val);
		break;
#if 1 <= SATA_PORTS
	case NAME_(AHCI_PORT0):
		ret = NAME_(ahci_p0_w)(cpssp, addr, bs, val);
		break;
#endif
#if 2 <= SATA_PORTS
	case NAME_(AHCI_PORT1):
		ret = NAME_(ahci_p1_w)(cpssp, addr, bs, val);
		break;
#endif
#if 3 <= SATA_PORTS
	case NAME_(AHCI_PORT2):
		ret = NAME_(ahci_p2_w)(cpssp, addr, bs, val);
		break;
#endif
#if 4 <= SATA_PORTS
	case NAME_(AHCI_PORT3):
		ret = NAME_(ahci_p3_w)(cpssp, addr, bs, val);
		break;
#endif
#if 5 <= SATA_PORTS
	case NAME_(AHCI_PORT4):
		ret = NAME_(ahci_p4_w)(cpssp, addr, bs, val);
		break;
#endif
#if 6 <= SATA_PORTS
	case NAME_(AHCI_PORT5):
		ret = NAME_(ahci_p5_w)(cpssp, addr, bs, val);
		break;
#endif
	case NAME_(AHCI_NONE):
	default:
		ret = 1;
		break;
	}

	return ret;
}

static int
NAME_(ahci_map_r)(
	void *_cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp,
	char **haddrp
)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	switch (NAME_(ahci_region)(cpssp, addr)) {
	case NAME_(AHCI_HOST):
		*cfp = NAME_(ahci_host_r);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#if 1 <= SATA_PORTS
	case NAME_(AHCI_PORT0):
		*cfp = NAME_(ahci_p0_r);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 2 <= SATA_PORTS
	case NAME_(AHCI_PORT1):
		*cfp = NAME_(ahci_p1_r);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 3 <= SATA_PORTS
	case NAME_(AHCI_PORT2):
		*cfp = NAME_(ahci_p2_r);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 4 <= SATA_PORTS
	case NAME_(AHCI_PORT3):
		*cfp = NAME_(ahci_p3_r);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 5 <= SATA_PORTS
	case NAME_(AHCI_PORT4):
		*cfp = NAME_(ahci_p4_r);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 6 <= SATA_PORTS
	case NAME_(AHCI_PORT5):
		*cfp = NAME_(ahci_p5_r);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
	case NAME_(AHCI_NONE):
	default:
		ret = 1;
		break;
	}

	return ret;
}

static int
NAME_(ahci_map_w)(
	void *_cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp,
	char **haddrp
)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	switch (NAME_(ahci_region)(cpssp, addr)) {
	case NAME_(AHCI_HOST):
		*cfp = NAME_(ahci_host_w);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#if 1 <= SATA_PORTS
	case NAME_(AHCI_PORT0):
		*cfp = NAME_(ahci_p0_w);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 2 <= SATA_PORTS
	case NAME_(AHCI_PORT1):
		*cfp = NAME_(ahci_p1_w);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 3 <= SATA_PORTS
	case NAME_(AHCI_PORT2):
		*cfp = NAME_(ahci_p2_w);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 4 <= SATA_PORTS
	case NAME_(AHCI_PORT3):
		*cfp = NAME_(ahci_p3_w);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 5 <= SATA_PORTS
	case NAME_(AHCI_PORT4):
		*cfp = NAME_(ahci_p4_w);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
#if 6 <= SATA_PORTS
	case NAME_(AHCI_PORT5):
		*cfp = NAME_(ahci_p5_w);
		*csp = cpssp;
		*haddrp = NULL;
		ret = 0;
		break;
#endif
	case NAME_(AHCI_NONE):
	default:
		ret = 1;
		break;
	}

	return ret;
}
#endif /* SATA_PORTS */

/* I/O Interface ------------------------------------------------------- */

static void
NAME_(port_read)(
	struct cpssp *cpssp,
	unsigned int nr,
	unsigned int port,
	uint16_t *valp
)
{
#if ! SATA_PORTS
	/* PATA only */
	DEFAULT_NAME_(pata_inw)(cpssp, nr, port, valp);

#elif PATA_PORTS && SATA_PORTS
	/* PATA/SATA */
	if ((cpssp->NAME.map_value >> nr) & 1) {
		DEFAULT_NAME_(pata_inw)(cpssp, port, valp);
	} else {
		NAME_(emul_port_read)(cpssp, nr, port, valp);
	}

#elif ! PATA_PORTS
	/* SATA only */
	NAME_(emul_port_read)(cpssp, nr, port, valp);
#endif
}

static void
NAME_(port_write)(
	struct cpssp *cpssp,
	unsigned int nr,
	unsigned int port,
	uint16_t val
)
{
#if ! SATA_PORTS
	/* PATA only */
	DEFAULT_NAME_(pata_outw)(cpssp, nr, port, val);

#elif PATA_PORTS && SATA_PORTS
	/* PATA/SATA */
	if ((cpssp->NAME.map_value >> nr) & 1) {
		DEFAULT_NAME_(pata_outw)(cpssp, port, val);
	} else {
		NAME_(emul_port_write)(cpssp, nr, port, val);
	}

#elif ! PATA_PORTS
	/* SATA only */
	NAME_(emul_port_write)(cpssp, nr, port, val);
#endif
}

static void
NAME_(dma_read)(
	struct cpssp *cpssp,
	unsigned int nr,
	uint16_t *valp
)
{
#if ! SATA_PORTS
	/* PATA only */
	DEFAULT_NAME_(pata_dmainw)(cpssp, nr, valp);

#elif PATA_PORTS && SATA_PORTS
	/* PATA/SATA */
	if ((cpssp->NAME.map_value >> nr) & 1) {
		DEFAULT_NAME_(pata_dmainw)(cpssp, valp);
	} else {
		NAME_(emul_dma_read)(cpssp, nr, valp);
	}

#elif ! PATA_PORTS
	/* SATA only */
	NAME_(emul_dma_read)(cpssp, nr, valp);
#endif
}

static void
NAME_(dma_read_end)(struct cpssp *cpssp, unsigned int nr)
{
#if ! SATA_PORTS
	/* PATA only */
	/* Nothing to do... */

#elif PATA_PORTS && SATA_PORTS
	/* PATA/SATA */
	if ((cpssp->NAME.map_value >> nr) & 1) {
		/* Nothing to do... */
	} else {
		NAME_(emul_dma_read_end)(cpssp, nr);
	}

#elif ! PATA_PORTS
	/* SATA only */
	NAME_(emul_dma_read_end)(cpssp, nr);
#endif
}

static void
NAME_(dma_write)(
	struct cpssp *cpssp,
	unsigned int nr,
	uint16_t val
)
{
#if ! SATA_PORTS
	/* PATA only */
	DEFAULT_NAME_(pata_dmaoutw)(cpssp, nr, val);

#elif PATA_PORTS && SATA_PORTS
	/* PATA/SATA */
	if (! ((cpssp->NAME.map_value >> nr) & 1)) {
		NAME_(emul_dma_write)(cpssp, nr, val);
	} else {
		DEFAULT_NAME_(pata_dmaoutw)(cpssp, val);
	}

#elif ! PATA_PORTS
	/* SATA only */
	NAME_(emul_dma_write)(cpssp, nr, val);
#endif
}

static void
NAME_(dma_write_end)(struct cpssp *cpssp, unsigned int nr)
{
#if ! SATA_PORTS
	/* PATA only */
	/* Nothing to do... */

#elif PATA_PORTS && SATA_PORTS
	/* PATA/SATA */
	if ((cpssp->NAME.map_value >> nr) & 1) {
		/* Nothing to do... */
	} else {
		NAME_(emul_dma_write_end)(cpssp, nr);
	}

#elif ! PATA_PORTS
	/* SATA only */
	NAME_(emul_dma_write_end)(cpssp, nr);
#endif
}

static int
NAME_(cmd_r)(
	struct cpssp *cpssp,
	unsigned int nr,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	uint16_t val16;
	int ret;

	switch (addr & 0x4) {
	case 0:
		if (bs == (0b1111 << 0)) {
			NAME_(port_read)(cpssp, nr, 0, &val16);
			*valp = val16 << 0;
			NAME_(port_read)(cpssp, nr, 0, &val16);
			*valp |= val16 << 16;
			ret = 0;

		} else if (bs == (0b11 << 0)) {
			NAME_(port_read)(cpssp, nr, 0, &val16);
			*valp = val16 << 0;
			ret = 0;

		} else if (bs == (1 << 1)) {
			NAME_(port_read)(cpssp, nr, 1, &val16);
			*valp = val16 << 8;
			ret = 0;

		} else if (bs == (1 << 2)) {
			NAME_(port_read)(cpssp, nr, 2, &val16);
			*valp = val16 << 16;
			ret = 0;

		} else if (bs == (1 << 3)) {
			NAME_(port_read)(cpssp, nr, 3, &val16);
			*valp = val16 << 24;
			ret = 0;

		} else {
			ret = 1;
		}
		break;

	case 4:
		if (bs == (1 << 0)) {
			NAME_(port_read)(cpssp, nr, 4, &val16);
			*valp = val16 << 0;
			ret = 0;

		} else if (bs == (1 << 1)) {
			NAME_(port_read)(cpssp, nr, 5, &val16);
			*valp = val16 << 8;
			ret = 0;

		} else if (bs == (1 << 2)) {
			NAME_(port_read)(cpssp, nr, 6, &val16);
			*valp = val16 << 16;
			ret = 0;

		} else if (bs == (1 << 3)) {
			NAME_(port_read)(cpssp, nr, 7, &val16);
			*valp = val16 << 24;
			ret = 0;

		} else {
			ret = 1;
		}
		break;

	default:
		assert(0);
	}

	if (DEBUG) {
		fprintf(stderr, "%s: nr=%d, addr=0x%02x, bs=0x%x, val=0x%08x, ret=%d\n",
				__FUNCTION__, nr, addr, bs, *valp, ret);
	}

	return ret;
}

static int
NAME_(cmd_w)(
	struct cpssp *cpssp,
	unsigned int nr,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	int ret;

	switch (addr & 0x4) {
	case 0:
		if (bs == (0b1111 << 0)) {
			NAME_(port_write)(cpssp, nr, 0, (val >> 0) & 0xffff);
			NAME_(port_write)(cpssp, nr, 0, (val >> 16) & 0xffff);
			ret = 0;

		} else if (bs == (0b11 << 0)) {
			NAME_(port_write)(cpssp, nr, 0, (val >> 0) & 0xffff);
			ret = 0;

		} else if (bs == (1 << 1)) {
			NAME_(port_write)(cpssp, nr, 1, (val >> 8) & 0xff);
			ret = 0;

		} else if (bs == (1 << 2)) {
			NAME_(port_write)(cpssp, nr, 2, (val >> 16) & 0xff);
			ret = 0;

		} else if (bs == (1 << 3)) {
			NAME_(port_write)(cpssp, nr, 3, (val >> 24) & 0xff);
			ret = 0;

		} else {
			ret = 1;
		}
		break;

	case 4:
		if (bs == (1 << 0)) {
			NAME_(port_write)(cpssp, nr, 4, (val >> 0) & 0xff);
			ret = 0;

		} else if (bs == (1 << 1)) {
			NAME_(port_write)(cpssp, nr, 5, (val >> 8) & 0xff);
			ret = 0;

		} else if (bs == (1 << 2)) {
			NAME_(port_write)(cpssp, nr, 6, (val >> 16) & 0xff);
			ret = 0;

		} else if (bs == (1 << 3)) {
			cpssp->NAME.busy = 1;
			NAME_(port_write)(cpssp, nr, 7, (val >> 24) & 0xff);
			ret = 0;

		} else {
			ret = 1;
		}
		break;

	default:
		assert(0);
	}

	if (DEBUG) {
		fprintf(stderr, "%s: nr=%d, addr=0x%02x, bs=0x%x, val=0x%08x, ret=%d\n",
				__FUNCTION__, nr, addr, bs, val, ret);
	}

	return ret;
}

static int
NAME_(cnl_r)(
	struct cpssp *cpssp,
	unsigned int nr,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	uint16_t val16;
	int ret;

	if (bs == (1 << 2)) {
		NAME_(port_read)(cpssp, nr, 8 + 6, &val16);
		*valp = val16 << 16;
		ret = 0;

	} else {
		ret = 1;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: nr=%d, addr=0x%02x, bs=0x%x, val=0x%08x, ret=%d\n",
				__FUNCTION__, nr, addr, bs, *valp, ret);
	}

	return ret;
}

static int
NAME_(cnl_w)(
	struct cpssp *cpssp,
	unsigned int nr,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	int ret;

	if (bs == (1 << 2)) {
		NAME_(port_write)(cpssp, nr, 8 + 6, (val >> 16) & 0xffff);
		ret = 0;

	} else {
		ret = 1;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: nr=%d, addr=0x%02x, bs=0x%x, val=0x%08x, ret=%d\n",
				__FUNCTION__, nr, addr, bs, val, ret);
	}

	return ret;
}

static int
NAME_(pcmd_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(cmd_r)(cpssp, 0, addr, bs, valp);
}

static int
NAME_(pcmd_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(cmd_w)(cpssp, 0, addr, bs, val);
}

static int
NAME_(pcnl_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(cnl_r)(cpssp, 0, addr, bs, valp);
}

static int
NAME_(pcnl_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(cnl_w)(cpssp, 0, addr, bs, val);
}

static int
NAME_(scmd_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(cmd_r)(cpssp, 1, addr, bs, valp);
}

static int
NAME_(scmd_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(cmd_w)(cpssp, 1, addr, bs, val);
}

static int
NAME_(scnl_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(cnl_r)(cpssp, 1, addr, bs, valp);
}

static int
NAME_(scnl_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(cnl_w)(cpssp, 1, addr, bs, val);
}

/* DMA Interface ------------------------------------------------------- */

/*
 * Some hints...:
 *
 * Start dma_do if
 * - DMA REQUEST is asserted from drive
 * *and*
 * - DMA is allowed (BMICX bit 1 is set).
 *
 * Stop dma_do if
 * - DMA REQUEST is deasserted
 * *or*
 * - EOT is set *and* COUNT is zero
 */
static void
NAME_(dma_do)(struct cpssp *cpssp, unsigned int nr)
{
	uint32_t val;
	uint16_t word;

	if (STATE_GET(eot[nr]) && STATE_GET(count[nr]) == 0) {
		/* Nothing more to transfer. */
		return;
	}

	while (cpssp->NAME.dma_val[nr]
	    && cpssp->NAME.bmi_active[nr]) {
		if (STATE_GET(count[nr]) == 0) {
			/*
			 * Read next table entry.
			 */
			uint32_t w0;
			uint32_t w1;

			DEFAULT_NAME_(mem_read)(cpssp, cpssp->NAME.bmi_addr[nr], 0xf, &w0);
			cpssp->NAME.bmi_addr[nr] += sizeof(w0);
			DEFAULT_NAME_(mem_read)(cpssp, cpssp->NAME.bmi_addr[nr], 0xf, &w1);
			cpssp->NAME.bmi_addr[nr] += sizeof(w1);

			STATE_SET(addr[nr], w0 & 0xfffffffe);
			STATE_SET(count[nr], w1 & 0xfffe);
			STATE_SET(eot[nr], (w1 >> 31) & 1);
		}

		/*
		 * Transfer data by DMA.
		 */
		/* Should read/write chunks of 64 bytes. FIXME VOSSI */
		if (STATE_GET(bmi_write[nr])) {
			/* DMA READ : Device -> Memory */
			NAME_(dma_read)(cpssp, nr, &word);

			if (STATE_GET(addr[nr]) & 2) {
				val = word << 16;
				DEFAULT_NAME_(mem_write)(cpssp,
					STATE_GET(addr[nr]) & ~2, 0x3 << 2,
					val);
			} else {
				val = word << 0;
				DEFAULT_NAME_(mem_write)(cpssp,
					STATE_GET(addr[nr]) & ~2, 0x3 << 0,
					val);
			}

		} else {
			/* DMA WRITE: Memory -> Device */
			if (STATE_GET(addr[nr]) & 2) {
				DEFAULT_NAME_(mem_read)(cpssp,
					STATE_GET(addr[nr]) & ~2, 0x3 << 2,
					&val);
				word = val >> 16;
			} else {
				DEFAULT_NAME_(mem_read)(cpssp,
					STATE_GET(addr[nr]) & ~2, 0x3 << 0,
					&val);
				word = val >> 0;
			}

			NAME_(dma_write)(cpssp, nr, word);
		}

		STATE_SET(addr[nr], STATE_GET(addr[nr]) + 2);
		STATE_SET(count[nr], STATE_GET(count[nr]) - 2);

		if (STATE_GET(eot[nr])
		 && STATE_GET(count[nr]) == 0) {
			if (STATE_GET(bmi_write[nr])) {
				NAME_(dma_read_end)(cpssp, nr);
			} else {
				NAME_(dma_write_end)(cpssp, nr);
			}
			break;
		}
	}
}

static void
NAME_(dma_start)(struct cpssp *cpssp, unsigned int nr)
{
	if (__sync_bool_compare_and_swap(&cpssp->NAME.bmi_active[nr], 0, 1)) {
		NAME_(dma_do)(cpssp, nr);
		cpssp->NAME.bmi_active[nr] = 0;
	}
}

static void
NAME_(pata_dmarq_set)(struct cpssp *cpssp, unsigned int nr, unsigned int val)
{
	assert(nr < 2);

	cpssp->NAME.dma_val[nr] = val;
	if (val == 0) {
		return;
	}

	__sync_synchronize();

	if (STATE_GET(bmi_start)) {
		NAME_(dma_start)(cpssp, nr);
	}
}

static void
NAME_(pata_irqrq_in_set)(struct cpssp *cpssp, unsigned int nr, unsigned int val)
{
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	cpssp->NAME.irq_val[nr] = val;
#endif

	if (val) {
		cpssp->NAME.bmi_int[nr] = 1;
	}

#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
	if (cpssp->NAME.native_enable) {
		NAME_(irqrq_out_set)(cpssp,
			cpssp->NAME.irq_val[0] | cpssp->NAME.irq_val[1]);
	} else
#endif
	{
		DEFAULT_NAME_(legacy_irqrq_out_set)(cpssp, nr, val);
	}
}

static int
NAME_(legacy_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;
	unsigned int nr;

	*valp = 0;
	switch (addr & 0xf) {
	case 0x00:
	case 0x08:
		/* Bus Master IDE Command Register Primary/Secondary */
		/* 14.2.1 */
		nr = (addr >> 3) & 1;
		*valp |= STATE_GET(bmi_start[nr]) << 0;
		*valp |= 0b00 << 1; /* Reserved */
		*valp |= STATE_GET(bmi_write[nr]) << 3;
		*valp |= 0b0000 << 4; /* Reserved */

		*valp |= 0x00 << 8; /* Reserved */

		/* Bus Master IDE Status Register Primary/Secondary */
		*valp |= cpssp->NAME.bmi_active[nr] << (0 + 16);
		*valp |= 0b0 << (1 + 16); /* FIXME */
		*valp |= cpssp->NAME.bmi_int[nr] << (2 + 16);
		*valp |= 0b00 << (3 + 16); /* Reserved */
		*valp |= 0b1 << (5 + 16); /* FIXME */
		*valp |= 0b1 << (6 + 16); /* FIXME */
		*valp |= cpssp->NAME.bmi_prdis[nr] << (7 + 16);

		*valp |= 0x00 << 24; /* Reserved */
		break;

	case 0x04:
	case 0x0c:
		/* Bus Master IDE Descriptor Table Pointer Register */
		/* 14.2.3 */
		nr = (addr >> 3) & 1;
		*valp |= cpssp->NAME.bmi_addr[nr];
		break;

	default:
		assert(0);
	}
	return 0;
}

static int
NAME_(legacy_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;
	unsigned int nr;

	switch (addr & 0xf) {
	case 0x00:
	case 0x08:
		/* Bus Master IDE Command Register */
		/* 14.2.1 */
		nr = (addr >> 3) & 1;
		if ((bs >> 0) & 1) {
			int start;

			start = ! STATE_GET(bmi_start[nr]) && ((val >> 0) & 1);

			if (start) {
				/* Reset DMA state. */
				STATE_SET(count[nr], 0);
				STATE_SET(eot[nr], 0);
			}

			STATE_SET(bmi_start[nr], (val >> 0) & 0b1);
			/* Bit 1-2: Reserved */
			STATE_SET(bmi_write[nr], (val >> 3) & 0b1);
			/* Bit 4-7: Reserved */

			__sync_synchronize();

			if (cpssp->NAME.dma_val[nr]
			 && start) {
				/* Start DMA. */
				NAME_(dma_start)(cpssp, nr);
			}
		}

		if ((bs >> 1) & 1) {
			/* Reserved */
		}

		/* Bus Master IDE Status Register */
		/* 14.2.2 */
		if ((bs >> 2) & 1) {
			/* Bit 0: Read-only */
			/* Bit 1: FIXME */
			cpssp->NAME.bmi_int[nr] &= ~((val >> (2 + 16)) & 1);
			/* Bit 3-4: Reserved */
			/* Bit 5: FIXME */
			/* Bit 6: FIXME */
			cpssp->NAME.bmi_prdis[nr] &= ~((val >> (7 + 16)) & 1);
		}

		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x04:
	case 0x0c:
		/* Bus Master IDE Descriptor Table Pointer Register */
		/* 14.2.3 */
		nr = (addr >> 3) & 1;
		if ((bs >> 0) & 1) {
			cpssp->NAME.bmi_addr[nr] &= ~(0xfc << 0);
			cpssp->NAME.bmi_addr[nr] |= val & (0xfc << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.bmi_addr[nr] &= ~(0xff << 8);
			cpssp->NAME.bmi_addr[nr] |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.bmi_addr[nr] &= ~(0xff << 16);
			cpssp->NAME.bmi_addr[nr] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.bmi_addr[nr] &= ~(0xff << 24);
			cpssp->NAME.bmi_addr[nr] |= val & (0xff << 24);
		}
		break;

	default:
		assert(0);
	}
	return 0;
}

#if SATA_PORTS
static int
NAME_(legacy_ext_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	*valp = 0;
	switch (addr & 0xf) {
	case 0x0:
		/* AHCI Index Register */
		/* 14.2.4 */
		*valp |= 0b00 << 0; /* Reserved */
		*valp |= cpssp->NAME.ahci_index << (2-2);
		*valp |= 0b000000000000000000000 << 11; /* Reserved */
		break;

	case 0x4:
		/* AHCI Data Register */
		/* 14.2.5 */
		NAME_(ahci_r)(cpssp, cpssp->NAME.ahci_index, bs, valp);
		break;

	case 0x8:
	case 0xc:
		*valp |= 0x00000000 << 0; /* Reserved */
		break;

	default:
		assert(0);
	}
	return 0;
}

static int
NAME_(legacy_ext_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	switch (addr & 0xf) {
	case 0x0:
		/* AHCI Index Register */
		/* 14.2.4 */
		if ((bs >> 0) & 1) {
			/* Bit 0-1: Reserved */
			cpssp->NAME.ahci_index &= ~(0xfc << 0);
			cpssp->NAME.ahci_index |= val & (0xfc << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.ahci_index &= ~(0x03 << 8);
			cpssp->NAME.ahci_index |= val & (0x03 << 8);
			/* Bit 10-15: Reserved Correct? FIXME */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x4:
		/* AHCI Data Register */
		/* 14.2.5 */
		NAME_(ahci_w)(cpssp, cpssp->NAME.ahci_index, bs, val);
		break;

	case 0x8:
	case 0xc:
		/* Reserved */
		break;

	default:
		assert(0);
	}
	return 0;
}
#endif

#if SATA_PORTS
static int
NAME_(sidpba_r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	*valp = 0;
	switch (addr) {
	case 0x0:
		/* SATA Index Register */
		/* 15.3.1 */
		*valp |= cpssp->NAME.register_index << 0;
		*valp |= cpssp->NAME.port_index << 8;
		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x4:
		/* SATA Data Register */
		/* 15.3.2 */
		/* FIXME */
		fprintf(stderr, "WARNING: %s: port_index=%d, register_index=%d.\n",
				__FUNCTION__,
				cpssp->NAME.port_index,
				cpssp->NAME.register_index);
		break;

	default:
		*valp |= 0x00000000 << 0; /* Reserved */
		break;
	}

	return 0;
}

static int
NAME_(sidpba_w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	switch (addr) {
	case 0x0:
		/* SATA Index Register */
		/* 15.3.1 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.register_index = (val >> 0) & 0xff;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.port_index = (val >> 8) & 0xff;
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x4:
		/* SATA Data Register */
		/* 15.3.2 */
		/* FIXME */
		fprintf(stderr, "WARNING: %s: port_index=%d, register_index=%d val=0x%08x.\n",
				__FUNCTION__,
				cpssp->NAME.port_index,
				cpssp->NAME.register_index,
				val);
		break;

	default:
		/* Reserved */
		break;
	}

	return 0;
}
#endif /* DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00 */

static enum NAME_(io_region) {
	NAME_(IO_PCMD),
	NAME_(IO_PCNL),
	NAME_(IO_SCMD),
	NAME_(IO_SCNL),
	NAME_(IO_LEGACY_BUS_MASTER),
#if SATA_PORTS
	NAME_(IO_LEGACY_BUS_MASTER_EXT),
	NAME_(IO_SIDPBA),
#endif
	NAME_(IO_NONE),
} NAME_(io_region)(struct cpssp *cpssp, uint32_t addr, unsigned int bs)
{
	if (cpssp->NAME.io_enable) {
		if (cpssp->NAME.ide0) {
			if (0x01f0 == (addr & ~0x7)
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			 || cpssp->NAME.pcmd_bar == (addr & ~0x7)
#endif
			) {
				return NAME_(IO_PCMD);
			}
			if (bs == (1 << 2)
			 && (0x03f4 == (addr & ~0x3)
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			  || cpssp->NAME.pcnl_bar == (addr & ~0x3)
#endif
			)) {
				return NAME_(IO_PCNL);
			}
		}
		if (cpssp->NAME.ide1) {
			if (0x0170 == (addr & ~0x7)
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			 || cpssp->NAME.scmd_bar == (addr & ~0x7)
#endif
			) {
				return NAME_(IO_SCMD);
			}
			if (bs == (1 << 2)
			 && (0x0374 == (addr & ~0x3)
#if DEVICE_ID == 0x27c0 || DEVICE_ID == 0x3a00
			  || cpssp->NAME.scnl_bar == (addr & ~0x3)
#endif
			)) {
				return NAME_(IO_SCNL);
			}
		}
		if (cpssp->NAME.bar == (addr & ~0xf)) {
			return NAME_(IO_LEGACY_BUS_MASTER);
		}
#if SATA_PORTS
		if (cpssp->NAME.sata_mode_select != 0b00 /* Non-IDE */
		 && cpssp->NAME.bar == (addr & ~0x1f)) {
			return NAME_(IO_LEGACY_BUS_MASTER_EXT);
		}
		if (cpssp->NAME.sata_mode_select == 0b00 /* IDE */
		 && cpssp->NAME.sidpba_abar == (addr & ~0xf)) {
			return NAME_(IO_SIDPBA);
		}
#endif
	}
	return NAME_(IO_NONE);
}

static int
NAME_(ior)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	switch (NAME_(io_region)(cpssp, addr, bs)) {
	case NAME_(IO_PCMD):
		return NAME_(pcmd_r)(cpssp, addr, bs, valp);
	case NAME_(IO_PCNL):
		return NAME_(pcnl_r)(cpssp, addr, bs, valp);
	case NAME_(IO_SCMD):
		return NAME_(scmd_r)(cpssp, addr, bs, valp);
	case NAME_(IO_SCNL):
		return NAME_(scnl_r)(cpssp, addr, bs, valp);
	case NAME_(IO_LEGACY_BUS_MASTER):
		return NAME_(legacy_r)(cpssp, addr, bs, valp);
#if SATA_PORTS
	case NAME_(IO_LEGACY_BUS_MASTER_EXT):
		return NAME_(legacy_ext_r)(cpssp, addr, bs, valp);
	case NAME_(IO_SIDPBA):
		return NAME_(sidpba_r)(cpssp, addr, bs, valp);
#endif
	case NAME_(IO_NONE):
	default:
		return 1;
	}
}

static int
NAME_(iow)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	switch (NAME_(io_region)(cpssp, addr, bs)) {
	case NAME_(IO_PCMD):
		return NAME_(pcmd_w)(cpssp, addr, bs, val);
	case NAME_(IO_PCNL):
		return NAME_(pcnl_w)(cpssp, addr, bs, val);
	case NAME_(IO_SCMD):
		return NAME_(scmd_w)(cpssp, addr, bs, val);
	case NAME_(IO_SCNL):
		return NAME_(scnl_w)(cpssp, addr, bs, val);
	case NAME_(IO_LEGACY_BUS_MASTER):
		return NAME_(legacy_w)(cpssp, addr, bs, val);
#if SATA_PORTS
	case NAME_(IO_LEGACY_BUS_MASTER_EXT):
		return NAME_(legacy_ext_w)(cpssp, addr, bs, val);
	case NAME_(IO_SIDPBA):
		return NAME_(sidpba_w)(cpssp, addr, bs, val);
#endif
	case NAME_(IO_NONE):
	default:
		return 1;
	}
}

static int
NAME_(ior_info)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp
)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	switch (NAME_(io_region)(cpssp, addr, bs)) {
	case NAME_(IO_PCMD):
		*cfp = NAME_(pcmd_r);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_PCNL):
		*cfp = NAME_(pcnl_r);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_SCMD):
		*cfp = NAME_(scmd_r);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_SCNL):
		*cfp = NAME_(scnl_r);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_LEGACY_BUS_MASTER):
		*cfp = NAME_(legacy_r);
		*csp = cpssp;
		ret = 0;
		break;
#if SATA_PORTS
	case NAME_(IO_LEGACY_BUS_MASTER_EXT):
		*cfp = NAME_(legacy_ext_r);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_SIDPBA):
		*cfp = NAME_(sidpba_r);
		*csp = cpssp;
		ret = 0;
		break;
#endif
	case NAME_(IO_NONE):
	default:
		ret = 1;
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%04x, bs=0x%x, ret=%d\n",
				__FUNCTION__, addr, bs, ret);
	}

	return ret;
}

static int
NAME_(iow_info)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp
)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	switch (NAME_(io_region)(cpssp, addr, bs)) {
	case NAME_(IO_PCMD):
		*cfp = NAME_(pcmd_w);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_PCNL):
		*cfp = NAME_(pcnl_w);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_SCMD):
		*cfp = NAME_(scmd_w);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_SCNL):
		*cfp = NAME_(scnl_w);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_LEGACY_BUS_MASTER):
		*cfp = NAME_(legacy_w);
		*csp = cpssp;
		ret = 0;
		break;
#if SATA_PORTS
	case NAME_(IO_LEGACY_BUS_MASTER_EXT):
		*cfp = NAME_(legacy_ext_w);
		*csp = cpssp;
		ret = 0;
		break;
	case NAME_(IO_SIDPBA):
		*cfp = NAME_(sidpba_w);
		*csp = cpssp;
		ret = 0;
		break;
#endif
	case NAME_(IO_NONE):
	default:
		ret = 1;
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%04x, bs=0x%x, ret=%d\n",
				__FUNCTION__, addr, bs, ret);
	}

	return ret;
}

/* Memory Interface ---------------------------------------------------- */

static enum NAME_(mem_region) {
	NAME_(MEM_NONE),
#if SATA_PORTS
	NAME_(MEM_AHCI),
#endif
} NAME_(mem_region)(struct cpssp *cpssp, uint32_t addr)
{
#if SATA_PORTS
	if (! cpssp->NAME.mem_enable) {
		return NAME_(MEM_NONE);
	}
	if (cpssp->NAME.sata_mode_select != 0b00 /* Non-IDE */
	 && (cpssp->NAME.sidpba_abar & 0xffff0000)
	 && cpssp->NAME.sidpba_abar == (addr & ~0x3ff)) {
		return NAME_(MEM_AHCI);
	}
#endif

	return NAME_(MEM_NONE);
}

static int
NAME_(mr)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	switch (NAME_(mem_region)(cpssp, addr)) {
#if SATA_PORTS
	case NAME_(MEM_AHCI):
		return NAME_(ahci_r)(cpssp, addr, bs, valp);
#endif
	case NAME_(MEM_NONE):
	default:
		return 1;
	}
}

static int
NAME_(mw)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	switch (NAME_(mem_region)(cpssp, addr)) {
#if SATA_PORTS
	case NAME_(MEM_AHCI):
		return NAME_(ahci_w)(cpssp, addr, bs, val);
#endif
	case NAME_(MEM_NONE):
	default:
		return 1;
	}
}

static int
NAME_(map_r)(
	void *_cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp,
	char **haddrp
)
{
	struct cpssp *cpssp = _cpssp;

	switch (NAME_(mem_region)(cpssp, addr)) {
#if SATA_PORTS
	case NAME_(MEM_AHCI):
		return NAME_(ahci_map_r)(cpssp, addr, cfp, csp, haddrp);
#endif
	case NAME_(MEM_NONE):
	default:
		return 1;
	}
}

static int
NAME_(map_w)(
	void *_cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp,
	char **haddrp
)
{
	struct cpssp *cpssp = _cpssp;

	switch (NAME_(mem_region)(cpssp, addr)) {
#if SATA_PORTS
	case NAME_(MEM_AHCI):
		return NAME_(ahci_map_w)(cpssp, addr, cfp, csp, haddrp);
#endif
	case NAME_(MEM_NONE):
	default:
		return 1;
	}
}

#ifndef IDE_INSPECTION
/* Event Interface ----------------------------------------------------- */

static void
NAME_(timer)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */
	if (cpssp->is_under_inspection) {
		DEFAULT_NAME_(busy_set)(cpssp, cpssp->NAME.busy);
		cpssp->NAME.busy = 0;

		cpssp->NAME.time += TIME_HZ / 2;
		time_call_at(cpssp->NAME.time, NAME_(timer), cpssp);
	} else {
		DEFAULT_NAME_(busy_set)(cpssp, cpssp->NAME.busy);
		cpssp->NAME.busy = 0;

		cpssp->NAME.time += TIME_HZ / 2;
		time_call_at(cpssp->NAME.time, NAME_(timer), cpssp);
	}
}

/* Create/Destroy Interface -------------------------------------------- */

static void
NAME_(create)(struct cpssp *cpssp)
{
	cpssp->NAME.time = TIME_HZ / 2;
	time_call_at(cpssp->NAME.time, NAME_(timer), cpssp);

	STATE_DECL("addr0", addr[0], DATA_TYPE_UINT32_T);
	STATE_DECL("addr1", addr[1], DATA_TYPE_UINT32_T);
	STATE_DECL("count0", count[0], DATA_TYPE_UINT32_T);
	STATE_DECL("count1", count[1], DATA_TYPE_UINT32_T);
	STATE_DECL("eot0", eot[0], DATA_TYPE_UINT8_T);
	STATE_DECL("eot1", eot[1], DATA_TYPE_UINT8_T);
	STATE_DECL("bmi_start0", bmi_start[0], DATA_TYPE_UINT8_T);
	STATE_DECL("bmi_start1", bmi_start[1], DATA_TYPE_UINT8_T);
	STATE_DECL("bmi_write0", bmi_write[0], DATA_TYPE_UINT8_T);
	STATE_DECL("bmi_write1", bmi_write[1], DATA_TYPE_UINT8_T);
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}
#endif /* IDE_INSPECTION */

#endif /* BEHAVIOR */
