/*
	Description: history graph computation

	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include "common.h"
#include "lanes.h"

#define IS_NODE(x) (x == NODE || x == NODE_R || x == NODE_L)

using namespace QGit;

void Lanes::init(const QString& expectedSha) {

	clear();
	activeLane = 0;
	setBoundary(false);
	add(BRANCH, expectedSha, activeLane);
}

void Lanes::clear() {

	lanes.clear();
}

void Lanes::setBoundary(bool b) {
// changes the state so must be called as first one

	NODE = (b) ? BOUNDARY_C : MERGE_FORK;
	NODE_R = (b) ? BOUNDARY_R : MERGE_FORK_R;
	NODE_L = (b) ? BOUNDARY_L : MERGE_FORK_L;
	boundary = b;

	if (boundary)
		lanes[activeLane].type = BOUNDARY;
}

bool Lanes::isFork(const QString& sha, bool& isDiscontinuity) {

	int pos = findNext(sha, 0);
	isDiscontinuity = (activeLane != pos);
	if (pos != -1)
		return (findNext(sha, pos + 1) != -1);

	return false; // new branch case

/*
	int cnt = 0;
	while (pos != -1) {
		cnt++;
		pos = findNext(sha, pos + 1);
//		if (isDiscontinuity)
//			isDiscontinuity = (activeLane != pos);
	}
	return (cnt > 1);
*/
}

void Lanes::setFork(const QString& sha) {

	int rangeStart, rangeEnd, idx;
	rangeStart = rangeEnd = idx = findNext(sha, 0);

	while (idx != -1) {
		rangeEnd = idx;
		lanes[idx].type = TAIL;
		idx = findNext(sha, idx + 1);
	}
	lanes[activeLane].type = NODE;

	int& startT = lanes[rangeStart].type;
	int& endT = lanes[rangeEnd].type;

	if (startT == NODE)
		startT = NODE_L;

	if (endT == NODE)
		endT = NODE_R;

	if (startT == TAIL)
		startT = TAIL_L;

	if (endT == TAIL)
		endT = TAIL_R;

	for (int i = rangeStart + 1; i < rangeEnd; i++) {

		int& t = lanes[i].type;

		if (t == NOT_ACTIVE)
			t = CROSS;

		if (t == EMPTY)
			t = CROSS_EMPTY;
	}
}

void Lanes::setMerge(const QStringList& parents) {
// setFork() must be called before setMerge()

	if (boundary)
		return; // handle as a simple active line

	int& t = lanes[activeLane].type;
	bool wasFork   = (t == NODE);
	bool wasFork_L = (t == NODE_L);
	bool wasFork_R = (t == NODE_R);
	bool joinWasACross = false;

	t = NODE;

	int rangeStart = activeLane, rangeEnd = activeLane;
	QStringList::const_iterator it(parents.constBegin());
	for (++it; it != parents.constEnd(); ++it) { // skip first parent

		int idx = findNext(*it, 0);
		if (idx != -1) {

			if (lanes[idx].type == CROSS)
				joinWasACross = true;

			lanes[idx].type = JOIN;

			if (idx > rangeEnd)
				rangeEnd = idx;

			if (idx < rangeStart)
				rangeStart = idx;
		} else
			rangeEnd = add(HEAD, *it, rangeEnd + 1);
	}
	int& startT = lanes[rangeStart].type;
	int& endT = lanes[rangeEnd].type;

	if (startT == NODE && !wasFork && !wasFork_R)
		startT = NODE_L;

	if (endT == NODE && !wasFork && !wasFork_L)
		endT = NODE_R;

	if (startT == JOIN && !joinWasACross)
		startT = JOIN_L;

	if (endT == JOIN && !joinWasACross)
		endT = JOIN_R;

	if (startT == HEAD)
		startT = HEAD_L;

	if (endT == HEAD)
		endT = HEAD_R;

	for (int i = rangeStart + 1; i < rangeEnd; i++) {

		int& t = lanes[i].type;

		if (t == NOT_ACTIVE)
			t = CROSS;

		if (t == EMPTY)
			t = CROSS_EMPTY;

		if (t == TAIL_R || t == TAIL_L)
			t = TAIL;
	}
}

void Lanes::setInitial() {

	int& t = lanes[activeLane].type;
	if (!IS_NODE(t) && t != APPLIED)
		t = (boundary) ? BOUNDARY : INITIAL;
}

void Lanes::setApplied() {

	// applied patches are not merges, nor forks
	lanes[activeLane].type = APPLIED; // TODO test with boundaries
}

void Lanes::changeActiveLane(const QString& sha) {

	int& t = lanes[activeLane].type;
	if (t == INITIAL || isBoundary(t))
		t = EMPTY;
	else
		t = NOT_ACTIVE;

	int idx = findNext(sha, 0); // find  first sha
	if (idx != -1)
		lanes[idx].type = ACTIVE; // called before setBoundary()
	else
		idx = add(BRANCH, sha, activeLane); // new branch

	activeLane = idx;
}

void Lanes::afterMerge() {

	if (boundary)
		return; // will be reset by changeActiveLane()

	for (uint i = 0; i < lanes.count(); i++) {

		int& t = lanes[i].type;

		if (isHead(t) || isJoin(t) || t == CROSS)
			t = NOT_ACTIVE;

		if (t == CROSS_EMPTY)
			t = EMPTY;

		if (IS_NODE(t))
			t = ACTIVE;
	}
}

void Lanes::afterFork() {

	for (uint i = 0; i < lanes.count(); i++) {

		int& t = lanes[i].type;

		if (t == CROSS)
			t = NOT_ACTIVE;

		if (isTail(t) || t == CROSS_EMPTY)
			t = EMPTY;

		if (!boundary && IS_NODE(t))
			t = ACTIVE; // boundary will be reset by changeActiveLane()
	}
	while (lanes.last().type == EMPTY)
		lanes.pop_back();
}

bool Lanes::isBranch() {

	return (lanes[activeLane].type == BRANCH);
}

void Lanes::afterBranch() {

	lanes[activeLane].type = ACTIVE; // TODO test with boundaries
}

void Lanes::afterApplied() {

	lanes[activeLane].type = ACTIVE; // TODO test with boundaries
}

void Lanes::nextParent(const QString& sha) {

	lanes[activeLane].next = (boundary) ? "" : sha;
}

void Lanes::getLanes(QValueVector<int> &ln) {

	ln.clear();
	for (uint i = 0; i < lanes.count(); i++)
		ln.append(lanes[i].type);
}

int Lanes::findNext(const QString& next, int pos) {

	for (uint i = pos; i < lanes.count(); i++)
		if (lanes[i].next == next)
			return i;
	return -1;
}

int Lanes::findType(int type, int pos) {

	for (uint i = pos; i < lanes.count(); i++)
		if (lanes[i].type == type)
			return i;
	return -1;
}

int Lanes::add(int type, const QString& next, int pos) {

	// first check empty lanes starting from pos
	if (pos < (int)lanes.count()) {
		pos = findType(EMPTY, pos);
		if (pos != -1) {
			lanes[pos].type = type;
			lanes[pos].next = next;
			return pos;
		}
	}
	// if not enough we add new lanes
	Lane l;
	l.type = type;
	l.next = next;
	lanes.append(l);
	return lanes.count() - 1;
}
