/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#ifndef GIT_H
#define GIT_H

#include <qtimer.h>
#include "exceptionmanager.h"
#include "common.h"
#include "lanes.h"
#include "myprocess.h"

class Cache;
class Annotate;
class MyProcess;
class QTextCodec;
class DataLoader;
class QRegExp;

class Git : public QObject {
Q_OBJECT
public:
	Git(QWidget* par = 0, const char* name = 0);
	~Git();

	const QString getBaseDir(bool* c, SCRef wd, bool* ok = NULL, QString* gd = NULL);
	bool init(SCRef workDir, bool askForRange, QStringList* filterList, bool* quit);
	void stop(bool notSaveCache);
	void setLane(SCRef sha, bool isHistory);
	Annotate* startAnnotate(QObject* guiObj);
	const FileAnnotation* lookupAnnotation(Annotate* ann, SCRef fileName, SCRef sha);
	void cancelAnnotate(Annotate* ann);
	void cancelFileHistoryLoading();
	void cancelProcess(MyProcess* p);
	bool isCommittingMerge(); // check for existence of MERGE_HEAD
	bool isStGITStack();
	bool isTagName(SCRef tag);
	bool isPatchName(SCRef patchName);
	bool isUnapplied(SCRef sha);
	bool isSameFiles(SCRef tree1Sha, SCRef tree2Sha);
	bool isNothingToCommit() const { return nothingToCommit; }
	bool isUnknownFiles() const { return unknownFiles; }
	const Rev* revLookup(SCRef sha, bool historyRevs = false);
	MyProcess* getDiff(SCRef sha, QObject* receiver, SCRef diffToSha, bool combined);
	MyProcess* getFile(SCRef file, SCRef revSha, QObject* receiver, QString* runOutput);
	void getFileHistory(SCRef name);
	void getFileFilter(SCRef path, QMap<QString, bool>& shaMap);
	bool getPatchFilter(SCRef exp, bool isRegExp, QMap<QString, bool>& shaMap);
	const RevFile* getFiles(SCRef sha, SCRef sha2 = "", bool all = false, SCRef path = "");
	void getTree(SCRef ts, SList nm, SList sha, SList type, bool wd, SCRef treePath);
	const QString getLocalDate(SCRef gitDate);
	const QString getDesc(SCRef sha, QRegExp& shortLogRE, QRegExp& longLogRE);
	const QString getDefCommitMsg();
	const QString getLaneParent(SCRef fromSHA, uint laneNum);
	const QStringList getChilds(SCRef parent);
	const QStringList getNearTags(bool down, SCRef sha);
	const QStringList getDescBranches(SCRef sha);
	const QString getShortLog(SCRef sha);
	const QStringList getTagNames(bool onlyLoaded = false);
	const QStringList getBranchNames();
	const QString getTagMsg(SCRef sha);
	const QString getRefSha(SCRef refName);
	const QString getRevInfo(SCRef sha, bool isHistory);
	void getWorkDirFiles(SCRef status, SList files, SList dirs);
	QTextCodec* getTextCodec(bool* isGitArchive);
	bool formatPatch(SCList shaList, SCRef dirPath, SCRef remoteDir = "");
	bool updateIndex(SCList selFiles);
	bool commitFiles(SCList files, SCRef msg);
	bool makeTag(SCRef sha, SCRef tag, SCRef msg);
	bool deleteTag(SCRef sha);
	bool applyPatchFile(SCRef patchPath, bool commit, bool fold, bool dropping);
	bool resetCommits(int parentDepth);
	bool stgCommit(SCList selFiles, SCRef msg, SCRef patchName, bool add);
	bool stgPush(SCRef sha);
	bool stgPop(SCRef sha);
	bool writeToFile(SCRef fileName, SCRef data);
	bool readFromFile(SCRef fileName, QString& data);
	void setTextCodec(QTextCodec* tc);
	int findFileIndex(const RevFile& rf, SCRef name);
	const QString filePath(const RevFile& rf, uint i) {
		return dirNamesVec[rf.dirs[i]] + fileNamesVec[rf.names[i]];
	}
	const QString getPatchName(SCRef sha) {
		return patchNames[sha];
	}
	void incRunningProcesses() { runningProcesses++; }
	void decRunningProcesses() { runningProcesses--; }

signals:
	void newRevsAdded(const QValueVector<QString>&);
	void newHistRevsAdded(const QValueVector<QString>&);
	void loadCompleted(const QString&);
	void histLoadCompleted();
	void cancelHistLoading();
	void cancelAllProcesses();
	void annotateReady(Annotate* ann, const QString& fileName,
	                   bool ok, const QString& msg);

public slots:
	void on_procDataReady(const QString&);
	void on_eof() { filesLoadingPending = filesLoadingCurSha = ""; }

protected slots:
	void annotateExited(Annotate* ann);
	void loadFileNames();
	void on_loaded(ulong,const QTime&,bool,const QString&,const QString&);
	void on_histLoaded(ulong,const QTime&,bool,const QString&,const QString&);
	void on_newDataReady();
	void on_newHistDataReady();
	void on_runAsScript_eof();

private:
	friend class Annotate;
	friend class MainImpl;
	friend class DataLoader;
	friend class ConsoleImpl;

	typedef QDict<Rev> RevMap;             // faster then a map
	typedef QMap<QString, QString> StrMap; // used to map SHA and patches file names
	typedef QMap<QString, int> StrSet;     // for fast QString set lookup

	bool run(SCRef cmd, QString* out = NULL, QObject* rcv = NULL, SCRef buf = "");
	MyProcess* runAsync(SCRef cmd, QObject* rcv, SCRef buf = "");
	MyProcess* runAsScript(SCRef cmd, QObject* rcv = NULL, SCRef buf = "");
	const QString getArgs(bool askForRange, bool* quit);
	bool getRefs();
	bool getStGITPatches();
	bool lookUpPatchesSHA(SCRef pn, SList files, SList filesSHA, QString& sha);
	void dirWalker(SCRef dirPath, SList files, SList filesSHA, SCRef nameFilter = "");
	void clearRevs();
	bool startRevList(SCRef args, bool startHistoryProc = false);
	bool startUnappliedList();
	bool startParseProc(SCRef initCmd);
	bool startHistoryProc(SCRef initCmd);
	void addChunk(bool addToHistory, SCRef parseBuffer);
	void parseDiffFormat(RevFile& rf, SCRef buf);
	void parseDiffFormatLine(RevFile& rf, SCRef line, int parNum);
	void getDiffIndex();
	const QString getFileSha(SCRef file, SCRef revSha);
	const Rev* fakeWorkDirRev(SCRef parent, SCRef log, SCRef longLog, int idx);
	void copyDiffIndex(RevMap& histRevs, SCRef histFileName);
	const RevFile* getAllMergeFiles(const Rev* r);
	bool isParentOf(SCRef par, SCRef child);
	bool isSameTree(SCRef sha);
	void indexTree();
	void updateDescMap(const Rev* r, uint i, QMap<QPair<uint, uint>,bool>& dm,
	                   QMap<uint, QValueVector<int> >& dv);
	void mergeNearTags(bool down, Rev* p, const Rev* r, const QMap<QPair<uint, uint>, bool>&dm);
	void mergeBranches(Rev* p, const Rev* r);
	void updateRefs(Rev& c, bool addToHistory);
	void updateLanes(Rev& c, Lanes& lns);
	void removeFiles(SCList selFiles, SCRef workDir, SCRef ext);
	void restoreFiles(SCList selFiles, SCRef workDir, SCRef ext);
	bool mkPatchFromIndex(SCRef msg, SCRef patchFile);
	const QStringList getOthersFiles();
	const QStringList getOtherFiles(SCList selFiles, bool onlyCached);
	const QString colorMatch(SCRef txt, QRegExp& regExp);
	void appendFileName(RevFile& rf, SCRef name);
	void populateFileDict();
	const QStringList noSpaceSepHack(SCRef cmd);
	void removeDeleted(SCList selFiles);

	EM_DECLARE(exGitStopped);

	QWidget* par;
	Cache* cache;
	QString workDir; // workDir is always without trailing '/'
	QString gitDir;
	QStringList tags;
	QStringList loadedTagNames;
	QStringList loadedBranchNames;
	QStringList tagsSHA;
	StrMap tagsObj;
	StrMap refsStillToFind; // to quick check if a rev is a ref during loading
	StrMap refsStillToFindMasterCopy; // to restore original content
	QStringList heads;
	QStringList headsSHA;
	QStringList refs;
	QStringList refsSHA;
	QStringList unAppliedSHA;
	QStringList appliedSHA;
	StrMap patchNames;
	QString currentBranchSHA;
	QString filesLoadingPending;
	QString filesLoadingCurSha;
	bool cacheNeedsUpdate;
	bool errorReportingEnabled;
	bool isMergeHead;
	bool isStGIT;
	bool isGIT;
	bool loadingUnAppliedPatches;
	bool unknownFiles;
	bool nothingToCommit;
	int runningProcesses;
	QString rangeSelectArgs;
	QString rangeSelectOptions;
	QString histFileName;
	QString firstNonStGitPatch;
	RevMap revs;
	RevMap histRevs;
	RevFileMap revsFiles;
	Lanes lns;
	Lanes histLns;
	int patchesStillToFind;
	StrVect fileNamesVec;
	StrVect dirNamesVec;
	StrSet fileNames; // quick look-up file name
	StrSet dirNames; // quick look-up dir name
	StrVect revOrder, histRevOrder;
	uint firstFreeLane, firstHistFreeLane;
};

#endif
