# arch-tag: 46e3b3ff-aebc-46ab-9bd3-c7cb1dff34e5
# Copyright (C) 2003-2004 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

__all__ = [ 'deprecated_callable', 'deprecated_usage' ]

import warnings
import inspect

def callable_name(obj):
    if inspect.isfunction(obj):
        return '.'.join((obj.__module__, obj.__name__))
    if inspect.ismethod(obj):
        kls = obj.im_class
        return '.'.join((kls.__module__, kls.__name__, obj.__name__))
    if isinstance(obj, tuple):
        # user-defined properties have no name, so they have to specified by a
        # (class, attribute) tuple.
        assert len(obj) == 2
        assert isinstance(obj[0], type)
        assert isinstance(obj[1], str)
        assert hasattr(*obj)
        return '.'.join((obj[0].__module__, obj[0].__name__, obj[1]))
    return obj.__name__

def callable_type(obj):
    if isinstance(obj, tuple):
        return type(getattr(*obj)).__name__
    return type(obj).__name__

def deprecated_callable(old, new=None, because=None):
    assert 1 == (new is not None) + (because is not None)
    what = callable_type(old)
    header = "%s %s is deprecated." % (what, callable_name(old))
    if new is not None:
        warnings.warn("%s Use %s %s instead." %
                      (header, callable_type(new), callable_name(new)),
                      DeprecationWarning, stacklevel=3)
    elif because is not None:
        warnings.warn("%s %s" % (header, because),
                      DeprecationWarning, stacklevel=3)

def deprecated_usage(callable, msg):
    warnings.warn("Deprecated usage of %s. %s"
                  % (callable_name(callable), msg),
                  DeprecationWarning, stacklevel=3)
