# arch-tag: david@allouche.net - 2003-11-24 15:35:19 050584000
# Copyright (C) 2003 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""High level bindings for the Arch revision control system

Archive Namespace Class Hierarchy
---------------------------------

:group Namespace Classes: ArchiveLocation, Archive, Category, Branch, Version,
    Revision

:group Abstract Namespace Classes: NamespaceObject, Setupable, Package,
    CategoryIterable, BranchIterable, VersionIterable, RevisionIterable,
    ArchiveItem, CategoryItem, BranchItem, VersionItem

:group Archive-Related Classes: RevisionFile, NameParser

The `Archive`, `Category`, `Branch`, `Version` and `Revision` classes
model the Arch namespace. Namespace objects can be created without the
corresponding archive structure being available.

Since they form a hierarchy of containers with shared methods and
properties in both directions, but do not have any subclass
relationship, they are defined using a collection of mixin classes.

The `RevisionIterable`, `VersionIterable`, `BranchIterable` and
`CategoryIterable` classes define the features which are inherited by
enclosing archive containers. Many methods in that hierarchy are
defined abstract (they raise UnimplementedError). They are always
overriden and are required to prevent legitimate PyChecker warnings.

The `ArchiveItem`, `CategoryItem`, `BranchItem` and `VersionItem`
classes provides features which are inherited by enclosed archive
items. The `NamespaceObject`, `Setupable` and `Package` classes
provide miscellaneous features and define aspects which do not fit
within the rest of the hierarchy.

:group Source Tree Classes: SourceTree, ForeignTree, ArchSourceTree,
    LibraryTree, WorkingTree

:group Changeset and Log Classes: Changeset, Patchlog, LogMessage

:group Incremental Ouput: ChangesetCreation, ChangesetApplication,
    Chatter, TreeChange, FileAddition, FileDeletion, FileModification,
    FilePermissionsChange, FileRename, SymlinkModification,
    MergeOutcome, PatchConflict

:group Archive Functions: archives, iter_archives, make_archive,
    register_archive, get, get_patch, make_continuation

:group Source Tree Functions: init_tree, in_source_tree, tree_root

:group User Functions: default_archive, my_id, set_my_id

:group Changeset Generation Functions: changeset, delta, iter_delta

:group Pika Escaping Functions: name_escape, name_unescape

:group Revision Library Functions: register_revision_library,
    unregister_revision_library, iter_revision_libraries, library_archives,
    iter_library_archives

:group Incremental Output Functions: classify_chatter,
    classify_changeset_creation, classify_changeset_application

:group Obsolete Utility Functions: filter_archive_logs, filter_revisions,
    grep_summary, grep_summary_interactive, last_revision, map_name_id,
    revision_which_created, revisions_merging, suspected_move, temphack

:var backend: Backend controller.

    This object is used to configure the backend system: name of the
    executable, process handling strategy and command-line logging.

:type backend: `backends.commandline.CommandLineBackend`
"""

__all__ = [
    'errors',
    'pathname',
    'compat',
    'backends',
    # Deprecated modules
    'util',
    ]

from _output import *
from _builtin import *
from _escaping import *
from _location import *

def _import_builtin():
    # Along with the monkey-patch in mypydoc, this causes Epydoc to
    # document the public interface as part of the pybaz package
    # instead of the pybaz._builtin module (where it is actually
    # defined). This is provides no useful runtime functionality, and
    # is purely needed for documentation purposes.
    import _builtin
    import _output
    import _escaping
    all = _builtin.__all__
    all += _output.__all__
    all += _escaping.__all__
    all += _location.__all__
    for name in all:
        getattr(_builtin, name).__module__ = __name__
    __all__.extend(all)
    del _builtin.public

_import_builtin()
# Since this function is only useful for documentation, delete it so
# Epydoc will not show it in the private listing. There is really
# little point in documenting documentation-related stuff...
del _import_builtin

# Deprecated public stuff from before we used __all__.
from errors import ExecProblem
__all__.extend(['ExecProblem'])
from pathname import PathName, FileName, DirName
__all__.extend(['PathName', 'FileName', 'DirName'])
