package GCGraphicComponents;

###################################################
#
#  Copyright 2005 Tian
#
#  This file is part of GCfilms.
#
#  GCfilms is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCfilms is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCfilms; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################
use utf8;
use Gtk2;
use GCBorrowings;

use strict;

{
    package GCMovieImage;
    
    use base "Gtk2::Image";
    use File::Spec;
    use File::Basename;
    
    sub new
    {
        my ($proto, $options, $defaultImage, $fixedSize, $width, $height) = @_;
        my $class = ref($proto) || $proto;
        my $self  = Gtk2::Image->new_from_file($defaultImage);
        $self->{options} = $options;
        $self->{defaultImage} = $defaultImage;
        $self->{displayedImage} = '';
        $self->{fixedSize} = $fixedSize;
        bless ($self, $class);
        if ($width && $height)
        {
            $self->{width} = $width;
            $self->{height} = $height;
        }
        else
        {
            $self->{width} = 120;
            $self->{height} = 160;
        }
        return $self;
    }
    
    sub setImage
    {
        my ($self, $displayedImage) = @_;
        $self->{displayedImage} = $displayedImage;
        
        $displayedImage = GCUtils::getDisplayedImage($displayedImage,
                                                     $self->{defaultImage},
                                                     $self->{options}->file);
        my $pixbuf;
        eval
        {
            $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($displayedImage);
        };
        if ($@)
        {
            $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($self->{defaultImage});
        }
        $pixbuf = GCUtils::scaleMaxPixbuf($pixbuf, $self->{width}, $self->{height});
        $self->set_from_pixbuf($pixbuf);
        $self->set_size_request($self->{width}, $self->{height}) if $self->{fixedSize};
    }
    
    sub getImage
    {
        my $self = shift;
        return $self->{displayedImage};
    }
}

{
    package GCColourLabel;
    
    use base "Gtk2::EventBox";

    sub new
    {
        my ($proto, $color) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;

        bless ($self, $class);
    
        $self->modify_bg('normal', $color);
        $self->{label} = Gtk2::Label->new;
        $self->{label}->show;
        $self->{hboxFill} = new Gtk2::HBox(0,0);
        $self->{hboxFill}->pack_start($self->{label},1,1,0);
        $self->add($self->{hboxFill});

        return $self;
    }

    sub set_markup
    {
        my ($self, $text) = @_;
        
        $text =~ s/&/&amp;/g;
        $self->{label}->set_markup($text);
    }

    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        $self->{label}->$name(@_);
    }
}

{
    package GCDialogHeader;
    
    use base "Gtk2::HBox";
    
    sub new
    {
        my ($proto, $text, $imageStock, $logosDir) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;

        bless ($self, $class);
        
        $self->{label} = new Gtk2::Label;
        $self->{label}->set_markup("<span size='large' weight='bold'>$text</span>");
        $self->{label}->set_alignment(0,0.5);
        
        if (-f $logosDir.$imageStock.'.png')
        {
            $self->{image} = Gtk2::Image->new_from_file($logosDir.$imageStock.'.png');
            $self->pack_start($self->{image},0,1,5);
        }
        
        $self->pack_start($self->{label},0,1,5);
        
        return $self;
    }
}

{
    package GCImageBox;
    
    use base "Gtk2::VBox";
    
    sub new_from_file
    {
        my ($proto, $imageFile, $label) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new();

        bless ($self, $class);

        my $image = Gtk2::Image->new_from_file($imageFile);
        $self->init($image, $label);
        
        return $self;
    }
    sub new_from_stock
    {
        my ($proto, $stockId, $label) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new();

        bless ($self, $class);

        my $image = Gtk2::Image->new_from_stock($stockId, 'large-toolbar');
        $self->init($image, $label);
         
        return $self;        
    }
    
    sub init
    {
        my ($self, $image, $label) = @_;
        
        $self->{label} = new Gtk2::Label($label);
        
        $self->pack_start($image, 0, 0, 0);
        $self->pack_start($self->{label}, 0, 0, 0);
        
        $self->show_all;
    }
}

{
    package GCAgeChooser;
    use base "Gtk2::OptionMenu";
    
    sub get_text
    {
        my $self = shift;

        my $result = -1;
        my $strValue = $self->child->get_label if $self->child;

        $result = 0 if $strValue eq '';
        $result = 1 if $strValue eq $self->{unknownLabel};
        $result = 2 if $strValue eq $self->{noneLabel};
        $result = 5 if $strValue eq $self->{parentLabel};
        $result = $strValue if $result == -1;

        return $result;
    }

    sub set_text
    {
        my ($self, $value) = @_;

        $value = 0 if !$value;

        $self->{mainMenu}->set_active($value) if $value < 2;
        $self->{mainMenu}->set_active(2) if ($value > 1) && ($value < 5);
        $self->{mainMenu}->set_active(3) if ($value > 4) && ($value < 10);
        if ($value > 9)
        {
            my $idx = $value - 5;
            $self->{mainMenu}->set_active($idx);
        }
        $self->remove_menu;
        $self->set_menu($self->{mainMenu});
    }

    sub set_value
    {
        my ($self, $value) = @_;

        $self->set_text($value);
    }

    sub setEditable
    {
        my ($self, $editable) = @_;
        
        $self->set_sensitive($editable);
    }
    
    sub new
    {
        my ($proto, $lang, $type, $width) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;

        $self->{lang} = $lang;
        
        if ($type eq 'public')
        {
            $self->{unknownLabel} = '';
            $self->{noneLabel} = $lang->{MenuAgeNone};
            $self->{parentLabel} = $lang->{PanelAgeParent};
        }
        else
        {
            $self->{unknownLabel} = $lang->{PanelAgeUnknown};
            $self->{noneLabel} = $lang->{PanelAgeNone};
            $self->{parentLabel} = $lang->{PanelAgeParent};
        }

        bless ($self, $class);

		$self->{mainMenu} = new Gtk2::Menu();
		my $emptyItem = Gtk2::MenuItem->new_with_label('');
        $self->{mainMenu}->append($emptyItem);
        my $unknownItem = Gtk2::MenuItem->new_with_label($self->{unknownLabel});
        $self->{mainMenu}->append($unknownItem) if $self->{unknownLabel};
        $self->{mainMenu}->append(Gtk2::SeparatorMenuItem->new) if ! $self->{unknownLabel};
        my $noneItem = Gtk2::MenuItem->new_with_label($self->{noneLabel});
        $self->{mainMenu}->append($noneItem);
        my $parentItem = Gtk2::MenuItem->new_with_label($self->{parentLabel});
        $self->{mainMenu}->append($parentItem);
        
        $self->{mainMenu}->append(Gtk2::SeparatorMenuItem->new);
        
		foreach (10..21)
		{
		    my $item = Gtk2::MenuItem->new_with_label($_);
		    $self->{mainMenu}->append($item);
		}

        $self->set_menu($self->{mainMenu});
        #$self->{viewOption}->set_alignment(0.0,0.0);

        
        return $self;
    }
}

{
    package GCAgeChooserSlider;
    use base "Gtk2::HBox";
    
    sub getValue
    {
        my $self = shift;
        return int $self->{ageSlider}->get_value;
    }

    sub setValue
    {
        my ($self, $value) = @_;

        $value = 0 if !$value;
        $self->{ageSlider}->set_value($value);
        $self->setAgeLabel;
    }

    sub setAgeLabel
    {
        my $self = shift;
        my $value = int $self->{ageSlider}->get_value;
        if ($value < 2)
        {
            $value = $self->{unknownLabel};
        }
        elsif ($value < 5)
        {
            $value = $self->{noneLabel};
        }
        elsif ($value < 8)
        {
            $value = $self->{parentLabel};
        }
        $self->{ageValue}->set_label($value);
    }
    
    sub setEditable
    {
        my ($self, $editable) = @_;
        
        $self->{ageValue}->set_sensitive($editable);
        $self->{ageSlider}->set_sensitive($editable);
    }
    
    sub new
    {
        my ($proto, $lang, $type, $width) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new(0,0);

        $self->{lang} = $lang;
        
        if ($type eq 'public')
        {
            $self->{unknownLabel} = $lang->{SearchUndef};
            $self->{noneLabel} = $lang->{MenuAgeNone};
            $self->{parentLabel} = $lang->{PanelAgeParent};
        }
        else
        {
            $self->{unknownLabel} = $lang->{PanelAgeUnknown};
            $self->{noneLabel} = $lang->{PanelAgeNone};
            $self->{parentLabel} = $lang->{PanelAgeParent};
        }

        bless ($self, $class);

        $self->{ageValue} = Gtk2::Label->new;
        $self->{ageValue}->set_size_request(100, 1); 
        $self->pack_start($self->{ageValue},0,0,5);
        my $adj = Gtk2::Adjustment->new(0.0, 0.0, 26.0, 1.0, 5.0, 1.0);
        $self->{ageSlider} = Gtk2::HScale->new($adj);
        $self->{ageSlider}->signal_connect('value_changed' => sub {
            $self->setAgeLabel;
        });
        $self->{ageSlider}->set_size_request($width, 1); 
        $self->{ageSlider}->set_draw_value(0);
        $self->pack_start($self->{ageSlider},0,0,0);
        
        return $self;
    }
}


{
    package GCMenuBar;
    use base "Gtk2::MenuBar";

    use GCExport;
    use GCImport;

    sub addHistoryMenu
    {
        my $self = shift;
        my $parent = $self->{parent};

        $parent->{options}->historysize (5) if $parent->{options}->historysize eq "";
        $parent->{options}->history ("none|" x ($parent->{options}->historysize - 1) . "none") if $parent->{options}->history eq "";

        my @tmpHistory = split(/\|/, $parent->{options}->history);
        my $current_size = scalar @tmpHistory;
        if ($current_size < $parent->{options}->historysize)
        {
            $parent->{options}->history ($parent->{options}->history . "|none" x ($parent->{options}->historysize - $current_size));
        }
        elsif ($current_size > $parent->{options}->historysize)
        {
            my $size = $parent->{options}->historysize - 1;
            (my $history = $parent->{options}->history) =~ s/^((?:[^\|]*\|){$size})?([^\|]*)\|.*$/$1$2/;
            $parent->{options}->history ($history);
        }

        my @history_array = split /\|/, $parent->{options}->history;
        foreach my $filename(@history_array)
        {
            next if $filename eq "none";
            next if $filename eq "";
            my $item = Gtk2::MenuItem->new_with_label($filename);
            $item->signal_connect ("activate" => sub {
						$parent->openMovie($filename);
					});
            $self->{menuHistory}->append($item);
            $item->show();
        }
        $self->{menuHistoryItem}->set_submenu($self->{menuHistory});
    }

    sub initFilters
    {
        my ($self, $seen, $borrower, $type, $rating, $age) = @_;
		
		$self->{parent}->{menubar}->{deactivateFilters} = 1;
		
		$self->{menuSeen}->set_active($seen + 1);
		$self->{menuSeen}->get_active->set_active(1);

        $self->{menuRating}->set_active(($rating > 0) ? 1 : 0);
		$self->{menuRating}->get_active->set_active(1);

        $self->{menuAge}->set_active(0) if $age < 2;
        $self->{menuAge}->set_active(1) if ($age > 1) && ($age < 5);
        $self->{menuAge}->set_active(2) if ($age > 4) && ($age < 8);
        $self->{menuAge}->set_active(3) if $age > 7;
        $self->{menuAge}->get_active->set_active(1);

        my @children = $self->{menuCategories}->get_children;
        my $idx = 0;
        foreach (@children)
        {
            last if $_->child->get_label eq $type;
            $idx++;
        }
        $self->{menuCategories}->set_active($idx);
        $self->{menuCategories}->get_active->set_active(1);

        $idx = 0;
        $idx = 1 if $borrower eq 'none';
        $self->{menuBorrowing}->set_active($idx);
        $self->{menuBorrowing}->get_active->set_active(1);

		$self->{parent}->{menubar}->{deactivateFilters} = 0;
    }
    
    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new();

        $self->{parent} = $parent;
        $self->{accel} = Gtk2::AccelGroup->new();

        $self->{menuFile} = Gtk2::Menu->new();

        my $fto = Gtk2::TearoffMenuItem->new();
        $self->{menuFile}->append($fto);

        my $newmovie = Gtk2::ImageMenuItem->new_from_stock('gtk-add', $self->{accel});
        $self->{menuFile}->append($newmovie);
        $newmovie->signal_connect ("activate" => sub {
            $self->{parent}->newMovie;
        });


        my $savedb = Gtk2::ImageMenuItem->new_from_stock('gtk-save', $self->{accel});
        $self->{menuFile}->append($savedb);
        $savedb->signal_connect ("activate" => sub {
            $self->{parent}->saveList;
        });

        $self->{menuFile}->append(Gtk2::SeparatorMenuItem->new);
        #my $newdb = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuNewList});
        my $newdb = Gtk2::ImageMenuItem->new_from_stock('gtk-new', $self->{accel});
        $newdb->signal_connect ("activate" => sub {
            $self->{parent}->newList;
        });
        $self->{menuFile}->append($newdb);
        my $saveasdb = Gtk2::ImageMenuItem->new_from_stock('gtk-save-as', $self->{accel});
        $saveasdb->signal_connect ("activate" => sub {
            $self->{parent}->saveAs;
        });
        $self->{menuFile}->append($saveasdb);
        my $opendb = Gtk2::ImageMenuItem->new_from_stock('gtk-open', $self->{accel});
        $opendb->signal_connect ("activate" => sub {
            $self->{parent}->openList;
        });
        $self->{menuFile}->append($opendb);

       	$self->{menuHistoryItem} = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuHistory});
        $self->{menuHistory} = Gtk2::Menu->new();
    		addHistoryMenu($self);
        $self->{menuFile}->append($self->{menuHistoryItem});

        $self->{playSeparator} = Gtk2::SeparatorMenuItem->new;
        $self->{menuFile}->append($self->{playSeparator});

        $self->{play} = Gtk2::ImageMenuItem->new_from_stock('gtk-media-play', $self->{accel});
        $self->{play}->signal_connect ("activate" => sub {
            $self->{parent}->playTrailer;
        });
        $self->{menuFile}->append($self->{play});
        
        $self->{menuFile}->append(Gtk2::SeparatorMenuItem->new);

        my $lend = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuLend});
        $lend->signal_connect ("activate" => sub {
            $self->{parent}->showBorrowed;
        });
        $self->{menuFile}->append($lend);
                
        #my $importItem = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuImport});
        my $importItem = Gtk2::ImageMenuItem->new_from_stock('gtk-convert', $self->{accel});
        $self->{menuImport} = Gtk2::Menu->new();        
        foreach my $importer(@GCImport::importersArray)
        {
            my $item = Gtk2::MenuItem->new_with_mnemonic($importer->getName($parent));
            $item->signal_connect ("activate" => sub {
                    $self->{parent}->import($importer);
                });
            $self->{menuImport}->append($item);
        }        
        $importItem->set_submenu($self->{menuImport});
        $self->{menuFile}->append($importItem);        

        #my $exportItem = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuExport});
        my $exportItem = Gtk2::ImageMenuItem->new_from_stock('gtk-revert-to-saved', $self->{accel});
        $self->{menuExport} = Gtk2::Menu->new();        
        foreach my $exporter(@GCExport::exportersArray)
        {
            my $item = Gtk2::MenuItem->new_with_mnemonic($exporter->getName);
            $item->signal_connect ("activate" => sub {
                    $self->{parent}->export($exporter);
                });
            $self->{menuExport}->append($item);
        }        
        $exportItem->set_submenu($self->{menuExport});
        $self->{menuFile}->append($exportItem);        

        $self->{menuFile}->append(Gtk2::SeparatorMenuItem->new);

        my $leave = Gtk2::ImageMenuItem->new_from_stock('gtk-quit', $self->{accel});
        $self->{menuFile}->append($leave);
        $leave->signal_connect("activate" ,sub {
            $self->{parent}->leave;
        });

        my $fileitem = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuMovies});
        $fileitem->set_submenu($self->{menuFile});

        $self->append($fileitem);

		#Edit menu
		#Container creation
		$self->{menuEdit} = Gtk2::Menu->new();
		my $eto = Gtk2::TearoffMenuItem->new();
        $self->{menuEdit}->append($eto);

		my $duplicateItem = Gtk2::ImageMenuItem->new_from_stock('gtk-dnd', $self->{accel});
        $duplicateItem->signal_connect("activate" , sub {
            $self->{parent}->duplicateMovie;
        });
        $self->{menuEdit}->append($duplicateItem);

		my $deleteCurrentItem = Gtk2::ImageMenuItem->new_from_stock('gtk-delete', $self->{accel});
        $deleteCurrentItem->signal_connect("activate" , sub {
            $self->{parent}->deleteCurrentMovie;
        });
        $self->{menuEdit}->append($deleteCurrentItem);

		my $lockMoviesItem = Gtk2::CheckMenuItem->new_with_mnemonic($parent->{lang}->{MenuEditLockMovies});
		$self->{menuEdit}->append($lockMoviesItem);
	 	$lockMoviesItem->set_active($parent->{options}->lockPanel);
 		$lockMoviesItem->signal_connect("activate" , sub {
						my $parent = $self;
						my $self = shift;
						$parent->{parent}->{panel}->changeState($parent->{parent}->{panel}, $self->get_active());
						$parent->{parent}->{options}->lockPanel($self->get_active());
						$self->toggled();
			},$lockMoviesItem);
		$self->{menuEditLockMoviesItem}=$lockMoviesItem;
		$lockMoviesItem->{parent}=$self->{menuEdit};

		#L'entête du menu
		my $edititem = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuEdit});
        $edititem->set_submenu($self->{menuEdit});
        $self->append($edititem);
		## fin du menu Edition

        $self->{menuDisplay} = Gtk2::Menu->new();

        my $dto = Gtk2::TearoffMenuItem->new();
        $self->{menuDisplay}->append($dto);

        my $allMovies = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuViewAllMovies});
        $allMovies->signal_connect("activate" , sub {
                $self->selectAll;
        });
        $self->{menuDisplay}->append($allMovies);

        my $search = Gtk2::ImageMenuItem->new_from_stock('gtk-find', $self->{accel});
        $search->signal_connect("activate" , sub {
                $self->{parent}->search('all');
        }); 
        $self->{menuDisplay}->append($search);


        $self->{menuDisplay}->append(new Gtk2::SeparatorMenuItem);
        
        $self->{menuSeen} = Gtk2::Menu->new();
        my $allSeen = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$parent->{lang}->{MenuAllMovies});
        $self->{menuSeen}->append($allSeen);
        $allSeen->signal_connect("activate" ,\&on_seenChange, $self);
        $self->{seenGroup} = $allSeen->get_group();
        my $notseen = Gtk2::RadioMenuItem->new_with_mnemonic($self->{seenGroup},$parent->{lang}->{MenuNotSeen});
        $self->{menuSeen}->append($notseen);
        $notseen->signal_connect("activate" ,\&on_seenChange, $self);
        $self->{seenItem} = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuSeenOrNot});
        $self->{seenItem}->set_submenu($self->{menuSeen});
        $self->{menuDisplay}->append($self->{seenItem});

        $self->{menuBorrowing} = Gtk2::Menu->new();
        my $allBorrowed = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$parent->{lang}->{MenuAllMovies});
        $self->{menuBorrowing}->append($allBorrowed);
        $allBorrowed->signal_connect("activate" ,\&on_borrowedChange, $self);
        $self->{borrowedGroup} = $allBorrowed->get_group();
        my $notBorrowed = Gtk2::RadioMenuItem->new_with_mnemonic($self->{borrowedGroup},$parent->{lang}->{MenuNotBorrowed});
        $self->{menuBorrowing}->append($notBorrowed);
        $notBorrowed->signal_connect("activate" ,\&on_borrowedChange, $self);
        my $isBorrowed = Gtk2::RadioMenuItem->new_with_mnemonic($self->{borrowedGroup},$parent->{lang}->{MenuBorrowed});
        $self->{menuBorrowing}->append($isBorrowed);
        $isBorrowed->signal_connect("activate" ,\&on_borrowedChange, $self);
        $self->{borrowedItem} = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuBorrowing});
        $self->{borrowedItem}->set_submenu($self->{menuBorrowing});
        $self->{menuDisplay}->append($self->{borrowedItem});

        $self->{menuCategories} = Gtk2::Menu->new();
        my $allCat = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$parent->{lang}->{MenuAllCategories});
        $self->{menuCategories}->append($allCat);
        $allCat->signal_connect("activate" ,\&on_catChange, $self);
        $self->{catGroup} = $allCat->get_group();
        $self->{catitem} = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuCategories});
        $self->{catitem}->set_submenu($self->{menuCategories});
        $self->{menuDisplay}->append($self->{catitem});

        $self->{menuAge} = Gtk2::Menu->new();
        my $allMoviesAge = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$parent->{lang}->{MenuAllMovies});
        $self->{menuAge}->append($allMoviesAge);
        $allMoviesAge->signal_connect("activate" ,\&on_ageChange, $self);
        $self->{ageGroup} = $allMoviesAge->get_group();
        my $noneAge = Gtk2::RadioMenuItem->new_with_mnemonic($self->{ageGroup},$parent->{lang}->{MenuAgeNone});
        $self->{menuAge}->append($noneAge);
        $noneAge->signal_connect("activate" ,\&on_ageChange, $self);        
        my $parentAge = Gtk2::RadioMenuItem->new_with_mnemonic($self->{ageGroup},$parent->{lang}->{MenuAgeParent});
        $self->{menuAge}->append($parentAge);
        $parentAge->signal_connect("activate" ,\&on_ageChange, $self);        
        my $chooseAge = Gtk2::RadioMenuItem->new_with_mnemonic($self->{ageGroup},$parent->{lang}->{MenuAgeLess});
        $self->{menuAge}->append($chooseAge);
        $chooseAge->signal_connect("activate" ,\&on_ageChange, $self);
        $self->{ageItem} = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuAge});
        $self->{ageItem}->set_submenu($self->{menuAge});
        $self->{menuDisplay}->append($self->{ageItem});

        $self->{menuRating} = Gtk2::Menu->new();
        my $allRating = Gtk2::RadioMenuItem->new_with_mnemonic(undef,$parent->{lang}->{MenuAllRating});
        $self->{menuRating}->append($allRating);
        $allRating->signal_connect("activate" ,\&on_ratingChange, $self);
        $self->{ratingGroup} = $allRating->get_group();
        my $chooseRating = Gtk2::RadioMenuItem->new_with_mnemonic($self->{ratingGroup},$parent->{lang}->{MenuChooseRating});
        $self->{menuRating}->append($chooseRating);
        $chooseRating->signal_connect("activate" ,\&on_ratingChange, $self);
        $self->{ratingItem} = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuRating});
        $self->{ratingItem}->set_submenu($self->{menuRating});
        $self->{menuDisplay}->append($self->{ratingItem});

        $self->{displayItem} = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuDisplay});
        $self->{displayItem}->set_submenu($self->{menuDisplay});

        $self->append($self->{displayItem});

        $self->{menuConfig} = Gtk2::Menu->new();
        my $cto = Gtk2::TearoffMenuItem->new();
        $self->{menuConfig}->append($cto);
        my $options = Gtk2::ImageMenuItem->new_from_stock('gtk-preferences',$self->{accel});
        $options->signal_connect("activate" , sub {
            my ($widget, $window) = @_;
            $window->options;
        }, $self->{parent});
        $self->{menuConfig}->append($options);

        my $displayOptions = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuDisplayOptions});
        $displayOptions->signal_connect ("activate" => sub {
            $self->{parent}->displayOptions;
        });
        $self->{menuConfig}->append($displayOptions);

        my $borrowers = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuBorrowers});
        $borrowers->signal_connect ("activate" => sub {
            $self->{parent}->borrowers;
        });
        $self->{menuConfig}->append($borrowers);

        my $genresConversion = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuGenresConversion});
        $genresConversion->signal_connect ("activate" => sub {
            $self->{parent}->genresConversion;
        });
        $self->{menuConfig}->append($genresConversion);

        my $rename = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuRename});
        $rename->signal_connect ("activate" => sub {
            $self->{parent}->rename;
        });
        $self->{menuConfig}->append($rename);
        
        my $configitem = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuConfiguration});
        $configitem->set_submenu($self->{menuConfig});
        $self->append($configitem);

        $self->{menuHelp} = Gtk2::Menu->new();
        my $hto = Gtk2::TearoffMenuItem->new();
        $self->{menuHelp}->append($hto);
        my $help = Gtk2::ImageMenuItem->new_from_stock('gtk-help',$self->{accel});
        $help->signal_connect("activate" , sub {
            my ($widget, $window) = @_;
            $window->help;
        }, $self->{parent});
        $self->{menuHelp}->append($help);
        my $depend = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{InstallDependencies});
        $depend->signal_connect ("activate" => sub {
            $self->{parent}->showDependencies;
        });
        $self->{menuHelp}->append($depend);
        my $about = Gtk2::ImageMenuItem->new_from_stock('gtk-about',$self->{accel});
        $about->signal_connect("activate" , sub {
            my ($widget, $window) = @_;
            $window->about;
        }, $self->{parent});
        $self->{menuHelp}->append($about);


        my $helpitem = Gtk2::MenuItem->new_with_mnemonic($parent->{lang}->{MenuHelp});
        $helpitem->set_submenu($self->{menuHelp});
        $self->append($helpitem);

        $parent->add_accel_group($self->{accel});

        $self->{ratingAsk} = new GCRatingDialog($parent);
        $self->{ageAsk} = new GCAgeDialog($parent);

        $self->{deactivateFilters} = 0;
        $self->{contextUpdating} = 0;

        bless ($self, $class);
        return $self;
    }

    sub on_seenChange
    {
        my ($widget, $self) = @_;
        

        
        if ($widget->get_active)
        {
            my $value = $widget->child->get_label();
            my $filter = 'seen';
            my $parameter = '';
            
            if ($value eq $self->{parent}->{lang}->{MenuAllMovies})
            {
                $parameter = 'all';
            }
            else
            {
                $parameter = 'notseen';
            }

            if (! $self->{contextUpdating})
            {
                $self->{contextUpdating} = 1;
                
                $self->{parent}->{contextMenuSeen}->set_active(($parameter eq 'all') ? 0 : 1);
                $self->{parent}->{contextMenuSeen}->get_active->set_active(1);
                
                $self->{contextUpdating} = 0;
            }
            
            return if $self->{deactivateFilters};

            $self->{parent}->setFilter($filter, $parameter);
        }
    }
    sub on_catChange
    {
        my ($widget, $self) = @_;
        
        if ($widget->get_active)
        {
            my $value = $widget->child->get_label();
            my $filter = 'type';
            my $parameter = '';
            
            if ($value eq $self->{parent}->{lang}->{MenuAllCategories})
            {
                $parameter = 'all';
            }
            else
            {
                $parameter = $value;
            }
            
            if (! $self->{contextUpdating})
            {
                $self->{contextUpdating} = 1;
                
                my $pos = 0;
                $pos = GCUtils::findPosition($value,$self->{menuCategories}) - 1 if $parameter ne 'all';
                
                $self->{parent}->{contextMenuCategories}->set_active($pos);
                $self->{parent}->{contextMenuCategories}->get_active->set_active(1);
                
                $self->{contextUpdating} = 0;
            }
            
            return if $self->{deactivateFilters};

            $self->{parent}->setFilter($filter, $parameter);
        }
    }
    sub on_ratingChange
    {
        my ($widget, $self) = @_;

        if ($widget->get_active)
        {
            my $value = $widget->child->get_label();
            my $filter = 'rating';
            my $parameter = '';
            
            if ($value eq $self->{parent}->{lang}->{MenuAllRating})
            {
                $parameter = 'all';
            }
            else
            {
                $parameter = $self->{ratingAsk}->getUserValue;
                return if ($parameter == -1);
            }
            if (! $self->{contextUpdating})
            {
                $self->{contextUpdating} = 1;
                
                $self->{parent}->{contextMenuRating}->set_active(($parameter eq 'all') ? 0 : 1);
                $self->{parent}->{contextMenuRating}->get_active->set_active(1);
                
                $self->{contextUpdating} = 0;
            }
            
            return if $self->{deactivateFilters};        

            $self->{parent}->setFilter($filter, $parameter);
        }
    }
    
    sub on_ageChange
    {
        my ($widget, $self) = @_;

        if ($widget->get_active)
        {
            my $value = $widget->child->get_label();
            my $filter = 'age';
            my $parameter = '';
            

            
            if ($value eq $self->{parent}->{lang}->{MenuAllMovies})
            {
                $parameter = 'all';
            }
            else
            {
                $parameter = 4 if $value eq $self->{parent}->{lang}->{MenuAgeNone};
                $parameter = 7 if $value eq $self->{parent}->{lang}->{MenuAgeParent};
                if (! $parameter)
                {
                    $parameter = $self->{ageAsk}->getUserValue if ! $self->{deactivateFilters};
                    $parameter = 8 if $self->{deactivateFilters};
                }
                return if ($parameter == -1);
            }
            
            if (! $self->{contextUpdating})
            {
                $self->{contextUpdating} = 1;
                
                $self->{parent}->{contextMenuAge}->set_active(0) if $parameter eq 'all';
                $self->{parent}->{contextMenuAge}->set_active(1) if $parameter == 4;
                $self->{parent}->{contextMenuAge}->set_active(2) if $parameter == 7;
                $self->{parent}->{contextMenuAge}->set_active(3) if $parameter > 7;
                $self->{parent}->{contextMenuAge}->get_active->set_active(1);
                
                $self->{contextUpdating} = 0;
            }
            
            return if $self->{deactivateFilters};        

            $self->{parent}->setFilter($filter, $parameter);            
        }
    }
    
    sub on_borrowedChange
    {
        my ($widget, $self) = @_;

        if ($widget->get_active)
        {
            my $value = $widget->child->get_label();
            my $filter = 'borrowing';
            my $parameter = '';
            
            if ($value eq $self->{parent}->{lang}->{MenuAllMovies})
            {
                $parameter = 'all';
            }
            elsif ($value eq $self->{parent}->{lang}->{MenuBorrowed})
            {
                $parameter = 'yes';
            }
            else
            {
                $parameter = 'no';
            }
            if (! $self->{contextUpdating})
            {
                $self->{contextUpdating} = 1;
                
                my $pos = 0;
                $pos = 1 if $parameter eq 'no';
                $pos = 2 if $parameter eq 'yes';
                $self->{parent}->{contextMenuBorrowing}->set_active($pos);
                $self->{parent}->{contextMenuBorrowing}->get_active->set_active(1);
                
                $self->{contextUpdating} = 0;
            }
            
            return if $self->{deactivateFilters};        

            $self->{parent}->setFilter($filter, $parameter);
        }
    }

    sub setTypes
    {
        my ($self, $values, $menu) = @_;

        my $default = 1;
        $default = 0  if $menu;

        $menu = $self->{menuCategories} if $default;
        my $catGroup = $self->{catGroup};
        $catGroup = $menu->{group} if ! $default;

        my @addValues = @$values;
        my @initialValues = $menu->get_children;
        my $init;
        if ((scalar @initialValues) > 1)
        {
            $init = 0;
            foreach (@initialValues)
            {
                my $label = $_->child->get_label;
                next if $label eq $self->{parent}->{lang}->{MenuAllCategories};
                my $idx = GCUtils::inArray($label, @addValues);
    
                if ($idx == undef)
                {
                    $menu->remove($_);
                }
                else
                {
                    splice @addValues, $idx, 1;
                }
            }
        }
        else
        {
            $init = 1;
            use locale;
            @addValues = sort @addValues;
        }
        my @itemsArray;
        my $pos = 0;
        foreach my $cat(@addValues)
        {
            next if !$cat;
            my $item = Gtk2::RadioMenuItem->new_with_mnemonic($catGroup, $cat);
            if ($init)
            {
                $menu->append($item);
                $pos++;
            }
            else
            {
                $pos = GCUtils::findPosition($cat, $menu);
                $menu->insert($item, $pos);
            }
            $item->signal_connect("activate", \&on_catChange, $self) if $default;
            splice @itemsArray, $pos - 1, 0, $item if ! $default;
        }

        if (! $default)
        {
            my $pos = 1;
            foreach my $item(@itemsArray)
            {
                $item->signal_connect("activate", sub {
                    shift;
                    my $idx = shift;
                    $self->{parent}->contextDisplayChange(undef, 'menuCategories', $idx);
                }, $pos);
                $pos++;
            }
        }

        $menu->hide_all;
        $menu->show_all;

        return;
    }
    
    sub selectAll
    {
        my $self = shift;
        
        $self->{parent}->removeSearch;
        $self->{parent}->blockListUpdates(1);
        $self->{menuSeen}->set_active(0);
        $self->{menuSeen}->get_active->set_active(1);
        $self->{menuCategories}->set_active(0);
        $self->{menuCategories}->get_active->set_active(1);
        $self->{menuRating}->set_active(0);
        $self->{menuRating}->get_active->set_active(1);
        $self->{menuAge}->set_active(0);
        $self->{menuAge}->get_active->set_active(1);
        $self->{menuBorrowing}->set_active(0);
        $self->{menuBorrowing}->get_active->set_active(1);
        $self->{parent}->blockListUpdates(0);
    }
}

{
    package GCToolBar;
    use base "Gtk2::Toolbar";

    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new();

        bless($self, $class);

        $self->{parent} = $parent;
        $self->{lang} = $parent->{lang};
        
        $self->set_name('GCToolbar');

        $self->insert_stock("gtk-add" ,$self->{lang}->{NewMovieTooltip}, undef, sub {$parent->newMovie}, undef, -1);
        $self->insert_stock("gtk-save" ,$self->{lang}->{SaveListTooltip}, undef, sub {$parent->saveList}, undef, -1);
        $self->insert_stock("gtk-preferences" ,$self->{lang}->{PreferencesTooltip}, undef, sub {$parent->options}, undef, -1);
        $self->append_space;
        $self->insert_stock("gtk-find" ,$self->{lang}->{SearchTooltip}, undef, sub {$parent->search('all')}, undef, -1);
        $self->insert_stock("gtk-refresh" ,$self->{lang}->{ToolbarAllTooltip}, undef, sub {$parent->allMovies}, undef, -1);
        $self->insert_stock("gtk-execute" ,$self->{lang}->{RandomTooltip}, undef, sub {$parent->randomMovie}, undef, -1);
        $self->{play} = $self->insert_stock("gtk-media-play" ,$self->{lang}->{PlayTooltip}, undef, sub {$parent->playTrailer}, undef, -1);
        $self->append_space;

        return $self;
    }
    
}

1;
