#############################################################################
#
# Copyright (c) 2002 Ingeniweb SARL
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.0 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE
#
##############################################################################

"""
ZAttachmentAttribute product
"""


from Acquisition import Implicit
from Globals import Persistent
from Globals import MessageDialog, DTMLFile      # fakes a method from a DTML file
from AccessControl import ClassSecurityInfo

import tempfile
import string
import cgi
import re


from global_symbols import *

INVALID_VALUE = "******** INVALID VALUE *********"

# HTML Strippping variables
header_re = re.compile("(.*?<body.*?>)(.*?)(</body>.*)", re.S | re.I)


class ZAbstractAttachment(Implicit, Persistent):
    """
    ZAbstractAttachment => abstract class that must be derived to handle attachments of a particular type.
    When deriving, ensure to call the __init__ method so that class will be instanciated properly.
    Then you must register your derived class using ZAttachmentRegistry facility.
    
    CODING INFORMATION : If you create additional methods or properties in ZAbstractAttachment that must
    be derived for the class to work, put them in the __must_derive__ tuple so that they will be checked
    at class registration. This will ensure better coding quality.

    Please note that ZAbstractAttachment-dervied objects are instanciated every time a file is uploaded.
    So you're guaranteed that the indexed file won't change during the class's lifetime.

    You can store additional files inside a ZAA with the addRelatedFile() method.
    If you want to get it back, use getRelatedFile() with the file's identifier.
    """
    # List of methods and properties that must be derived in subclasses
    __must_derive__ = (
        "icon_file",                            # (string) of the icon name IN A PLONE SKIN.
        "small_icon_file",                      # (string) of the SMALL icon name IN A PLONE SKIN.
        "content_types",                        # List (of strings) of content_types supported by the class
        "indexAttachment"  ,                    # (method returning a String) indexing the file
        "getContent",                           # (method returning a loooooong string) of the attachment data
        "isPreviewAvailable",                   # Return true if the current attachment can be previewed as HTML
        "convertPreview",                       # Return the HTML preview, or None if no preview can be generated
        )


    # attachment properties (MUST BE DERIVEd
    icon_file = None                    # Icon file (as an instanciated Image object)
    small_icon_file = None
    content_types = None                # Supported content-types (tuple of strings)

    # Other properties
    content = ''                        # Actual content
    index = ''                          # Index content
    content_type = None                 # Actual content type

    attachment_file = None              # Temporary file name

    # Indexing diagnosign
    indexing_done = 1                   # This is true by default as we assume everything is good. It's


    def getEncoding(self,):
        """
        getEncoding(self,) => return the encoding of the data returned by the converter

        Default is utf-8
        """
        return "utf-8"
    
    def __init__(self, content_type = '', stream = '', converter = None):
        """
        __init__(self, content_type = '', stream = '', converter = None) -> init method

        DO NOT OVERRIDE IT !
        """
        # Store converter info
        self._converter = converter
        
        # Store content_type
        self.content_type = content_type
        
        # Upload the file
        if type(stream) != type(''):
            try:
                stream.seek(0)
            except:
                pass    # Ignore pbs with file types not handeling seek method
            self.content = stream.read()
        else:
            content = stream

        # Index the file
        Log(LOG_DEBUG, len(self.content))
        self.indexing_done = 1
        try:
            self.index = self.indexAttachment()
        except:
            self.index = ''
            self.indexing_done = 0
            Log(LOG_WARNING, "Unable to invoke indexing for '%s' type attachment (see exception below)" % (content_type,))
            LogException()

        # Index words as a list of unique items
        Log(LOG_DEBUG, "Indexing the file")
        words = []
        for w in string.split(self.index):
            stripped = string.lower(string.strip(w))
            if not stripped in words:
                words.append(stripped)
        words.sort
        self.index_list = tuple(words)
        Log(LOG_DEBUG, len(self.index_list))


##    def __del__(self):
##        """Loop on all base classes, and invoke their destructors.
##        Protect against diamond inheritance."""
##        Log(LOG_DEBUG, "We are in __del__")
            
##        # Remove temp file
##        self.deleteAttachmentFile()

##        # We remove the following code because Python 2.3 changed its inheritance mechanism.
##        # So, this code proved to be very unstable with Zope 2.7+.
##        # We suppose the Persistent and Implicit do not have __del__ method, though.
##        # Loop destructors
##        for base in self.__class__.__bases__:
##            # Avoid problems with diamond inheritance.
##            basekey = 'del_' + str(base)
##            if not hasattr(self, basekey):
##                setattr(self, basekey, 1)
##            else:
##                continue
            
##            # Call this base class' destructor if it has one.
##            if hasattr(base, "__del__"):
##                base.__del__(self)


    def hasConverter(self,):
        """hasConverter(self) => None if the current class has no converter defined
        """
        return self._converter

    def getConverterPath(self,):
        """getConverterPath(self,) => Return converter path or raise
        """
        if self._converter:
            return self._converter
        else:
            raise ValueError, "Missing converter for %s" % self.__class__.__name__

    def isIndexed(self,):
        """
        isIndexed(self,) => Return true if the attach. is indexed
        """
        return self.indexing_done
    

    def getIcon(self,):
        """
        getIcon(self,) => return a suitable icon for that type.
        """
        return self.icon_file


    def getSmallIcon(self,):
        """
        getSmallIcon(self,) => return a suitable icon for that type.
        """
        return self.small_icon_file


    def getContent(self,):
        """
        getContent(self,) => return file content as a string
        """
        return self.content


    def getContentType(self,):
        """
        getContentType(self,) => return content type for this file type.
        """
        return self.content_type


    def getIndexableValue(self,):
        """
        getIndexableValue(self,) => (possibliy big) string 
        Return the ZCatalog-indexable string for that type.
        """
        return self.index


    def listIndexableValues(self,):
        """
        listIndexableValues(self,) => list
        return indexable values as a tuple of DISTINCT words, all lowercased.
        """
        ret = self.index_list
        Log(LOG_DEBUG, "10 first words :", ret[:10])
        return ret


    def indexAttachment(self,):
        """
        indexAttachment(self,) => return a string to be considered as the cataloguable value
        """
        raise NotImplementedError, "Must be derived in subclasses."


    def SearchableText(self,):
        """
        SearchableText(self,) => ZCatalog support
        """
        Log(LOG_DEBUG, "SearchableText")
        return string.join(self.listIndexableValues(), ' ')


    #                                                                   #
    #                         HTML PREVIEW SUPPORT                      #
    #                                                                   #

    def getPreview(self,):
        """
        getPreview(self,) => string or None

        Return the HTML preview (generating it if it's not already done) for this attachement.
        If the attachment is not previewable, or if there's a problem in the preview,
        retrun None.

        NOTES :
        
        * We use a html_preview attribute to store the preview text and avoid calling
        the conversion method several times.

        * Is something goes wrong, the html_preview_error stores information about
        what happend when trying to generate the preview (XXX TODO)

        * You may overload this, but then you must call getPreview() by yourself.
        See ZAAPlugins/Image for an example.
        """
        # Check if we can preview
        if not self.isPreviewAvailable():
            # XXX TODO : html_preview_error management
            Log(LOG_DEBUG, "No preview available", )
            self.html_preview = None
            return None

        # Convert if necessary
        if not getattr(self, 'html_preview', None):
            try:
                Log(LOG_DEBUG, "Converting...", )
                self.html_preview = self.convertPreview()
            except:
                # XXX TODO : html_preview_error management
                Log(LOG_DEBUG, "Error while converting", )
                LogException()
                self.html_preview = None

        # Return the actual preview
        return self.html_preview
    

    def getSmallPreview(self,):
        """
        getSmallPreview(self,) => string or None

        Default behaviour : if the preview string is shorter than MAX_PREVIEW_SIZE, return it, else return None.
        You can override this, of course.
        """
        ret = self.preview()
        if not ret:
            return None
        if len(ret) < MAX_PREVIEW_SIZE:
            return ret
        return None


    #                                                                   #
    #                           UTILITY METHODS                         #
    #                                                                   #
    #   Those methods can be called from your products to make your     #
    #   work easier when creating plugins.                              #
    #                                                                   #


    def callConverter(self, program_path = None, arguments = '', stdin = None, report_errors = 1):
        """
        callConverter(self, program_path = None, arguments = '', stdin = None, report_errors = 1) => convert file using program_path with given arguments.
        Return the output stream of the converter program.
        
        if stdin is given, it is feed into the program. Else, it is ignored.
        if report_errors is true, 2> ~/tempfile is appended at the end of the command line
        """
        # Get the converter program if not given
        if not program_path:
            program_path = self.getConverterPath()

        # Open read & write streams
        cmd = "%s %s" % (program_path, arguments,)
        Log(LOG_DEBUG, "Converting file using '%s' program and '%s' arguments" % (program_path, arguments, ))
        idx = ""
        err = ""
        stdout_done = 0
        stderr_done = 0
        if stdin:
            raise NotImplementedError, "implement stdin management"

        # Manage file for error reporting
        if report_errors:
            errfile = tempfile.mktemp()
            cmd = "%s 2> %s" % (cmd, errfile, )
        else:
            errfile = None
        
        # Actually execute command
        errors = ""
        try:
            r = os.popen(cmd, "r")
            idx = r.read()
        finally:
            if errfile:
                try:
                    f = open(errfile, "r")
                except:
                    Log(LOG_NOTICE, "Unable to open error file '%s'" % (errfile, ))
                else:
                    errors = f.read()
                    f.close()
                    os.unlink(errfile)

        # Report errors
        if not idx and not errors:
            raise RuntimeError, "'%s' returned nothing. No error reported by plugin. Indexing cancelled." % (cmd, )
        elif not idx:
            raise RuntimeError, "'%s' returned nothing. Error reported by plugin: '%s'. Indexing cancelled." % (cmd, errors, )
        elif idx and errors:
            Log(LOG_WARNING, "'%s' returned error while indexing: '%s'. Indexing done anyway." % (cmd, errors, ))

        Log(LOG_DEBUG, "Conversion done. Implicitly closing streams.")

        return idx


    def textToHTML(self, text):
        """
        textToHTML(self, text) => string

        Convert a plain-text string into pretty HTML, keeping you away from the need
        to use dirty '<pre>' tags and quoting the string.
        """
        # HTML-Quote the string
        text = cgi.escape(text)

        # Convert double linefeeds into paragraphs
        # XXX TODO

        # Convert double spaces into paragraphs
        text = re.sub("  ", "&nbsp;", text, )

        # Convert simple linefeeds into line breaks
        text = re.sub("\n", "<br />", text, )

        # Return this pretty converted string
        return text

    def stripBody(self, text):
        """
        stripBody(self, text) => string

        Strip a <body> tags in generated HTML
        """
        # XXX TODO !!!
        return text
        
        m = header_re.match(text)       # There's a recursion limit exceded bug with this line => WHY ???
        if m:
            g = m.groups()
            Log(LOG_DEBUG, g[2])
            return g[2]
        
        # No <body> tags match
        Log(LOG_DEBUG, "No body tags match", text)
        return text


    def writeAttachmentFile(self,):
        """
        writeAttachmentFile(self) => file name

        Write attachment into a temporary file and return tmp file's filename.
        This file is removed when the class is deleted.

        The attachment file name is also available as self.attachment_file
        """
        # If the file already exists, delete it
        self.deleteAttachmentFile()
        
        # Write in a temporary file
        fn = tempfile.mktemp()
        self.attachment_file = fn
        f = open(fn, "wb")
        f.write(self.content)
        f.close()
        return fn
        

    def deleteAttachmentFile(self,):
        """
        deleteAttachmentFile(self,) => Remove temporary attachment file. Ignore errors.
        """
        if self.attachment_file:
            if os.path.isfile(self.attachment_file):
                os.unlink(self.attachment_file)
                Log(LOG_DEBUG, "Removed attachment file '%s'" % (self.attachment_file,))
                self.attachment_file = None


    def getAttachmentFileDir(self,):
        """
        getAttachmentFileDir(self,) => string

        Return the directory the temporary file is stored in.
        """
        return tempfile.gettempdir()


    ##                                                                  ##
    ##                     Additional files support                     ##
    ##                                                                  ##

    related_files = {}

    def addRelatedFile(self, name, data, content_type):
        """
        addRelatedFile(self, name, data, content_type) => None

        Use this to add other files inside your attachments. They can
        be later accessed through ZAA.
        """
        self.related_files[name] = {
            "name": name,
            "data": data,
            "content_type": content_type,
            }
        self.related_files = self.related_files         # Persistency insurence


    def clearRelatedFiles(self,):
        """
        clearRelatedFiles(self,) => quite explicit :-)
        """
        self.related_files = {}


    def getRelatedFile(self, name, default = INVALID_VALUE):
        """
        getRelatedFile(self, name[, default]) => dict

        Return the file as it is named, or 'default' if you want
        to prevent rising an exception
        """
        Log(LOG_DEBUG, "name", name, "default", default, "keys", self.related_files.keys())
        if default == INVALID_VALUE:
            return self.related_files.get(name, )
        return self.related_files.get(name, default)



