/*

    xpuyopuyo - pwindow-gtk.c       Copyright(c) 1999 Havoc Pennington
    hp@pobox.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <xpuyopuyo.h>
#include <pconfigm.h>
#include <pwindow-gtk.h>
#include <pdialog-gtk.h>
#include <ptheme-gtk.h>
#include <pkey-gtk.h>
#include <pmenu-gtk.h>
#include <pai-gtk.h>
#include <pfield.h>
#include <ppiece.h>
#include <pmanip.h>
#include <psound.h>
#include <pnet.h>



/***  Pixmap manipulation  ***/



gint pixmap_width(GdkPixmap *p) {
/* pixmap_width */

   gint width;
   
   width = 0;
   gdk_window_get_size(p, &width, NULL);
   return(width);

}


gint pixmap_height(GdkPixmap *p) {
/* pixmap_height */

   gint height;
   
   height = 0;
   gdk_window_get_size(p, NULL, &height);
   return(height);

}



/***   Event handlers   ***/



static gint window_timeout(gpointer data) {
/* window_timeout */

   pwindow_gtk *w = data;
   pwindow *pw = data;
   int redraw = P_REDRAW_NONE;
   
   #if USE_SOUND
   p_sound_update(w->c->sound);
   #endif /* Sound? */

   /* If not paused, advance both players */   
   if(!w->c->paused) {
      redraw = p_game_advance_all(w->c);
      p_window_paint(w->c, pw, redraw);
   } else {
      /* Run while paused code */
      p_game_while_paused(w->c);
   }

   /* Restart game on a game over? */
   if(P_GAME_OVER(w->c) && w->c->restart) {
      p_game_new(w->c);
      p_window_paint(w->c, pw, redraw);
   }

   /* Update the menus */
   p_window_update_menus(w);
      
   return TRUE; /* leave timeout in place */

}


static gint drawing_area_configure(GtkWidget *drawing_area, GdkEventConfigure *event, gpointer data) {
/* drawing_area_configure */

   pwindow_gtk *w = data;
   
   /* Release any existing drawing buffer */
   if (w->buffer != NULL) gdk_pixmap_unref(w->buffer);

   /* Construct new offscreen-drawable */
   w->buffer = gdk_pixmap_new(drawing_area->window,
                              drawing_area->allocation.width,
                              drawing_area->allocation.height,
                              -1 /* Use depth of window */);
   gdk_draw_rectangle (w->buffer,
                       drawing_area->style->black_gc,
                       TRUE   /* Filled, this really ought to be a constant in GDK ... */,
                       0, 0,  /* Destination x, y */
                       drawing_area->allocation.width,
                       drawing_area->allocation.height);

   /* Need to invalidate both playing fields */
   w->c->needredraw = 1;

   return(TRUE);
   
}


static gint drawing_area_expose(GtkWidget *widget, GdkEventExpose *event, gpointer data) {
/* drawing_area_expose */

   pwindow_gtk *w = data;

   g_return_val_if_fail(w->buffer != NULL, FALSE);

   gdk_draw_pixmap(widget->window,                                /* Destination */
                   widget->style->fg_gc[GTK_WIDGET_STATE(widget)],/* What the hell? */
                   w->buffer,                                     /* Source */
                   event->area.x, event->area.y,                  /* src x, y */
                   event->area.x, event->area.y,                  /* Dest x, y */
                   event->area.width, event->area.height);        /* Size */

   /* Start game if this is our first drawing */
   if(!w->exposed && w->ready) {
      gtk_timeout_add(P_SLEEP_TIME, window_timeout, w);
      w->exposed = TRUE;
   }

   return(FALSE);
   
}


void p_window_statusbar_paint(pwindow_gtk_status *s) {

   GdkFont *font;
   int x;
   int y;

   gdk_draw_rectangle (s->buffer,
                       s->da->style->bg_gc[GTK_WIDGET_STATE(s->da)],
                       TRUE   /* Filled, this really ought to be a constant in GDK ... */,
                       0, 0,  /* Destination x, y */
                       s->da->allocation.width,
                       s->da->allocation.height);
   if(s->message != NULL) {
      font = s->da->style->font;
      x = 2;
      y = (s->da->allocation.height - (font->ascent + font->descent)) / 2 + font->ascent;
      gdk_draw_string (s->buffer,                     /* Buffer to write to */
                       s->da->style->font,            /* Font to use */
                       s->da->style->fg_gc[GTK_WIDGET_STATE(s->da)], 
                       x, y,                          /* Destination x, y */
                       s->message);                   /* Message to write */
   } /* If there is a message t print ... */
   gtk_widget_queue_draw_area(s->da, 0, 0, s->da->allocation.width, s->da->allocation.height);
   
}


static gint status_configure(GtkWidget *drawing_area, GdkEventConfigure *event, gpointer data) {
/* status_configure */

   pwindow_gtk_status *s = data;
   
   /* Release any existing drawing buffer */
   if (s->buffer != NULL) gdk_pixmap_unref(s->buffer);

   /* Construct new offscreen-drawable */
   s->buffer = gdk_pixmap_new(drawing_area->window,
                              drawing_area->allocation.width,
                              drawing_area->allocation.height,
                              -1 /* Use depth of window */);
   p_window_statusbar_paint(s);
   
   return(TRUE);
   
}


static gint status_expose(GtkWidget *widget, GdkEventExpose *event, gpointer data) {
/* status_expose */

   pwindow_gtk_status *s = data;

   g_return_val_if_fail(s->buffer != NULL, FALSE);

   gdk_draw_pixmap(widget->window,                                /* Destination */
                   widget->style->fg_gc[GTK_WIDGET_STATE(widget)],/* What the hell? */
                   s->buffer,                                     /* Source */
                   event->area.x, event->area.y,                  /* src x, y */
                   event->area.x, event->area.y,                  /* Dest x, y */
                   event->area.width, event->area.height);        /* Size */

   return(FALSE);
   
}



/***  Key Handler  ***/



static void p_process_key(pwindow_gtk *w, int player, int key) {

   pwindow *pw = (pwindow *)w;
   pplayer *p;
   int redraw;
   int x;
   int y;
   
   /* canNOT to anything while waiting for the network */
   if(w->c->waiting) return;

   if(w->c->paused) return;
   if(player > 0) player = P_PLAYER__(w->c);
   p = P_PLAYER(w->c, player);
   x = p->field->piece->x;
   y = p->field->piece->y;
   redraw = P_REDRAW_PLAYER(player);
   if(P_PLAYER_ACTIVE(p)) switch(key) {
      case P_KEY_LEFT:
         /* Move piece left one */
         if(p_piece_try_left(p->field)) {
            p_field_redraw_rect(p->field, x - 1, y, x + 1, y + 1);
            p_window_paint(w->c, pw, redraw);
            #if USE_NETWORK
            if(P_NETWORK(w->c)) {
               if(pnet_send_subboard(w->c, p, x - 1, y, x + 1, y + 1) < 0) p_window_set_message((pwindow *)w, pnet_get_error());
            } /* Sending the opponent a revised board? */
            #endif /* Network? */
         }
         return;
         
      case P_KEY_RIGHT:
         /* Move piece right one */
         if(p_piece_try_right(p->field)) {
            p_field_redraw_rect(p->field, x, y, x + 2, y + 1);
            p_window_paint(w->c, pw, redraw);
            #if USE_NETWORK
            if(P_NETWORK(w->c)) {
               if(pnet_send_subboard(w->c, p, x, y, x + 2, y + 1) < 0) p_window_set_message((pwindow *)w, pnet_get_error());
            } /* Sending the opponent a revised board? */
            #endif /* Network? */
         }
         return;

      case P_KEY_UP:
         /* Rotate piece, if possible */
         if(p_piece_try_rotate(p->field)) {
            p_field_redraw_rect(p->field, x - 1, y - 1, x + 2, y + 2);
            p_window_paint(w->c, pw, redraw);
            #if USE_NETWORK
            if(P_NETWORK(w->c)) {
               if(pnet_send_subboard(w->c, p, x - 1, y - 1, x + 2, y + 2) < 0) p_window_set_message((pwindow *)w, pnet_get_error());
            } /* Sending the opponent a revised board? */
            #endif /* Network? */
         }
         return;
         
      case P_KEY_DOWN:
         /* Drop piece */         
         p_game_set_new_state(p, P_STATE_FALLING);
         return;
         
   }
   
   return;
   
}

         
static gint window_key_press(GtkWidget *app, GdkEventKey *event, gpointer data) {
/* window_key_press */

   pwindow_gtk *w = data;
   guint keyval;
   int i;                  /* Iterate through possible keys */
   int *k;
   
   /* Get (uppercased) key code */
   keyval = gdk_keyval_to_upper(event->keyval);

   /* Check each player's keys */
   for(i = 0, k = &P_KEY_IW(w, 0); i < P_NUM_KEYS; i++, k++) {
      if(keyval == *k) {
         p_process_key(w, P_KEY_GET_PLAYER(i), P_KEY_GET_KEY_TYPE(i));
         return(TRUE);
      }
   }         

   /* Key was not defined for a player */
   switch(keyval) {
      case GDK_Return:
      case GDK_KP_Enter:
         if(w->c->paused) action_game_pause(w);
         else if(!P_TOURNAMENT(w->c) || P_GAME_INACTIVE(w->c)) action_game_restart(w);
         return(TRUE);
      case GDK_Escape:
         if(!P_TOURNAMENT(w->c)) action_game_new(w);
         return(TRUE);
      case GDK_P:
      case GDK_Pause:
         action_game_pause(w);
         return(TRUE);
   }

   return(FALSE);

}



/***  Miscellaneous callbacks  ***/



static gint delete_event_cb(GtkWidget *app, GdkEventAny *event, gpointer data) {
/* delete_event_cb */

   gtk_main_quit();
   return(TRUE);

}



/***  Window Creation  ***/



static void p_window_new_statusbar(pwindow_gtk_status *s, int minlength) {

   int width;
   int height;

   s->frame = gtk_frame_new(NULL);
   s->da = gtk_drawing_area_new();
   s->buffer = NULL;
   s->message = NULL;

   width = gdk_char_width(GTK_WIDGET(s->da)->style->font, 'x') * minlength;
   height = GTK_WIDGET(s->da)->style->font->ascent + GTK_WIDGET(s->da)->style->font->descent;
   
   gtk_widget_set_usize(s->da, width, height);
   gtk_widget_set_events(s->da, GDK_EXPOSURE_MASK);

   gtk_frame_set_shadow_type(GTK_FRAME(s->frame), GTK_SHADOW_IN);
   gtk_container_add(GTK_CONTAINER(s->frame), s->da);

   gtk_signal_connect(GTK_OBJECT(s->da), "expose_event",
                      GTK_SIGNAL_FUNC(status_expose), 
                      s);
   gtk_signal_connect(GTK_OBJECT(s->da), "configure_event",
                      GTK_SIGNAL_FUNC(status_configure), 
                      s);

}



pwindow *p_window_new(pconfig *c, int argc, char * const *argv) {
/* p_window_new */

   pwindow_gtk         *w;

   /* puyo and Gnome are fighting over what args are valid, etc.
      so just don't give Gnome anything to parse for now.
      The real solution is to use popt for puyo, or do some
      clever hacking here (a no-op popt table matching puyo's
      arguments).
   */
   gint disable_argc = 1;
   gchar* disable_argv[] = { "xpuyopuyo", NULL };
   char **disable_argv_p;
   GtkWidget *hbox;
   GtkWidget *cont;
   int i;

   /* Initialise GTK */   
   disable_argv_p = disable_argv;
   gtk_init(&disable_argc, &disable_argv_p);
   
   /* Initialise w */   
   w = g_new0(pwindow_gtk, 1);
   w->exposed = FALSE;
   w->ready = FALSE;
   w->c = c;
   w->c->window = (pwindow *)w;
   w->c->paused = FALSE;
   p_key_load_keys(w);

   /* Create the window */      
   w->app = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(w->app), "XPuyoPuyo");
   gtk_signal_connect(GTK_OBJECT(w->app), "delete_event",
                      GTK_SIGNAL_FUNC(delete_event_cb), 
                      w);
   gtk_window_set_policy(GTK_WINDOW(w->app), FALSE, FALSE, TRUE);
   
   /* Setup tooltips */
   w->tooltips = gtk_tooltips_new();
   if(w->c->tooltips) gtk_tooltips_enable(w->tooltips);
   else               gtk_tooltips_disable(w->tooltips);
   gtk_tooltips_set_delay(w->tooltips, P_GTK_TOOLTIP_DELAY);

   /* Create the main (vertical) container */
   cont = gtk_vbox_new(FALSE, 1);
   gtk_container_border_width(GTK_CONTAINER(cont), 1);
   gtk_container_add(GTK_CONTAINER(w->app), cont);

   /* Create menus, etc */
   p_window_create_menus(w);
   gtk_box_pack_start(GTK_BOX(cont), P_MENUW(w, P_MENU_MAIN), FALSE, TRUE, 0);

   /* Create drawable area */
   w->da  = gtk_drawing_area_new();
   gtk_widget_set_usize(w->da, P_WINDOW_WIDTH(c->fieldwidth), P_WINDOW_HEIGHT(c->fieldheight));
   gtk_widget_set_events(w->da, GDK_EXPOSURE_MASK);
   gtk_box_pack_start(GTK_BOX(cont), w->da, TRUE, TRUE, 0);
   gtk_signal_connect(GTK_OBJECT(w->da), "expose_event",
                      GTK_SIGNAL_FUNC(drawing_area_expose), 
                      w);
   gtk_signal_connect(GTK_OBJECT(w->da), "configure_event",
                      GTK_SIGNAL_FUNC(drawing_area_configure), 
                      w);
   
   /* Create statusbars */
   hbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(cont), hbox, FALSE, TRUE, 0);
   for(i = 0; i < 2; i++) {
      p_window_new_statusbar(&w->player[i], 12);
      gtk_box_pack_start(GTK_BOX(hbox), w->player[i].frame, TRUE, TRUE, 0);
   }
   p_window_new_statusbar(&w->status, 20);
   gtk_box_pack_start(GTK_BOX(cont), w->status.frame, FALSE, TRUE, 0);

   /* Show everything */
   gtk_widget_show_all(w->app);

   /* Load the images */
   p_window_init_theme((pwindow *)w);
   p_window_load_theme((pwindow *)w, w->c->theme);
   w->ready = TRUE;
   
   /* connect after loading, just to retain sanity. */
   gtk_signal_connect(GTK_OBJECT(w->app), "key_press_event",
                      GTK_SIGNAL_FUNC(window_key_press), 
                      w);

   /* Return the structure */
   return((pwindow *)w);

}



/***  Delete the Window  ***/



static void p_window_free_statusbar(pwindow_gtk_status *s) {

   if(s->buffer != NULL) gdk_pixmap_unref(s->buffer);
   free(s->message);

}



void p_window_free(pwindow **w_) {
/* p_window_free */

   pwindow_gtk *w = (pwindow_gtk *)*w_;

   gtk_widget_destroy(w->app);
   if(w->buffer) gdk_pixmap_unref(w->buffer);
   p_window_free_statusbar(&w->player[0]);
   p_window_free_statusbar(&w->player[1]);
   p_window_free_statusbar(&w->status);

   w->c->window = NULL;
   free(w);

   *w_ = NULL;

}



/***  Image Placement  ***/



static void p_window_put_subimage_to(const pwindow_gtk *w, GdkDrawable* dest, GdkPixmap *pix, GdkBitmap* mask, int sx, int sy, int dx, int dy, int w_, int h_) {
/* p_window_put_subimage_to */

   g_return_if_fail(w->da->style != NULL);

   if(mask) {
      gdk_gc_set_clip_mask(w->da->style->white_gc, mask); 
      gdk_gc_set_clip_origin(w->da->style->white_gc, dx - sx, dy - sy);
   }
   gdk_draw_pixmap(dest, w->da->style->white_gc, pix, sx, sy, dx, dy, w_, h_);
   if(mask) gdk_gc_set_clip_mask(w->da->style->white_gc, NULL); 
   return;

}


void p_window_put_subimage(const pwindow_gtk *w, GdkPixmap* pix, GdkBitmap* mask, int sx, int sy, int dx, int dy, int w_, int h_) {
/* p_window_put_subimage */

   g_return_if_fail(w->da->style != NULL);

   if(mask) {
      gdk_gc_set_clip_mask(w->da->style->white_gc, mask); 
      gdk_gc_set_clip_origin(w->da->style->white_gc, dx - sx, dy - sy);
   }
   gdk_draw_pixmap(w->buffer, w->da->style->white_gc, pix, sx, sy, dx, dy, w_, h_);
   if(mask) gdk_gc_set_clip_mask(w->da->style->white_gc, NULL); 
   gtk_widget_queue_draw_area(w->da, dx, dy, w_, h_);
   return;

}


void p_window_put_subimage_index(const pwindow_gtk *w, int imageindex, int sx, int sy, int dx, int dy, int w_, int h_) {
/* p_window_put_subimage_index */

   p_window_put_subimage(w, P_IMAGE(w->img, imageindex), P_IMAGE_MASK(w->img, imageindex), sx, sy, dx, dy, w_, h_);
   return;

}


static inline void p_window_put_image_to(const pwindow_gtk *w, GdkDrawable* dest, GdkPixmap* pix, GdkBitmap* mask, int dx, int dy) {

   p_window_put_subimage_to(w, dest, pix, mask, 0, 0, dx, dy, pixmap_width(pix), pixmap_height(pix));

}


void p_window_put_image(const pwindow_gtk *w, GdkPixmap *pix, GdkBitmap* mask, int dx, int dy) {

   p_window_put_subimage(w, pix, mask, 0, 0, dx, dy, pixmap_width(pix), pixmap_height(pix));

}


void p_window_put_image_index(const pwindow_gtk *w, int imageindex, int dx, int dy) {
/* p_window_put_image_index */

   p_window_put_image(w, P_IMAGE(w->img, imageindex), P_IMAGE_MASK(w->img, imageindex), dx, dy);
   return;

}


